// @ts-check
const { test, expect } = require('@playwright/test');

const BASE_URL = 'https://deepleafai.com';
const ADMIN_USERNAME = 'admin';
const ADMIN_PASSWORD = 'password';

// Helper function to generate random email
function generateEmail() {
    return `test_${Date.now()}@test.com`;
}

test.describe('LoudCannabis Platform E2E Tests', () => {
    
    test.describe('User Flow - Survey & AI Chat', () => {
        test('should complete survey flow', async ({ page }) => {
            console.log('🧪 Testing user survey flow...');
            
            // Navigate to homepage
            await page.goto(BASE_URL);
            await page.waitForLoadState('networkidle');
            
            // Check if page loaded
            await expect(page).toHaveTitle(/LoudCannabis/i);
            console.log('✅ Homepage loaded');
            
            // Look for survey start button or form
            const startButton = page.locator('button:has-text("Start"), button:has-text("Begin"), button:has-text("Quiz")').first();
            if (await startButton.count() > 0) {
                await startButton.click();
                console.log('✅ Survey started');
                await page.waitForTimeout(2000);
            }
            
            // Take screenshot
            await page.screenshot({ path: 'tests/screenshots/user-survey-start.png', fullPage: true });
        });
        
        test('should handle email verification', async ({ page }) => {
            console.log('🧪 Testing email verification...');
            
            await page.goto(BASE_URL);
            await page.waitForLoadState('networkidle');
            
            // Look for email input (might be in a modal)
            // First try to trigger the modal if it exists
            const startBtn = page.locator('button:has-text("Start"), button:has-text("Begin"), button:has-text("Quiz")').first();
            if (await startBtn.count() > 0 && await startBtn.isVisible()) {
                await startBtn.click();
                await page.waitForTimeout(2000); // Wait for modal to appear
            }
            
            const emailInput = page.locator('input[type="email"], input#modal_email').first();
            if (await emailInput.count() > 0) {
                // Wait for it to be visible
                await emailInput.waitFor({ state: 'visible', timeout: 5000 }).catch(() => {});
                const isVisible = await emailInput.isVisible();
                if (isVisible) {
                    const testEmail = generateEmail();
                    await emailInput.fill(testEmail);
                    console.log(`✅ Email entered: ${testEmail}`);
                    
                    // Look for submit button
                    const submitButton = page.locator('button[type="submit"], button:has-text("Submit"), button:has-text("Continue")').first();
                    if (await submitButton.count() > 0) {
                        await submitButton.click();
                        await page.waitForTimeout(2000);
                        console.log('✅ Email submitted');
                    }
                } else {
                    console.log('⚠️ Email input not visible (modal may need different trigger)');
                }
            } else {
                console.log('⚠️ Email input not found');
            }
            
            await page.screenshot({ path: 'tests/screenshots/user-email-verification.png', fullPage: true });
        });
    });
    
    test.describe('Admin Flow', () => {
        test('should login to admin dashboard', async ({ page }) => {
            console.log('🧪 Testing admin login...');
            
            await page.goto(`${BASE_URL}/admin/login.php`);
            await page.waitForLoadState('networkidle');
            
            // Fill login form
            await page.fill('input[name="username"]', ADMIN_USERNAME);
            await page.fill('input[name="password"]', ADMIN_PASSWORD);
            
            // Submit form (admin login uses button with id="login")
            await page.click('button#login, button[name="login"]');
            await page.waitForTimeout(2000); // Wait for AJAX login
            
            // Check if redirected to dashboard
            const currentUrl = page.url();
            console.log(`✅ Login completed, current URL: ${currentUrl}`);
            
            await page.screenshot({ path: 'tests/screenshots/admin-login.png', fullPage: true });
        });
        
        test('should access admin dashboard', async ({ page }) => {
            console.log('🧪 Testing admin dashboard access...');
            
            // Login first
            await page.goto(`${BASE_URL}/admin/login.php`);
            await page.fill('input[name="username"]', ADMIN_USERNAME);
            await page.fill('input[name="password"]', ADMIN_PASSWORD);
            await page.click('button#login, button[name="login"]');
            await page.waitForTimeout(2000);
            
            // Navigate to admin dashboard
            await page.goto(`${BASE_URL}/admin_dashboard.html`);
            await page.waitForLoadState('networkidle');
            await page.waitForTimeout(3000); // Wait for stats to load
            
            // Check for dashboard elements
            const overviewTab = page.locator('button:has-text("Overview"), a:has-text("Overview")').first();
            if (await overviewTab.count() > 0) {
                console.log('✅ Admin dashboard loaded');
            }
            
            // Check for stats
            const stats = page.locator('.stat-card, .stat-value, [class*="stat"]');
            const statsCount = await stats.count();
            console.log(`✅ Found ${statsCount} stat elements`);
            
            await page.screenshot({ path: 'tests/screenshots/admin-dashboard-overview.png', fullPage: true });
        });
        
        test('should navigate admin dashboard tabs', async ({ page }) => {
            console.log('🧪 Testing admin dashboard navigation...');
            
            // Login
            await page.goto(`${BASE_URL}/admin/login.php`);
            await page.fill('input[name="username"]', ADMIN_USERNAME);
            await page.fill('input[name="password"]', ADMIN_PASSWORD);
            await page.click('button#login, button[name="login"]');
            await page.waitForTimeout(3000); // Wait for AJAX login to complete
            
            // Go to dashboard
            await page.goto(`${BASE_URL}/admin_dashboard.html`);
            await page.waitForLoadState('networkidle');
            await page.waitForTimeout(3000);
            
            // Test each tab
            const tabs = ['Affiliates', 'Sponsors', 'Offers', 'Tracking', 'Reports'];
            
            for (const tabName of tabs) {
                console.log(`  Testing ${tabName} tab...`);
                const tab = page.locator(`button:has-text("${tabName}"), a:has-text("${tabName}")`).first();
                
                if (await tab.count() > 0) {
                    await tab.click();
                    await page.waitForTimeout(2000); // Wait for content to load
                    console.log(`  ✅ ${tabName} tab clicked`);
                    
                    await page.screenshot({ 
                        path: `tests/screenshots/admin-dashboard-${tabName.toLowerCase()}.png`, 
                        fullPage: true 
                    });
                } else {
                    console.log(`  ⚠️ ${tabName} tab not found`);
                }
            }
        });
        
        test('should test affiliate management', async ({ page }) => {
            console.log('🧪 Testing affiliate management...');
            
            // Login and navigate
            await page.goto(`${BASE_URL}/admin/login.php`);
            await page.fill('input[name="username"]', ADMIN_USERNAME);
            await page.fill('input[name="password"]', ADMIN_PASSWORD);
            await page.click('button#login, button[name="login"]');
            await page.waitForTimeout(2000);
            
            await page.goto(`${BASE_URL}/admin_dashboard.html`);
            await page.waitForLoadState('networkidle');
            await page.waitForTimeout(3000);
            
            // Click Affiliates tab
            const affiliatesTab = page.locator('button:has-text("Affiliates"), a:has-text("Affiliates")').first();
            if (await affiliatesTab.count() > 0) {
                await affiliatesTab.click();
                await page.waitForTimeout(2000);
                
                // Check for affiliate table
                const affiliateTable = page.locator('table, .table, [class*="table"]');
                const tableCount = await affiliateTable.count();
                console.log(`✅ Found ${tableCount} table elements`);
                
                // Look for view/edit buttons
                const viewButtons = page.locator('button:has-text("View"), button:has-text("👁️")');
                const viewCount = await viewButtons.count();
                console.log(`✅ Found ${viewCount} view buttons`);
                
                await page.screenshot({ path: 'tests/screenshots/admin-affiliates.png', fullPage: true });
            }
        });
        
        test('should test sponsor management', async ({ page }) => {
            console.log('🧪 Testing sponsor management...');
            
            // Login and navigate
            await page.goto(`${BASE_URL}/admin/login.php`);
            await page.fill('input[name="username"]', ADMIN_USERNAME);
            await page.fill('input[name="password"]', ADMIN_PASSWORD);
            await page.click('button#login, button[name="login"]');
            await page.waitForTimeout(2000);
            
            await page.goto(`${BASE_URL}/admin_dashboard.html`);
            await page.waitForLoadState('networkidle');
            await page.waitForTimeout(3000);
            
            // Click Sponsors tab
            const sponsorsTab = page.locator('button:has-text("Sponsors"), a:has-text("Sponsors")').first();
            if (await sponsorsTab.count() > 0) {
                await sponsorsTab.click();
                await page.waitForTimeout(2000);
                
                // Look for add sponsor button
                const addButton = page.locator('button:has-text("Add"), button:has-text("➕")').first();
                if (await addButton.count() > 0) {
                    console.log('✅ Add sponsor button found');
                }
                
                await page.screenshot({ path: 'tests/screenshots/admin-sponsors.png', fullPage: true });
            }
        });
    });
    
    test.describe('Affiliate Flow', () => {
        test('should access affiliate dashboard', async ({ page }) => {
            console.log('🧪 Testing affiliate dashboard access...');
            
            await page.goto(`${BASE_URL}/affiliate_dashboard.html`);
            await page.waitForLoadState('networkidle');
            
            // Check for login form
            const loginForm = page.locator('input[type="email"]');
            if (await loginForm.count() > 0) {
                console.log('✅ Affiliate login form found');
                
                // Try to login with test email
                const testEmail = generateEmail();
                await loginForm.fill(testEmail);
                
                const loginButton = page.locator('button:has-text("Login")').first();
                if (await loginButton.count() > 0) {
                    await loginButton.click();
                    await page.waitForTimeout(2000);
                    console.log('✅ Login attempted');
                }
            }
            
            await page.screenshot({ path: 'tests/screenshots/affiliate-dashboard.png', fullPage: true });
        });
    });
    
    test.describe('API Endpoints', () => {
        test('should test public API endpoints', async ({ request }) => {
            console.log('🧪 Testing API endpoints...');
            
            const endpoints = [
                { url: '/api/health.php', name: 'Health Check' },
                { url: '/api/admin_stats.php', name: 'Admin Stats' },
                { url: '/api/system_stats.php', name: 'System Stats' }
            ];
            
            for (const endpoint of endpoints) {
                try {
                    const response = await request.get(`${BASE_URL}${endpoint.url}`);
                    const status = response.status();
                    console.log(`  ${endpoint.name}: ${status === 200 ? '✅' : '❌'} Status ${status}`);
                } catch (error) {
                    console.log(`  ${endpoint.name}: ❌ Error - ${error.message}`);
                }
            }
        });
    });
});

