<?php

use PHPUnit\Framework\TestCase;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Unit tests for RAGService
 * Tests RAG integration with mocked HTTP client
 */
class RAGServiceTest extends TestCase {

    private RAGService $ragService;
    private MockObject $httpClientMock;

    protected function setUp(): void {
        // Create a partial mock to avoid actual HTTP calls
        $this->ragService = $this->getMockBuilder(RAGService::class)
            ->onlyMethods(['callRAGAPI', 'checkRAGAvailability'])
            ->getMock();

        $this->ragService->method('checkRAGAvailability')
            ->willReturn(true);
    }

    public function testCanCreateRAGServiceInstance() {
        $this->assertInstanceOf(RAGService::class, $this->ragService);
    }

    public function testQueryReturnsStructuredResponse() {
        $this->ragService->method('callRAGAPI')
            ->willReturn([
                'context' => 'Test context from RAG',
                'sources' => [['title' => 'Source 1', 'url' => 'url1', 'score' => 0.9]],
                'confidence' => 0.85
            ]);

        $result = $this->ragService->query('test query');

        $this->assertIsArray($result);
        $this->assertArrayHasKey('context', $result);
        $this->assertArrayHasKey('sources', $result);
        $this->assertArrayHasKey('confidence', $result);
        $this->assertEquals('Test context from RAG', $result['context']);
        $this->assertEquals(0.85, $result['confidence']);
    }

    public function testQueryWithHistoryIncludesConversationContext() {
        $this->ragService->method('callRAGAPI')
            ->willReturn([
                'context' => 'Context based on history',
                'sources' => [],
                'confidence' => 0.7
            ]);

        $history = [
            ['text' => 'Previous message'],
            ['user_message' => 'Another question', 'ai_response' => 'AI response']
        ];

        $result = $this->ragService->query('current question', $history);

        $this->assertIsArray($result);
        $this->assertArrayHasKey('context', $result);
    }

    public function testQueryWithPreferencesPassesPreferencesToAPI() {
        $this->ragService->method('callRAGAPI')
            ->willReturn([
                'context' => 'Preference-based context',
                'sources' => [],
                'confidence' => 0.6
            ]);

        $preferences = ['topics' => ['cannabis', 'business']];
        $result = $this->ragService->query('question', [], $preferences);

        $this->assertIsArray($result);
        $this->assertArrayHasKey('context', $result);
    }

    public function testQueryHandlesAPIErrorGracefully() {
        $this->ragService->method('callRAGAPI')
            ->willThrowException(new RuntimeException('API Error'));

        $result = $this->ragService->query('test query');

        $this->assertIsArray($result);
        $this->assertNull($result['context']);
        $this->assertEmpty($result['sources']);
        $this->assertEquals(0.0, $result['confidence']);
    }

    public function testQueryWhenServiceUnavailableReturnsEmptyResult() {
        $ragService = $this->getMockBuilder(RAGService::class)
            ->onlyMethods(['checkRAGAvailability'])
            ->getMock();

        $ragService->method('checkRAGAvailability')->willReturn(false);

        $result = $ragService->query('test query');

        $this->assertIsArray($result);
        $this->assertNull($result['context']);
        $this->assertEmpty($result['sources']);
        $this->assertEquals(0.0, $result['confidence']);
    }

    public function testGetStatusReturnsServiceInformation() {
        $this->ragService->method('checkRAGAvailability')->willReturn(true);

        $status = $this->ragService->getStatus();

        $this->assertIsArray($status);
        $this->assertArrayHasKey('enabled', $status);
        $this->assertArrayHasKey('endpoint', $status);
        $this->assertArrayHasKey('available', $status);
        $this->assertTrue($status['enabled']);
        $this->assertStringContains('localhost:5000', $status['endpoint']);
    }

    public function testUpdateConfigurationChangesSettings() {
        $newConfig = [
            'endpoint' => 'http://custom-endpoint:8080/query',
            'enabled' => false
        ];

        $this->ragService->updateConfiguration($newConfig);

        $status = $this->ragService->getStatus();
        $this->assertEquals('http://custom-endpoint:8080/query', $status['endpoint']);
    }

    public function testBuildRAGPayloadCreatesCorrectStructure() {
        $reflection = new ReflectionClass(RAGService::class);
        $method = $reflection->getMethod('buildRAGPayload');
        $method->setAccessible(true);

        $result = $method->invokeArgs($this->ragService, [
            'Test query',
            [['text' => 'Previous conversation']],
            ['topic' => 'cannabis']
        ]);

        $this->assertIsArray($result);
        $this->assertEquals('Test query', $result['query']);
        $this->assertArrayHasKey('conversation_context', $result);
        $this->assertArrayHasKey('preferences', $result);
        $this->assertArrayHasKey('top_k', $result);
        $this->assertArrayHasKey('include_sources', $result);
        $this->assertEquals(5, $result['top_k']);
        $this->assertTrue($result['include_sources']);
    }

    public function testBuildRAGPayloadHandlesEmptyHistory() {
        $reflection = new ReflectionClass(RAGService::class);
        $method = $reflection->getMethod('buildRAGPayload');
        $method->setAccessible(true);

        $result = $method->invokeArgs($this->ragService, ['Test query', [], []]);

        $this->assertIsArray($result);
        $this->assertEquals('Test query', $result['query']);
        $this->assertIsString($result['conversation_context']);
    }

    public function testProcessRAGResponseFormatsSourcesCorrectly() {
        $reflection = new ReflectionClass(RAGService::class);
        $method = $reflection->getMethod('processRAGResponse');
        $method->setAccessible(true);

        $apiResponse = [
            'context' => 'Test context',
            'sources' => [
                ['title' => 'Source 1', 'url' => 'url1', 'score' => 0.9],
                ['title' => 'Source 2', 'url' => 'url2', 'score' => 0.8]
            ],
            'confidence' => 0.85
        ];

        $result = $method->invokeArgs($this->ragService, [$apiResponse]);

        $this->assertIsArray($result);
        $this->assertStringContains('Test context', $result['context']);
        $this->assertStringContains('Sources:', $result['context']);
        $this->assertStringContains('Source 1', $result['context']);
        $this->assertCount(2, $result['sources']);
        $this->assertEquals(0.85, $result['confidence']);
    }

    public function testProcessRAGResponseHandlesMissingSources() {
        $reflection = new ReflectionClass(RAGService::class);
        $method = $reflection->getMethod('processRAGResponse');
        $method->setAccessible(true);

        $apiResponse = [
            'context' => 'Test context without sources',
            'confidence' => 0.7
        ];

        $result = $method->invokeArgs($this->ragService, [$apiResponse]);

        $this->assertIsArray($result);
        $this->assertEquals('Test context without sources', $result['context']);
        $this->assertEmpty($result['sources']);
        $this->assertEquals(0.7, $result['confidence']);
    }

    public function testCheckRAGAvailabilityHandlesConnectionFailure() {
        $ragService = $this->getMockBuilder(RAGService::class)
            ->onlyMethods(['callRAGAPI'])
            ->getMock();

        $ragService->method('callRAGAPI')
            ->willThrowException(new Exception('Connection failed'));

        $reflection = new ReflectionClass(RAGService::class);
        $method = $reflection->getMethod('checkRAGAvailability');
        $method->setAccessible(true);

        $result = $method->invoke($ragService);
        $this->assertFalse($result);
    }

    public function testServiceInitializationDefaultsToEnabled() {
        $status = $this->ragService->getStatus();
        $this->assertTrue($status['enabled']);
    }

    public function testConfigurationUpdateAllowsDisablingService() {
        $this->ragService->updateConfiguration(['enabled' => false]);
        $status = $this->ragService->getStatus();
        $this->assertFalse($status['enabled']);
    }
}
