<?php

use PHPUnit\Framework\TestCase;

/**
 * Unit tests for OfferService
 * Tests monetization offer analysis and recommendation logic
 */
class OfferServiceTest extends TestCase {

    private OfferService $offerService;

    protected function setUp(): void {
        $this->offerService = new OfferService();
    }

    public function testCanCreateOfferServiceInstance() {
        $this->assertInstanceOf(OfferService::class, $this->offerService);
    }

    public function testAnalyzeForOffersReturnsArray() {
        $offers = $this->offerService->analyzeForOffers(
            'I want to start a cannabis business',
            ['topics' => ['business']],
            [],
            ['answers' => ['10' => 'yes']],
            []
        );

        $this->assertIsArray($offers);
    }

    public function testBusinessOwnerGetsBusinessOffers() {
        $offers = $this->offerService->analyzeForOffers(
            'I want to start a dispensary',
            ['topics' => ['business']],
            [],
            ['answers' => ['10' => 'yes, I own a business']],
            []
        );

        $this->assertIsArray($offers);

        if (!empty($offers)) {
            $this->assertArrayHasKey('title', $offers[0]);
            $this->assertArrayHasKey('cta', $offers[0]);
            $this->assertStringContains('business', strtolower($offers[0]['title'] ?? ''));
        }
    }

    public function testDebtConcernsGetDebtOffers() {
        $offers = $this->offerService->analyzeForOffers(
            'I have debt problems and need money',
            ['topics' => ['debt']],
            [],
            ['answers' => ['04' => 'yes, I have debt']],
            []
        );

        $this->assertIsArray($offers);

        if (!empty($offers)) {
            $this->assertArrayHasKey('title', $offers[0]);
            $this->assertStringContains('debt', strtolower($offers[0]['title'] ?? ''));
        }
    }

    public function testMedicalUsersGetMedicalOffers() {
        $offers = $this->offerService->analyzeForOffers(
            'I need medical cannabis information',
            ['topics' => ['health']],
            [],
            ['answers' => ['01' => 'medical use']],
            []
        );

        $this->assertIsArray($offers);

        if (!empty($offers)) {
            $this->assertArrayHasKey('title', $offers[0]);
            $this->assertStringContains('medical', strtolower($offers[0]['title'] ?? ''));
        }
    }

    public function testNoOffersForGenericQuestions() {
        $offers = $this->offerService->analyzeForOffers(
            'What is the weather like?',
            ['topics' => []],
            [],
            [],
            []
        );

        $this->assertIsArray($offers);
        // Generic questions should not trigger offers
        $this->assertEmpty($offers);
    }

    public function testCompletedOffersAreExcluded() {
        $offers = $this->offerService->analyzeForOffers(
            'I want to start a business',
            ['topics' => ['business']],
            [],
            ['answers' => ['10' => 'yes']],
            ['completed_offers' => ['cannabis_business_call']]
        );

        $this->assertIsArray($offers);
        // Should not include the completed offer
        foreach ($offers as $offer) {
            $this->assertNotEquals('cannabis_business_call', $offer['id'] ?? '');
        }
    }

    public function testRejectedOffersAreExcluded() {
        $offers = $this->offerService->analyzeForOffers(
            'I want to start a business',
            ['topics' => ['business']],
            [],
            ['answers' => ['10' => 'yes']],
            ['rejected_offers' => ['cannabis_business_call']]
        );

        $this->assertIsArray($offers);
        // Should not include the rejected offer
        foreach ($offers as $offer) {
            $this->assertNotEquals('cannabis_business_call', $offer['id'] ?? '');
        }
    }

    public function testMultipleTopicsIncreaseRelevance() {
        $offers1 = $this->offerService->analyzeForOffers(
            'business license',
            ['topics' => ['business']],
            [],
            [],
            []
        );

        $offers2 = $this->offerService->analyzeForOffers(
            'business license and energy',
            ['topics' => ['business', 'energy']],
            [],
            [],
            []
        );

        // More specific matches should return offers
        $this->assertIsArray($offers1);
        $this->assertIsArray($offers2);
    }

    public function testSurveySignalMatching() {
        $offers = $this->offerService->analyzeForOffers(
            'I need money',
            ['topics' => ['money']],
            [],
            ['answers' => ['09' => 'yes, I need income']],
            []
        );

        $this->assertIsArray($offers);
        // Should match offers related to money/income
        if (!empty($offers)) {
            $title = strtolower($offers[0]['title'] ?? '');
            $this->assertTrue(
                strpos($title, 'debt') !== false ||
                strpos($title, 'income') !== false ||
                strpos($title, 'money') !== false,
                "Offer should be related to financial needs"
            );
        }
    }

    public function testKeywordMatchingInConversationHistory() {
        $offers = $this->offerService->analyzeForOffers(
            'Hello',
            ['topics' => []],
            [
                ['text' => 'I want to start a cannabis business'],
                ['text' => 'Looking for dispensary license info']
            ],
            [],
            []
        );

        $this->assertIsArray($offers);
        // Should find business-related offers from history
        if (!empty($offers)) {
            $this->assertArrayHasKey('title', $offers[0]);
        }
    }

    public function testOffersIncludeRequiredFields() {
        $offers = $this->offerService->analyzeForOffers(
            'business startup',
            ['topics' => ['business']],
            [],
            ['answers' => ['10' => 'yes']],
            []
        );

        $this->assertIsArray($offers);

        foreach ($offers as $offer) {
            $this->assertArrayHasKey('id', $offer);
            $this->assertArrayHasKey('title', $offer);
            $this->assertArrayHasKey('cta', $offer);
            $this->assertArrayHasKey('score', $offer);
            $this->assertIsString($offer['title']);
            $this->assertIsString($offer['cta']);
        }
    }

    public function testOfferScoringMechanism() {
        // Test that offers are ordered by relevance score
        $offers = $this->offerService->analyzeForOffers(
            'I want to start a business and save on energy costs',
            ['topics' => ['business', 'energy']],
            [],
            ['answers' => ['10' => 'yes', '08' => 'yes']],
            []
        );

        $this->assertIsArray($offers);

        if (count($offers) > 1) {
            // Check that offers are ordered by score (highest first)
            for ($i = 0; $i < count($offers) - 1; $i++) {
                $this->assertGreaterThanOrEqual(
                    $offers[$i + 1]['score'] ?? 0,
                    $offers[$i]['score'] ?? 0,
                    'Offers should be ordered by score descending'
                );
            }
        }
    }

    public function testOfferAnalyticsRecording() {
        // Test that offer interactions can be recorded
        $this->offerService->recordOfferInteraction('user123', 'test_offer', 'shown');

        // This is primarily a smoke test - in real implementation would verify logging
        $this->assertTrue(true, 'Offer analytics recording should not throw exceptions');
    }

    public function testPersonalizedOffersMethodExists() {
        $this->assertTrue(method_exists($this->offerService, 'getPersonalizedOffers'));
        $result = $this->offerService->getPersonalizedOffers('user123');
        $this->assertIsArray($result);
    }
}
