<?php

use PHPUnit\Framework\TestCase;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Unit tests for ChatService
 * Tests AI chat business logic with mocked dependencies
 */
class ChatServiceTest extends TestCase {

    private ChatService $chatService;
    private MockObject $httpClientMock;

    protected function setUp(): void {
        // Mock the HTTP client to avoid actual API calls
        $this->httpClientMock = $this->createMock(CurlHttpClient::class);

        // We'll need to modify ChatService to accept an HTTP client for testing
        // For now, we'll test the methods that don't require external calls
        $this->chatService = $this->getMockBuilder(ChatService::class)
            ->onlyMethods(['callAIAPI'])
            ->getMock();
    }

    public function testCanCreateChatServiceInstance() {
        $this->assertInstanceOf(ChatService::class, $this->chatService);
    }

    public function testBuildSystemPromptReturnsValidString() {
        $reflection = new ReflectionClass(ChatService::class);
        $method = $reflection->getMethod('buildChatPrompt');
        $method->setAccessible(true);

        $result = $method->invokeArgs($this->chatService, [
            'Hello AI',
            [],
            ['topic' => 'cannabis'],
            ['survey' => 'business owner'],
            'Some RAG context',
            ['name' => 'John'],
            [['title' => 'Business Course', 'cta' => 'Learn More']],
            ['tier' => 'free']
        ]);

        $this->assertIsString($result);
        $this->assertStringContains('DeepLeaf', $result);
        $this->assertStringContains('Hello AI', $result);
        $this->assertStringContains('Business Course', $result);
    }

    public function testBuildOfferSectionForFreeTier() {
        $reflection = new ReflectionClass(ChatService::class);
        $method = $reflection->getMethod('buildOfferSection');
        $method->setAccessible(true);

        $offer = ['title' => 'Test Offer', 'cta' => 'Click Here'];
        $result = $method->invokeArgs($this->chatService, [$offer, true]);

        $this->assertIsString($result);
        $this->assertStringContains('Test Offer', $result);
        $this->assertStringContains('Click Here', $result);
        $this->assertStringContains('TEASER MODE', $result);
    }

    public function testBuildOfferSectionForPremiumTier() {
        $reflection = new ReflectionClass(ChatService::class);
        $method = $reflection->getMethod('buildOfferSection');
        $method->setAccessible(true);

        $offer = ['title' => 'Premium Offer', 'cta' => 'Upgrade Now'];
        $result = $method->invokeArgs($this->chatService, [$offer, false]);

        $this->assertIsString($result);
        $this->assertStringContains('Premium Offer', $result);
        $this->assertStringContains('Upgrade Now', $result);
        $this->assertStringNotContains('TEASER MODE', $result);
    }

    public function testBuildHistorySectionWithEmptyHistory() {
        $reflection = new ReflectionClass(ChatService::class);
        $method = $reflection->getMethod('buildHistorySection');
        $method->setAccessible(true);

        $result = $method->invokeArgs($this->chatService, [[]]);

        $this->assertIsString($result);
        $this->assertStringContains('CONVERSATION HISTORY', $result);
    }

    public function testBuildHistorySectionWithMessages() {
        $reflection = new ReflectionClass(ChatService::class);
        $method = $reflection->getMethod('buildHistorySection');
        $method->setAccessible(true);

        $history = [
            ['user_message' => 'Hello', 'ai_response' => 'Hi there!'],
            ['user_message' => 'How are you?', 'ai_response' => 'I am doing well!']
        ];

        $result = $method->invokeArgs($this->chatService, [$history]);

        $this->assertIsString($result);
        $this->assertStringContains('USER: Hello', $result);
        $this->assertStringContains('AI: Hi there!', $result);
    }

    public function testGetUserAccessTierFree() {
        $reflection = new ReflectionClass(ChatService::class);
        $method = $reflection->getMethod('getUserAccessTier');
        $method->setAccessible(true);

        $result = $method->invokeArgs($this->chatService, [['completed_offers' => []]]);
        $this->assertEquals('free', $result);
    }

    public function testGetUserAccessTierEngaged() {
        $reflection = new ReflectionClass(ChatService::class);
        $method = $reflection->getMethod('getUserAccessTier');
        $method->setAccessible(true);

        $result = $method->invokeArgs($this->chatService, [['completed_offers' => ['offer1']]]);
        $this->assertEquals('engaged', $result);
    }

    public function testGetUserAccessTierPremium() {
        $reflection = new ReflectionClass(ChatService::class);
        $method = $reflection->getMethod('getUserAccessTier');
        $method->setAccessible(true);

        $result = $method->invokeArgs($this->chatService, [['completed_offers' => ['offer1', 'offer2', 'offer3']]]);
        $this->assertEquals('premium', $result);
    }

    public function testCalculateSurveyScoreWithMatchingSignals() {
        $reflection = new ReflectionClass(ChatService::class);
        $method = $reflection->getMethod('calculateSurveyScore');
        $method->setAccessible(true);

        $offer = [
            'survey_signals' => [
                ['question' => '10', 'needles' => ['yes', 'business']]
            ]
        ];

        $surveyData = ['answers' => ['10' => 'yes, I own a business']];

        $result = $method->invokeArgs($this->chatService, [$offer, $surveyData]);
        $this->assertGreaterThan(0, $result);
    }

    public function testCalculateSurveyScoreWithNoMatches() {
        $reflection = new ReflectionClass(ChatService::class);
        $method = $reflection->getMethod('calculateSurveyScore');
        $method->setAccessible(true);

        $offer = [
            'survey_signals' => [
                ['question' => '10', 'needles' => ['business']]
            ]
        ];

        $surveyData = ['answers' => ['10' => 'no']];

        $result = $method->invokeArgs($this->chatService, [$offer, $surveyData]);
        $this->assertEquals(0, $result);
    }

    public function testPromptBuildingWithAllComponents() {
        $reflection = new ReflectionClass(ChatService::class);
        $method = $reflection->getMethod('buildChatPrompt');
        $method->setAccessible(true);

        $result = $method->invokeArgs($this->chatService, [
            'Test question',
            [['user_message' => 'Previous', 'ai_response' => 'Response']],
            ['topic' => 'business'],
            ['BUSINESS OWNER: Prime candidate'],
            'RAG knowledge about cannabis',
            ['name' => 'Test User'],
            [['title' => 'Business Offer', 'cta' => 'Call Now']],
            ['tier' => 'free']
        ]);

        $this->assertIsString($result);
        $this->assertStringContains('Test User', $result);
        $this->assertStringContains('BUSINESS OWNER', $result);
        $this->assertStringContains('RAG knowledge', $result);
        $this->assertStringContains('Business Offer', $result);
    }

    public function testFormatContextIncludesSourceAttribution() {
        $reflection = new ReflectionClass(ChatService::class);
        $method = $reflection->getMethod('formatContext');
        $method->setAccessible(true);

        $context = 'Some context';
        $sources = [['title' => 'Source 1', 'url' => 'url1']];
        $confidence = 0.85;

        $result = $method->invokeArgs($this->chatService, [$context, $sources, $confidence]);

        $this->assertIsArray($result);
        $this->assertEquals($context, $result['context']);
        $this->assertCount(1, $result['sources']);
        $this->assertEquals('Source 1', $result['sources'][0]['title']);
    }

    public function testPromptIncludesRAGContextWhenProvided() {
        $reflection = new ReflectionClass(ChatService::class);
        $method = $reflection->getMethod('buildChatPrompt');
        $method->setAccessible(true);

        $ragContext = "Cannabis has been used for medicinal purposes for thousands of years, with evidence from ancient Chinese and Indian texts.";
        $surveyInsights = ['MEDICAL USER: Interested in therapeutic benefits'];

        $result = $method->invokeArgs($this->chatService, [
            'What are the benefits of cannabis?',
            [],
            ['topics' => ['medical', 'health']],
            $surveyInsights,
            $ragContext,
            ['name' => 'John'],
            [],
            ['tier' => 'free']
        ]);

        $this->assertStringContains('KNOWLEDGE BASE CONTEXT', $result);
        $this->assertStringContains('Cannabis has been used', $result);
        $this->assertStringContains('Use this context to provide accurate information', $result);
        $this->assertStringContains('MEDICAL USER', $result);
    }

    public function testPromptIncludesSurveyInsightsWhenProvided() {
        $reflection = new ReflectionClass(ChatService::class);
        $method = $reflection->getMethod('buildChatPrompt');
        $method->setAccessible(true);

        $surveyInsights = [
            'BUSINESS OWNER: Prime candidate for B2B offers',
            'MEDICAL USER: Interested in therapeutic applications'
        ];

        $result = $method->invokeArgs($this->chatService, [
            'Tell me about business opportunities',
            [],
            ['topics' => ['business']],
            $surveyInsights,
            null,
            ['name' => 'Jane'],
            [],
            ['tier' => 'free']
        ]);

        $this->assertStringContains('USER SURVEY INSIGHTS', $result);
        $this->assertStringContains('BUSINESS OWNER', $result);
        $this->assertStringContains('MEDICAL USER', $result);
        $this->assertStringContains('therapeutic applications', $result);
    }

    public function testPromptHandlesEmptyRAGContextGracefully() {
        $reflection = new ReflectionClass(ChatService::class);
        $method = $reflection->getMethod('buildChatPrompt');
        $method->setAccessible(true);

        $result = $method->invokeArgs($this->chatService, [
            'Test question',
            [],
            [],
            [],
            null, // No RAG context
            ['name' => 'User'],
            [],
            ['tier' => 'free']
        ]);

        $this->assertStringNotContains('KNOWLEDGE BASE CONTEXT', $result);
        $this->assertStringNotContains('Use this context', $result);
    }

    public function testPromptHandlesEmptySurveyInsightsGracefully() {
        $reflection = new ReflectionClass(ChatService::class);
        $method = $reflection->getMethod('buildChatPrompt');
        $method->setAccessible(true);

        $result = $method->invokeArgs($this->chatService, [
            'Test question',
            [],
            [],
            [], // No survey insights
            'Some RAG context',
            ['name' => 'User'],
            [],
            ['tier' => 'free']
        ]);

        $this->assertStringNotContains('USER SURVEY INSIGHTS', $result);
        $this->assertStringContains('KNOWLEDGE BASE CONTEXT', $result);
    }

    public function testGetAIResponseIncorporatesAllInputs() {
        // Mock the callAIAPI method to return a predictable response
        $expectedResponse = "Based on your survey data showing you're a business owner and the knowledge base information about cannabis industry trends, here are some key opportunities...";

        $this->chatService->expects($this->once())
            ->method('callAIAPI')
            ->willReturn($expectedResponse);

        $result = $this->chatService->getAIResponse(
            'What business opportunities exist in cannabis?',
            [['user_message' => 'Hello', 'ai_response' => 'Hi there']],
            ['topics' => ['business', 'cannabis']],
            ['BUSINESS OWNER: Prime candidate for B2B offers'],
            'Cannabis industry is growing rapidly with legalization trends.',
            ['name' => 'John Doe', 'answers' => ['10' => 'yes']],
            [['title' => 'Business Consultation', 'cta' => 'Schedule Call']],
            ['tier' => 'free', 'completed_offers' => []]
        );

        $this->assertEquals($expectedResponse, $result);
    }

    public function testSurveyInsightsExtractionLogic() {
        // Test that survey insights are properly extracted
        $controller = new ChatController();
        $reflection = new ReflectionClass(ChatController::class);
        $method = $reflection->getMethod('extractSurveyInsights');
        $method->setAccessible(true);

        // Test business owner detection
        $result = $method->invokeArgs($controller, [
            ['answers' => ['10' => 'yes, I own my own business']]
        ]);

        $this->assertContains('BUSINESS OWNER: Prime candidate for B2B offers and scaling opportunities.', $result);

        // Test debt concerns detection
        $result = $method->invokeArgs($controller, [
            ['answers' => ['04' => 'yes, I have significant debt']]
        ]);

        $this->assertContains('DEBT CONCERNS: Interested in relief solutions and income opportunities.', $result);

        // Test no insights for irrelevant answers
        $result = $method->invokeArgs($controller, [
            ['answers' => ['01' => 'recreational']]
        ]);

        $this->assertEmpty($result);
    }

    public function testRAGContextIntegrationInChatController() {
        $controller = new ChatController();
        $reflection = new ReflectionClass(ChatController::class);
        $method = $reflection->getMethod('getRAGContext');
        $method->setAccessible(true);

        // Mock the RAG service to return predictable context
        $ragServiceMock = $this->createMock(RAGService::class);
        $ragServiceMock->method('query')
            ->willReturn([
                'context' => 'CBD has anti-inflammatory properties supported by clinical studies.',
                'sources' => [['title' => 'Clinical Study 2023', 'url' => 'example.com']],
                'confidence' => 0.95
            ]);

        // We can't easily mock the new RAGService() call, so we'll test that the method exists and handles errors
        $result = $method->invokeArgs($controller, ['What are CBD benefits?', [], ['topics' => ['health']]]);

        // Should return null on error (since RAG service may not be available in test environment)
        // This tests that the error handling works properly
        $this->assertTrue($result === null || is_string($result));
    }
}
