<?php

use PHPUnit\Framework\TestCase;

/**
 * Integration tests for Chat API
 * Tests full request/response cycle for chat endpoints
 */
class ChatApiTest extends TestCase {

    private string $baseUrl = 'http://localhost';
    private array $testSurveyData;
    private array $testUserSession;

    protected function setUp(): void {
        $this->testSurveyData = [
            'name' => 'Test User',
            'answers' => [
                '01' => 'medical use',
                '10' => 'yes, I own a business'
            ]
        ];

        $this->testUserSession = [
            'user_id' => 'test_user_' . uniqid(),
            'completed_offers' => [],
            'rejected_offers' => [],
            'token_count' => 0
        ];
    }

    public function testHealthEndpointReturns200() {
        $response = $this->makeHttpRequest('/api/health');

        $this->assertEquals(200, $response['http_code']);
        $this->assertStringContains('healthy', $response['body']);
    }

    public function testHealthEndpointReturnsJson() {
        $response = $this->makeHttpRequest('/api/health');

        $this->assertEquals(200, $response['http_code']);
        $this->assertJson($response['body']);

        $data = json_decode($response['body'], true);
        $this->assertIsArray($data);
        $this->assertArrayHasKey('status', $data);
        $this->assertArrayHasKey('timestamp', $data);
    }

    public function testChatEndpointRequiresPostMethod() {
        $response = $this->makeHttpRequest('/api/chat', 'GET');

        // Should return 405 Method Not Allowed or handle gracefully
        $this->assertContains($response['http_code'], [405, 404, 500]);
    }

    public function testChatEndpointRequiresMessage() {
        $payload = [
            'history' => [],
            'survey' => $this->testSurveyData
        ];

        $response = $this->makeHttpRequest('/api/chat', 'POST', $payload);

        $this->assertEquals(200, $response['http_code']);
        $this->assertJson($response['body']);

        $data = json_decode($response['body'], true);
        $this->assertIsArray($data);
        $this->assertArrayHasKey('success', $data);
        $this->assertFalse($data['success']);
    }

    public function testChatEndpointAcceptsValidRequest() {
        $payload = [
            'message' => 'What are the benefits of CBD?',
            'history' => [
                ['user_message' => 'Hello', 'ai_response' => 'Hi there!']
            ],
            'preferences' => ['topics' => ['health', 'medical']],
            'survey' => $this->testSurveyData
        ];

        $response = $this->makeHttpRequest('/api/chat', 'POST', $payload);

        $this->assertEquals(200, $response['http_code']);
        $this->assertJson($response['body']);

        $data = json_decode($response['body'], true);
        $this->assertIsArray($data);
        $this->assertArrayHasKey('success', $data);
        $this->assertArrayHasKey('response', $data);
        $this->assertArrayHasKey('user_id', $data);
    }

    public function testChatEndpointReturnsStructuredResponse() {
        $payload = [
            'message' => 'Tell me about cannabis business licenses',
            'history' => [],
            'preferences' => ['topics' => ['business']],
            'survey' => $this->testSurveyData
        ];

        $response = $this->makeHttpRequest('/api/chat', 'POST', $payload);

        $this->assertEquals(200, $response['http_code']);

        $data = json_decode($response['body'], true);
        $this->assertIsArray($data);
        $this->assertTrue($data['success']);
        $this->assertIsString($data['response']);
        $this->assertNotEmpty($data['response']);
        $this->assertIsArray($data['suggestions']);
        $this->assertIsString($data['user_id']);
    }

    public function testChatEndpointHandlesEmptyHistory() {
        $payload = [
            'message' => 'What is THC?',
            'history' => [],
            'preferences' => [],
            'survey' => $this->testSurveyData
        ];

        $response = $this->makeHttpRequest('/api/chat', 'POST', $payload);

        $this->assertEquals(200, $response['http_code']);

        $data = json_decode($response['body'], true);
        $this->assertIsArray($data);
        $this->assertTrue($data['success']);
        $this->assertIsString($data['response']);
    }

    public function testChatEndpointHandlesLargeHistory() {
        $largeHistory = [];
        for ($i = 0; $i < 20; $i++) {
            $largeHistory[] = [
                'user_message' => "Question $i",
                'ai_response' => "Answer $i"
            ];
        }

        $payload = [
            'message' => 'Follow up question',
            'history' => $largeHistory,
            'preferences' => [],
            'survey' => $this->testSurveyData
        ];

        $response = $this->makeHttpRequest('/api/chat', 'POST', $payload);

        $this->assertEquals(200, $response['http_code']);

        $data = json_decode($response['body'], true);
        $this->assertIsArray($data);
        $this->assertTrue($data['success']);
    }

    public function testChatEndpointIncludesOffersWhenRelevant() {
        $payload = [
            'message' => 'I want to start a cannabis business and need help with licensing',
            'history' => [],
            'preferences' => ['topics' => ['business', 'license']],
            'survey' => $this->testSurveyData
        ];

        $response = $this->makeHttpRequest('/api/chat', 'POST', $payload);

        $this->assertEquals(200, $response['http_code']);

        $data = json_decode($response['body'], true);
        $this->assertIsArray($data);
        $this->assertTrue($data['success']);

        // Should potentially include offers for business-related queries
        if (isset($data['offers'])) {
            $this->assertIsArray($data['offers']);
        }
    }

    public function testChatEndpointHandlesMalformedJson() {
        $response = $this->makeHttpRequest('/api/chat', 'POST', 'invalid json');

        // Should handle gracefully
        $this->assertContains($response['http_code'], [200, 400, 500]);

        if ($response['http_code'] === 200) {
            $data = json_decode($response['body'], true);
            $this->assertIsArray($data);
            $this->assertFalse($data['success']);
        }
    }

    public function testChatEndpointRateLimiting() {
        // Make multiple rapid requests to test rate limiting
        $payload = [
            'message' => 'Test message',
            'history' => [],
            'preferences' => [],
            'survey' => $this->testSurveyData
        ];

        $responses = [];
        for ($i = 0; $i < 10; $i++) {
            $responses[] = $this->makeHttpRequest('/api/chat', 'POST', $payload);
            usleep(100000); // 100ms delay
        }

        // At least some requests should succeed
        $successCount = 0;
        foreach ($responses as $response) {
            if ($response['http_code'] === 200) {
                $data = json_decode($response['body'], true);
                if (isset($data['success']) && $data['success']) {
                    $successCount++;
                }
            }
        }

        $this->assertGreaterThan(0, $successCount, 'At least some requests should succeed');
    }

    public function testCORSHeadersAreSet() {
        $response = $this->makeHttpRequest('/api/chat', 'OPTIONS');

        // CORS headers should be set
        $this->assertArrayHasKey('Access-Control-Allow-Origin', $response['headers']);
        $this->assertArrayHasKey('Access-Control-Allow-Methods', $response['headers']);
        $this->assertArrayHasKey('Access-Control-Allow-Headers', $response['headers']);
    }

    /**
     * Make HTTP request to test endpoint
     */
    private function makeHttpRequest(string $path, string $method = 'GET', $data = null): array {
        $url = $this->baseUrl . $path;

        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HEADER, true);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);

        if ($data !== null) {
            $jsonData = is_string($data) ? $data : json_encode($data);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $jsonData);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json',
                'Content-Length: ' . strlen($jsonData)
            ]);
        }

        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // For testing

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
        curl_close($ch);

        $headers = substr($response, 0, $headerSize);
        $body = substr($response, $headerSize);

        // Parse headers
        $parsedHeaders = [];
        $headerLines = explode("\r\n", trim($headers));
        foreach ($headerLines as $line) {
            if (strpos($line, ':') !== false) {
                [$key, $value] = explode(':', $line, 2);
                $parsedHeaders[trim($key)] = trim($value);
            }
        }

        return [
            'http_code' => $httpCode,
            'body' => $body,
            'headers' => $parsedHeaders,
            'full_response' => $response
        ];
    }

    public function testNonexistentEndpointReturns404() {
        $response = $this->makeHttpRequest('/api/nonexistent');

        $this->assertEquals(404, $response['http_code']);
    }

    public function testRootEndpointServesApplication() {
        $response = $this->makeHttpRequest('/');

        // Should return HTML or redirect
        $this->assertContains($response['http_code'], [200, 302]);
    }
}
