<?php
// Ensure clean JSON output
ob_start();
header('Content-Type: application/json; charset=UTF-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

// Start session for rate limiting
session_start();

// Load environment variables
// Detect if we're on live server (check both HTTP_HOST and server IP/domain)
$is_live = false;
if (isset($_SERVER['HTTP_HOST']) && strpos($_SERVER['HTTP_HOST'], 'deepleafai.com') !== false) {
	$is_live = true;
} elseif (isset($_SERVER['SERVER_NAME']) && strpos($_SERVER['SERVER_NAME'], 'deepleafai.com') !== false) {
	$is_live = true;
} elseif (gethostname() === 'deepleafai.com' || strpos(gethostname(), 'deepleafai') !== false) {
	$is_live = true;
}

// Try to load live config first, then fallback to general config
$envFiles = [];
if ($is_live && file_exists(__DIR__ . '/config.live.env')) {
	$envFiles[] = __DIR__ . '/config.live.env';
}
$envFiles[] = __DIR__ . '/config.env';

foreach ($envFiles as $envFile) {
if (file_exists($envFile)) {
	$lines = file($envFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
	foreach ($lines as $line) {
		if (strpos($line, '#') === 0 || strpos($line, '=') === false) continue;

			// Handle shell variable syntax ${VAR:-default}
			if (preg_match('/^\s*([A-Z_]+)\s*=\s*\$\{([^:-]+)(?::-([^}]*))?\}/', $line, $matches)) {
				$key = trim($matches[1]);
				$varName = trim($matches[2]);
				$defaultValue = isset($matches[3]) ? trim($matches[3]) : '';
				$value = getenv($varName) ?: $defaultValue;
			} else {
		list($key, $value) = explode('=', $line, 2);
		$key = trim($key);
		$value = trim($value);
			}

		if (!getenv($key)) {
			putenv("$key=$value");
		}
		}
		break; // Stop after first successful config load
	}
}

// Input validation
$email = trim($_POST["email"] ?? '');
if(empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
	ob_clean();
	echo json_encode(['error' => 'Please enter a valid email address']);
	exit;
}
$normalizedEmail = strtolower($email);

// Rate limiting keys
$email_rate_limit_key = 'email_rate_limit_' . md5($normalizedEmail);
$ip_rate_limit_key = 'email_rate_limit_' . ($_SERVER['REMOTE_ADDR'] ?? 'unknown');

if((isset($_SESSION[$email_rate_limit_key]) && (time() - $_SESSION[$email_rate_limit_key]) < 60) ||
   (isset($_SESSION[$ip_rate_limit_key]) && (time() - $_SESSION[$ip_rate_limit_key]) < 30)){
	ob_clean();
	echo json_encode(['error' => 'Please wait before requesting another verification code.']);
	exit;
}

$_SESSION[$email_rate_limit_key] = time();
$_SESSION[$ip_rate_limit_key] = time();

// Feature flag
$email_enabled = 'true'; // Email verification enabled
if($email_enabled !== 'true') {
	ob_clean();
	echo json_encode(['error' => 'Email verification temporarily unavailable']);
	exit;
}

try {
	$randomNumber = random_int(100000, 999999);

	// Generate unsubscribe URL BEFORE building message content (for DKIM consistency)
    // Ensure unsubscribe manager is available before use
    if (!class_exists('UnsubscribeManager')) {
        require_once __DIR__ . '/unsubscribe_manager.php';
    }
    $unsubscribe_manager = new UnsubscribeManager();
    $unsubscribe_url = $unsubscribe_manager->generateUnsubscribeUrl($normalizedEmail);

	$subject = "DeepLeaf AI - Email Verification Code";
	$message = "<html><body>";
	$message .= "<div style='font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px;'>";
	$message .= "<h2 style='color: #2d3748; text-align: center; margin-bottom: 10px;'>🌱 Hello!</h2>";
	$message .= "<p style='font-size: 16px; color: #4a5568; text-align: center;'>Welcome to DeepLeaf AI</p>";
	$message .= "<br>";
	$message .= "<p style='font-size: 16px; color: #4a5568; text-align: center;'>🔐 Your verification code is:</p>";
	$message .= "<div style='background: linear-gradient(135deg, #f7fafc 0%, #e2e8f0 100%); border: 2px solid #2b6cb0; border-radius: 12px; padding: 30px; text-align: center; margin: 25px 0;'>";
	$message .= "<span style='font-size: 36px; font-weight: bold; color: #2b6cb0; letter-spacing: 6px; font-family: \"Courier New\", monospace;'>" . $randomNumber . "</span>";
	$message .= "</div>";
	$message .= "<p style='font-size: 14px; color: #718096; text-align: center;'>⏰ This code will expire in <strong>10 minutes</strong></p>";
	$message .= "<br>";
	$message .= "<p style='font-size: 14px; color: #718096; text-align: center; font-style: italic;'>If you didn't request this code, please ignore this email.</p>";
	$message .= "<hr style='border: none; border-top: 2px solid #e2e8f0; margin: 40px 0;'>";

	// Add unsubscribe link for CAN-SPAM compliance
	$message .= "<div style='text-align: center; margin: 20px 0; padding: 15px; background: #f8f9fa; border-radius: 8px;'>";
	$message .= "<p style='font-size: 12px; color: #6c757d; margin: 0;'>";
	$message .= "Don't want to receive these emails? ";
	$message .= "<a href='{$unsubscribe_url}' style='color: #dc3545; text-decoration: underline;'>Unsubscribe here</a>";
	$message .= "</p>";
	$message .= "<p style='font-size: 11px; color: #9ca3af; margin: 5px 0 0 0;'>";
	$message .= "You can unsubscribe at any time. Your email address will be immediately removed from our mailing list.";
	$message .= "</p>";
	$message .= "</div>";

	$message .= "<div style='text-align: center;'>";
	$message .= "<p style='font-size: 14px; color: #4a5568; margin-bottom: 5px;'>📧 Best regards,</p>";
	$message .= "<p style='font-size: 16px; color: #2d3748; margin-bottom: 5px;'><strong>DeepLeaf AI Team</strong></p>";
	$message .= "<p style='font-size: 14px; color: #4a5568;'><a href='https://www.deepleafai.com' style='color: #3182ce; text-decoration: none;'>🌐 www.deepleafai.com</a></p>";
	$message .= "</div>";
	$message .= "</div></body></html>";

function sendGmailSMTP($to, $subject, $message, $from) {
	// Gmail SMTP configuration
	$smtp_host = 'smtp.gmail.com';
	$smtp_port = 587;
	$smtp_username = 'deepleafai@gmail.com';
	$smtp_password = 'pzzj qdbo ppaw lvot'; // App password provided by user (with spaces as given)

	try {
		// Create socket connection
		$socket = fsockopen($smtp_host, $smtp_port, $errno, $errstr, 30);
		if (!$socket) {
			error_log("Gmail SMTP connection failed: $errstr ($errno)");
			return false;
		}

		// Read server greeting
		$response = fgets($socket, 512);
		if (!preg_match('/^220/', $response)) {
			error_log("Gmail SMTP connection failed: $response");
			fclose($socket);
			return false;
		}

		// Send EHLO
		fputs($socket, "EHLO deepleafai.com\r\n");
		$response = '';
		while ($line = fgets($socket, 512)) {
			$response .= $line;
			if (preg_match('/^250\s/', $line)) break;
		}
		if (!preg_match('/^250/', $response)) {
			error_log("Gmail SMTP EHLO failed");
			fclose($socket);
			return false;
		}

		// Start TLS
		fputs($socket, "STARTTLS\r\n");
		$response = fgets($socket, 512);
		if (!preg_match('/^220/', $response)) {
			error_log("Gmail SMTP STARTTLS failed");
			fclose($socket);
			return false;
		}

		// Enable crypto
		stream_socket_enable_crypto($socket, true, STREAM_CRYPTO_METHOD_TLS_CLIENT);

		// Send EHLO again after TLS
		fputs($socket, "EHLO deepleafai.com\r\n");
		$response = '';
		while ($line = fgets($socket, 512)) {
			$response .= $line;
			if (preg_match('/^250\s/', $line)) break;
		}

		// Authenticate
		fputs($socket, "AUTH LOGIN\r\n");
		$response = fgets($socket, 512);
		if (!preg_match('/^334/', $response)) {
			error_log("Gmail SMTP authentication failed");
			fclose($socket);
			return false;
		}

		// Send username (base64 encoded)
		fputs($socket, base64_encode($smtp_username) . "\r\n");
		$response = fgets($socket, 512);
		if (!preg_match('/^334/', $response)) {
			error_log("Gmail SMTP username authentication failed");
			fclose($socket);
			return false;
		}

		// Send password (base64 encoded)
		fputs($socket, base64_encode($smtp_password) . "\r\n");
		$response = fgets($socket, 512);
		if (!preg_match('/^235/', $response)) {
			error_log("Gmail SMTP password authentication failed");
			fclose($socket);
			return false;
		}

		// Send MAIL FROM
		fputs($socket, "MAIL FROM:<$from>\r\n");
		$response = fgets($socket, 512);
		if (!preg_match('/^250/', $response)) {
			error_log("Gmail SMTP MAIL FROM failed: $response");
			fclose($socket);
			return false;
		}

		// Send RCPT TO
		fputs($socket, "RCPT TO:<$to>\r\n");
		$response = fgets($socket, 512);
		if (!preg_match('/^250/', $response)) {
			error_log("Gmail SMTP RCPT TO failed: $response");
			fclose($socket);
			return false;
		}

		// Send DATA
		fputs($socket, "DATA\r\n");
		$response = fgets($socket, 512);
		if (!preg_match('/^354/', $response)) {
			error_log("Gmail SMTP DATA failed: $response");
			fclose($socket);
			return false;
		}

		// Send email content
		$email_content = "From: DeepLeaf AI <$from>\r\n";
		$email_content .= "To: $to\r\n";
		$email_content .= "Subject: $subject\r\n";
        $email_content .= "MIME-Version: 1.0\r\n";
        $email_content .= "Content-Type: text/html; charset=UTF-8\r\n";
        $email_content .= "Content-Transfer-Encoding: base64\r\n";
		$email_content .= "Reply-To: support@deepleafai.com\r\n";
        $email_content .= "\r\n";
        $email_content .= rtrim(chunk_split(base64_encode($message)));
		$email_content .= "\r\n.\r\n";

		fputs($socket, $email_content);

		// Send QUIT
		fputs($socket, "QUIT\r\n");
		$response = fgets($socket, 512);

		fclose($socket);
		return true;

	} catch (Exception $e) {
		error_log("Gmail SMTP exception: " . $e->getMessage());
		return false;
	}
}

    // Check if email is unsubscribed
    if ($unsubscribe_manager->isUnsubscribed($normalizedEmail)) {
		// Email is unsubscribed, don't send
		error_log("Email not sent - $normalizedEmail is unsubscribed");
		echo json_encode(['success' => false, 'message' => 'Email is unsubscribed']);
		exit;
	}

    // HARDCODE DOMAIN TO ENSURE CORRECT EMAIL HEADERS
    $from = 'noreply@deepleafai.com';
    $replyTo = 'support@deepleafai.com';

	$headers = "From: DeepLeaf AI <{$from}>\r\n";
    $headers .= "Reply-To: {$replyTo}\r\n";
    $headers .= "MIME-Version: 1.0\r\n";
	$headers .= "Content-Type: text/html; charset=UTF-8\r\n";
    $headers .= "Content-Transfer-Encoding: base64\r\n";
	$headers .= "X-Mailer: PHP/" . phpversion();

	// Check if this is a resend attempt
	$is_resend = isset($_POST['resend']) && $_POST['resend'] === 'true';

    // Use envelope sender to align SPF/DMARC with From domain
    // Base64-encode body to prevent in-transit transformations that break DKIM
    $encoded_message = rtrim(chunk_split(base64_encode($message)));

    $additional_params = '-f ' . escapeshellarg($from);

    // Add debug logging
    error_log("Email sending debug:");
    error_log("  To: $normalizedEmail");
    error_log("  From: $from");
    error_log("  Subject: $subject");
    error_log("  Headers: $headers");
    error_log("  Additional params: $additional_params");

    $mail_sent = mail($normalizedEmail, $subject, $encoded_message, $headers, $additional_params);

	if(!$mail_sent) {
		// Try Gmail SMTP fallback if this is a resend attempt
		if($is_resend) {
			$mail_sent = sendGmailSMTP($normalizedEmail, $subject, $message, $from);
			if(!$mail_sent) {
				throw new Exception("Failed to send email via both primary and Gmail SMTP fallback");
			}
		} else {
		throw new Exception("Failed to send email");
		}
	}

	// Clear any previous tokens for same email
	foreach ($_SESSION as $key => $value) {
		if (strpos($key, 'email_verification_') === 0) {
			if (is_array($value) && isset($value['email']) && $value['email'] === $normalizedEmail) {
				unset($_SESSION[$key]);
			}
		}
	}

	$verification_token = bin2hex(random_bytes(32));
	$_SESSION['email_verification_' . $verification_token] = [
		'code' => password_hash((string)$randomNumber, PASSWORD_DEFAULT),
		'email' => $normalizedEmail,
		'timestamp' => time(),
		'attempts' => 0,
		'survey_token' => $_POST['survey_token'] ?? null
	];

	ob_clean();
	echo json_encode(['success' => true, 'token' => $verification_token]);

} catch (Exception $e) {
	error_log("Email sending error: " . $e->getMessage());
	ob_clean();
	echo json_encode(['error' => 'Failed to send email. Please try again.']);
}

exit;
?>