<?php

class FraudDetection {
    private $db;
    
    public function __construct($db) {
        $this->db = $db;
    }
    
    public function checkClickFraud($affiliate_id, $offer_id, $visitor_id, $ip_address, $user_agent) {
        $fraud_score = 0;
        $reasons = [];
        
        // Check for excessive clicks from same IP
        $stmt = $this->db->prepare("
            SELECT COUNT(*) as click_count 
            FROM tracking_events 
            WHERE ip_address = ? AND event_type = 'click' AND created_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)
        ");
        $stmt->execute([$ip_address]);
        $result = $stmt->fetch();
        
        if ($result['click_count'] > 50) {
            $fraud_score += 50;
            $reasons[] = 'Excessive clicks from IP';
        }
        
        // Check for clicks from same visitor to same offer
        $stmt = $this->db->prepare("
            SELECT COUNT(*) as visitor_clicks 
            FROM tracking_events 
            WHERE visitor_id = ? AND offer_id = ? AND event_type = 'click' AND created_at > DATE_SUB(NOW(), INTERVAL 24 HOUR)
        ");
        $stmt->execute([$visitor_id, $offer_id]);
        $result = $stmt->fetch();
        
        if ($result['visitor_clicks'] > 3) {
            $fraud_score += 30;
            $reasons[] = 'Multiple clicks from same visitor';
        }
        
        // Check for suspicious user agent patterns
        if (empty($user_agent) || $this->isBotUserAgent($user_agent)) {
            $fraud_score += 40;
            $reasons[] = 'Suspicious user agent';
        }
        
        // Check affiliate click patterns
        $stmt = $this->db->prepare("
            SELECT COUNT(*) as affiliate_clicks 
            FROM tracking_events 
            WHERE affiliate_id = ? AND event_type = 'click' AND created_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)
        ");
        $stmt->execute([$affiliate_id]);
        $result = $stmt->fetch();
        
        if ($result['affiliate_clicks'] > 100) {
            $fraud_score += 25;
            $reasons[] = 'Unusual affiliate activity';
        }
        
        return [
            'is_fraud' => $fraud_score >= 60,
            'score' => $fraud_score,
            'reasons' => $reasons
        ];
    }
    
    public function checkCallFraud($phone_number, $affiliate_id, $tracking_id) {
        $fraud_score = 0;
        $reasons = [];
        
        // Check for repeated calls from same number
        $stmt = $this->db->prepare("
            SELECT COUNT(*) as call_count 
            FROM tracking_events 
            WHERE caller_number = ? AND event_type LIKE 'call%' AND created_at > DATE_SUB(NOW(), INTERVAL 24 HOUR)
        ");
        $stmt->execute([$phone_number]);
        $result = $stmt->fetch();
        
        if ($result['call_count'] > 5) {
            $fraud_score += 45;
            $reasons[] = 'Repeated calls from same number';
        }
        
        return [
            'is_fraud' => $fraud_score >= 40,
            'score' => $fraud_score,
            'reasons' => $reasons
        ];
    }
    
    private function isBotUserAgent($user_agent) {
        $bot_patterns = [
            'bot', 'crawler', 'spider', 'scraper', 'curl', 'wget', 
            'python', 'java/', 'go-http-client', 'headless'
        ];
        
        $user_agent_lower = strtolower($user_agent);
        
        foreach ($bot_patterns as $pattern) {
            if (strpos($user_agent_lower, $pattern) !== false) {
                return true;
            }
        }
        
        return false;
    }
    
    public function logFraudAttempt($type, $data) {
        $stmt = $this->db->prepare("
            INSERT INTO fraud_logs (event_type, fraud_data, detected_at) 
            VALUES (?, ?, NOW())
        ");
        
        $stmt->execute([$type, json_encode($data)]);
    }
}
?>
