<?php
/**
 * Secure Configuration Parser
 * Loads configuration from config.env file with environment variable support
 */

/**
 * Load configuration from config.env file
 * Supports environment variable substitution using ${VAR_NAME} or ${VAR_NAME:-default} syntax
 *
 * @return array Configuration array
 */
function load_config() {
    $config = [];

    // Default config file path
    $config_file = __DIR__ . '/config.env';

    if (!file_exists($config_file)) {
        // Fallback: try to load from environment variables directly
        return load_config_from_env();
    }

    $lines = file($config_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);

    foreach ($lines as $line) {
        // Skip comments
        $line = trim($line);
        if (empty($line) || $line[0] === '#') {
            continue;
        }

        // Parse key=value pairs
        if (strpos($line, '=') !== false) {
            list($key, $value) = explode('=', $line, 2);
            $key = trim($key);
            $value = trim($value);

            // Process environment variable substitution
            $value = process_env_vars($value);

            $config[$key] = $value;
        }
    }

    return $config;
}

/**
 * Process environment variable substitution in config values
 * Supports ${VAR_NAME} and ${VAR_NAME:-default_value} syntax
 *
 * @param string $value The value to process
 * @return string The processed value
 */
function process_env_vars($value) {
    // Pattern for ${VAR_NAME} or ${VAR_NAME:-default}
    $pattern = '/\$\{([^}]+)\}/';

    return preg_replace_callback($pattern, function($matches) {
        $var_expr = $matches[1];

        // Check for default value syntax: VAR_NAME:-default
        if (strpos($var_expr, ':-') !== false) {
            list($var_name, $default) = explode(':-', $var_expr, 2);
            return getenv($var_name) ?: $default;
        }

        // Simple variable substitution
        return getenv($var_expr) ?: '';
    }, $value);
}

/**
 * Fallback function to load configuration from environment variables
 * Used when config.env file is not available
 *
 * @return array Configuration array
 */
function load_config_from_env() {
    return [
        'DB_HOST' => getenv('DB_HOST') ?: 'localhost',
        'DB_USER' => getenv('DB_USER') ?: 'leadgen',
        'DB_PASS' => getenv('DB_PASS') ?: $_ENV['DB_PASS'] ?? 'REPLACE_WITH_ACTUAL_PASSWORD',
        'DB_NAME' => getenv('DB_NAME') ?: 'loudcannabisnear_survey',

        'EMAIL_VERIFICATION_ENABLED' => getenv('EMAIL_VERIFICATION_ENABLED') ?: 'true',
        'EMAIL_FROM' => getenv('EMAIL_FROM') ?: 'noreply@deepleafai.com',
        'EMAIL_NAME' => getenv('EMAIL_NAME') ?: 'Email Verification',

        'APP_SECRET_KEY' => getenv('APP_SECRET_KEY') ?: 'your_secure_random_key',
        'SESSION_LIFETIME' => getenv('SESSION_LIFETIME') ?: '3600',
        'RATE_LIMIT_WINDOW' => getenv('RATE_LIMIT_WINDOW') ?: '60',
        'MAX_LOGIN_ATTEMPTS' => getenv('MAX_LOGIN_ATTEMPTS') ?: '5',

        'AI_PROVIDER' => getenv('AI_PROVIDER') ?: 'openai',
        'OPENAI_API_KEY' => getenv('OPENAI_API_KEY') ?: '',
        'OPENAI_MODEL' => getenv('OPENAI_MODEL') ?: 'gpt-4',
        'OPENAI_TEMPERATURE' => getenv('OPENAI_TEMPERATURE') ?: '0.7',
        'OPENAI_MAX_TOKENS' => getenv('OPENAI_MAX_TOKENS') ?: '300',

        'AI_PERSONA' => getenv('AI_PERSONA') ?: 'street_budtender',
        'AI_NAME' => getenv('AI_NAME') ?: 'StreetWise Budtender',
        'AI_ROLE' => getenv('AI_ROLE') ?: 'No-Bullshit Cannabis Expert & Truth Teller',
        'AI_TONE' => getenv('AI_TONE') ?: 'blunt, street-smart, knowledgeable, direct, rude_when_necessary',
        'AI_EXPERTISE' => getenv('AI_EXPERTISE') ?: 'cannabis_products, medical_cannabis, dispensaries, strains, effects, industry_scams, real_deals'
    ];
}

/**
 * Validate configuration values
 * Ensures required configuration is present and valid
 *
 * @param array $config Configuration array
 * @return array Array of validation errors (empty if valid)
 */
function validate_config($config) {
    $errors = [];

    // Required database configuration
    $required_db = ['DB_HOST', 'DB_USER', 'DB_PASS', 'DB_NAME'];
    foreach ($required_db as $key) {
        if (empty($config[$key])) {
            $errors[] = "Missing required database configuration: $key";
        }
    }

    // Required security configuration
    if (empty($config['APP_SECRET_KEY']) || $config['APP_SECRET_KEY'] === 'your_secure_random_key') {
        $errors[] = "APP_SECRET_KEY must be set to a secure random value";
    }

    return $errors;
}
?>