<?php

/**
 * RAG Service
 * Handles RAG (Retrieval-Augmented Generation) integration
 * Communicates with Python RAG backend for context retrieval
 */

class RAGService {

    private string $ragEndpoint = 'http://localhost:5000/query';
    private bool $ragEnabled = true;

    public function __construct() {
        // Check if RAG service is available
        $this->ragEnabled = $this->checkRAGAvailability();
    }

    /**
     * Query the RAG system for context
     */
    public function query(string $query, array $history = [], array $preferences = []): array {
        if (!$this->ragEnabled) {
            return [
                'context' => null,
                'sources' => [],
                'confidence' => 0.0
            ];
        }

        try {
            $payload = $this->buildRAGPayload($query, $history, $preferences);
            $response = $this->callRAGAPI($payload);

            return $this->processRAGResponse($response);

        } catch (Exception $e) {
            error_log("RAG query error: " . $e->getMessage());
            return [
                'context' => null,
                'sources' => [],
                'confidence' => 0.0
            ];
        }
    }

    /**
     * Build payload for RAG API
     */
    private function buildRAGPayload(string $query, array $history, array $preferences): array {
        // Extract recent conversation context
        $conversationContext = '';
        if (!empty($history)) {
            $recentMessages = array_slice($history, -5);
            foreach ($recentMessages as $msg) {
                if (isset($msg['text'])) {
                    $conversationContext .= $msg['text'] . ' ';
                } elseif (isset($msg['user_message'])) {
                    $conversationContext .= $msg['user_message'] . ' ' . ($msg['ai_response'] ?? '') . ' ';
                }
            }
        }

        return [
            'query' => $query,
            'conversation_context' => trim($conversationContext),
            'preferences' => $preferences,
            'top_k' => 5,
            'include_sources' => true
        ];
    }

    /**
     * Call RAG API
     */
    private function callRAGAPI(array $payload): array {
        $ch = curl_init($this->ragEndpoint);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . ($this->getRAGAuthToken() ?? '')
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 15); // Shorter timeout for RAG

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            throw new RuntimeException("RAG API connection error: $error");
        }

        if ($httpCode !== 200) {
            throw new RuntimeException("RAG API error: HTTP $httpCode");
        }

        $data = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new RuntimeException("RAG API returned invalid JSON");
        }

        return $data;
    }

    /**
     * Process RAG API response
     */
    private function processRAGResponse(array $response): array {
        $context = $response['context'] ?? '';
        $sources = $response['sources'] ?? [];
        $confidence = $response['confidence'] ?? 0.0;

        // Filter and format sources
        $formattedSources = [];
        foreach ($sources as $source) {
            if (is_array($source) && isset($source['title'])) {
                $formattedSources[] = [
                    'title' => $source['title'],
                    'url' => $source['url'] ?? '',
                    'relevance' => $source['score'] ?? 0.0
                ];
            }
        }

        // Enhance context with source attribution
        if (!empty($formattedSources)) {
            $context .= "\n\nSources:\n";
            foreach ($formattedSources as $source) {
                $context .= "- {$source['title']}\n";
            }
        }

        return [
            'context' => $context,
            'sources' => $formattedSources,
            'confidence' => $confidence
        ];
    }

    /**
     * Check if RAG service is available
     */
    private function checkRAGAvailability(): bool {
        try {
            $ch = curl_init($this->ragEndpoint . '/health');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 5);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            return $httpCode === 200;

        } catch (Exception $e) {
            error_log("RAG health check failed: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Get RAG authentication token
     */
    private function getRAGAuthToken(): ?string {
        // TODO: Implement proper RAG authentication
        // For now, return null (no auth required)
        return null;
    }

    /**
     * Get RAG system status
     */
    public function getStatus(): array {
        return [
            'enabled' => $this->ragEnabled,
            'endpoint' => $this->ragEndpoint,
            'available' => $this->checkRAGAvailability()
        ];
    }

    /**
     * Update RAG configuration
     */
    public function updateConfiguration(array $config): void {
        if (isset($config['endpoint'])) {
            $this->ragEndpoint = $config['endpoint'];
        }
        if (isset($config['enabled'])) {
            $this->ragEnabled = (bool) $config['enabled'];
        }
    }
}
