<?php

/**
 * Offer Service
 * Handles monetization offer analysis and recommendations
 * Extracted from openai_chat.php for better separation of concerns
 */

class OfferService {

    /**
     * Analyze conversation for offer recommendations
     */
    public function analyzeForOffers(
        string $message,
        array $preferences,
        array $history,
        array $surveyData = [],
        array $userSession = []
    ): array {
        $textSources = [$message];
        foreach ($history as $entry) {
            if (!empty($entry['text'])) {
                $textSources[] = $entry['text'];
            } elseif (!empty($entry['user_message'])) {
                $textSources[] = $entry['user_message'];
                $textSources[] = $entry['ai_response'] ?? '';
            }
        }

        $combinedText = strtolower(implode(' ', $textSources));
        $topics = array_map('strtolower', $preferences['topics'] ?? []);

        $offerCatalog = $this->getOfferCatalog();
        $rejectedOffers = $userSession['rejected_offers'] ?? [];
        $completedOffers = $userSession['completed_offers'] ?? [];
        $excludedOffers = array_merge($rejectedOffers, $completedOffers);

        $scores = [];
        foreach ($offerCatalog as $offerId => $offer) {
            if (in_array($offerId, $excludedOffers)) {
                continue;
            }

            $score = $this->calculateOfferScore($offer, $combinedText, $topics, $surveyData);
            if ($score > 0) {
                $scores[] = [
                    'id' => $offerId,
                    'score' => $score,
                    'offer' => $offer
                ];
            }
        }

        if (empty($scores)) {
            return [];
        }

        usort($scores, function ($a, $b) {
            return $b['score'] <=> $a['score'];
        });

        $filtered = array_filter($scores, function ($entry) {
            return $entry['score'] >= 3;
        });

        if (empty($filtered)) {
            $filtered = [$scores[0]];
        }

        $topOffers = array_slice($filtered, 0, 1);
        return $this->formatOfferResults($topOffers);
    }

    /**
     * Calculate relevance score for an offer
     */
    private function calculateOfferScore(array $offer, string $combinedText, array $topics, array $surveyData): float {
        $score = 0.0;

        // Keyword matching
        foreach ($offer['keywords'] as $keyword) {
            if (strpos($combinedText, $keyword) !== false) {
                $score += 2;
            }
        }

        // Topic matching
        foreach ($offer['topic_matches'] as $topic) {
            if (in_array($topic, $topics, true)) {
                $score += 1;
            }
        }

        // Survey signal matching
        $score += $this->calculateSurveyScore($offer, $surveyData);

        // Apply payout multiplier for value-based scoring
        if ($score > 0) {
            $payoutMultiplier = 1 + ($offer['payout'] / 100);
            $score *= $payoutMultiplier;
        }

        return $score;
    }

    /**
     * Calculate survey-based scoring
     */
    private function calculateSurveyScore(array $offer, array $surveyData): float {
        if (empty($surveyData['answers'])) {
            return 0.0;
        }

        $score = 0.0;
        $answers = $surveyData['answers'];

        foreach ($offer['survey_signals'] as $signal) {
            $questionId = $signal['question'];
            if (!empty($answers[$questionId])) {
                $answerLower = strtolower($answers[$questionId]);
                foreach ($signal['needles'] as $needle) {
                    if (strpos($answerLower, strtolower($needle)) !== false) {
                        $score += 3;
                        break 2;
                    }
                }
            }
        }

        return $score;
    }

    /**
     * Format offer results for API response
     */
    private function formatOfferResults(array $offers): array {
        return array_map(function ($entry) {
            $offer = $entry['offer'];
            $offer['id'] = $entry['id'];
            $offer['score'] = $entry['score'];

            if (empty($offer['reason']) && !empty($offer['cta'])) {
                $offer['reason'] = $offer['cta'];
            }

            return $offer;
        }, $offers);
    }

    /**
     * Get the complete offer catalog
     */
    private function getOfferCatalog(): array {
        return [
            'cannabis_business_call' => [
                'type' => 'phone',
                'title' => 'Cannabis Business Strategy Call',
                'url' => 'tel:+18001234567',
                'cta' => 'Talk with a strategist about licenses, funding, or scaling.',
                'keywords' => ['business', 'start', 'license', 'grow', 'dispensary', 'scale', 'wholesale', 'brand'],
                'topic_matches' => ['business', 'entrepreneur', 'b2b'],
                'survey_signals' => [
                    ['question' => '10', 'needles' => ['yes', 'own', 'plan', 'start', 'business']],
                    ['question' => '09', 'needles' => ['money', 'income', 'profit']]
                ],
                'payout' => 45.00,
                'points' => 50
            ],
            'medical_wellness_program' => [
                'type' => 'website',
                'title' => 'Medical Cannabis Wellness Program',
                'url' => 'https://example.com/medical-cannabis-guide',
                'cta' => 'Get doctor-backed dosing plans and high-CBD product bundles.',
                'keywords' => ['medical', 'pain', 'health', 'anxiety', 'doctor', 'symptom', 'dose'],
                'topic_matches' => ['health', 'medical', 'wellness'],
                'survey_signals' => [
                    ['question' => '01', 'needles' => ['medical', 'therapy']],
                    ['question' => '03', 'needles' => ['herbal', 'supplement', 'holistic']]
                ],
                'payout' => 25.00,
                'points' => 25
            ],
            'debt_relief_bundle' => [
                'type' => 'website',
                'title' => 'Debt Relief & Side Income Bundle',
                'url' => 'https://example.com/debt-relief',
                'cta' => 'Bundle includes debt cleanup plus income streams tied to cannabis affiliates.',
                'keywords' => ['debt', 'finance', 'budget', 'bills', 'credit', 'money stress'],
                'topic_matches' => ['debt', 'finance', 'money'],
                'survey_signals' => [
                    ['question' => '04', 'needles' => ['debt', 'struggle', 'hard']],
                    ['question' => '09', 'needles' => ['money', 'income']]
                ],
                'payout' => 60.00,
                'points' => 75
            ],
            'solar_grow_energy' => [
                'type' => 'phone',
                'title' => 'Grow Room Solar Upgrade',
                'url' => 'tel:+18001234567',
                'cta' => 'Slash grow costs with a utility-grade solar partner.',
                'keywords' => ['solar', 'energy', 'electric', 'power bill', 'sustainable', 'green'],
                'topic_matches' => ['solar', 'energy', 'green'],
                'survey_signals' => [
                    ['question' => '08', 'needles' => ['yes', 'interested', 'very']]
                ],
                'payout' => 30.00,
                'points' => 30
            ]
        ];
    }

    /**
     * Get personalized offers using memory system (future enhancement)
     */
    public function getPersonalizedOffers(string $userId, array $context = []): array {
        // TODO: Integrate with memory system for personalized offers
        // For now, return empty array
        return [];
    }

    /**
     * Record offer interaction for analytics
     */
    public function recordOfferInteraction(string $userId, string $offerId, string $action, array $context = []): void {
        try {
            error_log("Offer interaction: User $userId, Offer $offerId, Action $action");
            // TODO: Implement proper analytics tracking
        } catch (Exception $e) {
            error_log("Offer analytics error: " . $e->getMessage());
        }
    }
}
