<?php

/**
 * Logger Service
 * Comprehensive logging system with multiple levels and outputs
 */

class Logger {

    public const LEVEL_DEBUG = 'DEBUG';
    public const LEVEL_INFO = 'INFO';
    public const LEVEL_WARNING = 'WARNING';
    public const LEVEL_ERROR = 'ERROR';
    public const LEVEL_CRITICAL = 'CRITICAL';

    private string $logPath;
    private string $minLevel;
    private bool $consoleOutput;

    public function __construct(
        string $logPath = null,
        string $minLevel = self::LEVEL_INFO,
        bool $consoleOutput = false
    ) {
        $this->logPath = $logPath ?? (defined('LOGS_PATH') ? LOGS_PATH . '/app.log' : __DIR__ . '/../../storage/logs/app.log');
        $this->minLevel = $minLevel;
        $this->consoleOutput = $consoleOutput;

        // Ensure log directory exists
        $logDir = dirname($this->logPath);
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }
    }

    /**
     * Log a debug message
     */
    public function debug(string $message, array $context = []): void {
        $this->log(self::LEVEL_DEBUG, $message, $context);
    }

    /**
     * Log an info message
     */
    public function info(string $message, array $context = []): void {
        $this->log(self::LEVEL_INFO, $message, $context);
    }

    /**
     * Log a warning message
     */
    public function warning(string $message, array $context = []): void {
        $this->log(self::LEVEL_WARNING, $message, $context);
    }

    /**
     * Log an error message
     */
    public function error(string $message, array $context = []): void {
        $this->log(self::LEVEL_ERROR, $message, $context);
    }

    /**
     * Log a critical message
     */
    public function critical(string $message, array $context = []): void {
        $this->log(self::LEVEL_CRITICAL, $message, $context);
    }

    /**
     * Log an exception
     */
    public function exception(Exception $e, string $additionalMessage = ''): void {
        $context = [
            'exception' => get_class($e),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString()
        ];

        $message = $additionalMessage ?: $e->getMessage();
        $this->critical($message, $context);
    }

    /**
     * Core logging method
     */
    private function log(string $level, string $message, array $context = []): void {
        if (!$this->shouldLog($level)) {
            return;
        }

        $timestamp = date('Y-m-d H:i:s');
        $pid = getmypid();
        $memory = round(memory_get_peak_usage(true) / 1024 / 1024, 2) . 'MB';

        $logEntry = sprintf(
            "[%s] [%s] [%d] [%s] %s %s\n",
            $timestamp,
            $level,
            $pid,
            $memory,
            $message,
            $this->formatContext($context)
        );

        // Write to file
        $this->writeToFile($logEntry);

        // Write to console if enabled
        if ($this->consoleOutput) {
            $this->writeToConsole($level, $logEntry);
        }

        // Also send to PHP error log for critical errors
        if ($level === self::LEVEL_CRITICAL || $level === self::LEVEL_ERROR) {
            error_log($logEntry);
        }
    }

    /**
     * Check if message should be logged based on minimum level
     */
    private function shouldLog(string $level): bool {
        $levels = [
            self::LEVEL_DEBUG => 1,
            self::LEVEL_INFO => 2,
            self::LEVEL_WARNING => 3,
            self::LEVEL_ERROR => 4,
            self::LEVEL_CRITICAL => 5
        ];

        return ($levels[$level] ?? 0) >= ($levels[$this->minLevel] ?? 0);
    }

    /**
     * Format context array for logging
     */
    private function formatContext(array $context): string {
        if (empty($context)) {
            return '';
        }

        $formatted = [];
        foreach ($context as $key => $value) {
            if (is_array($value) || is_object($value)) {
                $value = json_encode($value, JSON_UNESCAPED_SLASHES);
            }
            $formatted[] = "$key=$value";
        }

        return '[' . implode(' ', $formatted) . ']';
    }

    /**
     * Write log entry to file
     */
    private function writeToFile(string $logEntry): void {
        try {
            file_put_contents($this->logPath, $logEntry, FILE_APPEND | LOCK_EX);
        } catch (Exception $e) {
            // If we can't write to file, at least try console
            if ($this->consoleOutput) {
                error_log("Failed to write to log file: " . $e->getMessage());
            }
        }
    }

    /**
     * Write to console with color coding
     */
    private function writeToConsole(string $level, string $logEntry): void {
        $color = match($level) {
            self::LEVEL_DEBUG => "\033[36m",    // Cyan
            self::LEVEL_INFO => "\033[32m",     // Green
            self::LEVEL_WARNING => "\033[33m",  // Yellow
            self::LEVEL_ERROR => "\033[31m",    // Red
            self::LEVEL_CRITICAL => "\033[35m", // Magenta
            default => "\033[0m"
        };

        $reset = "\033[0m";
        echo $color . rtrim($logEntry) . $reset . PHP_EOL;
    }

    /**
     * Get log file path
     */
    public function getLogPath(): string {
        return $this->logPath;
    }

    /**
     * Rotate log file if it gets too large
     */
    public function rotateLog(int $maxSizeMB = 10): void {
        if (!file_exists($this->logPath)) {
            return;
        }

        $size = filesize($this->logPath);
        $maxSize = $maxSizeMB * 1024 * 1024;

        if ($size > $maxSize) {
            $backupPath = $this->logPath . '.' . date('Y-m-d-H-i-s') . '.bak';
            rename($this->logPath, $backupPath);

            $this->info("Log rotated", [
                'original_size' => round($size / 1024 / 1024, 2) . 'MB',
                'backup_path' => $backupPath
            ]);
        }
    }

    /**
     * Get recent log entries
     */
    public function getRecentEntries(int $lines = 50): array {
        if (!file_exists($this->logPath)) {
            return [];
        }

        $content = file_get_contents($this->logPath);
        $entries = explode("\n", trim($content));

        return array_slice($entries, -$lines);
    }
}
