<?php

/**
 * Chat Service
 * Handles AI chat business logic and API communication
 * Extracted from openai_chat.php for better separation of concerns
 */

class ChatService {

    private string $apiKey;
    private string $apiUrl = 'https://api.x.ai/v1/chat/completions';
    private string $model = 'grok-4-1-fast-reasoning';

    public function __construct() {
        $this->apiKey = $this->getApiKey();
        if (!$this->apiKey) {
            throw new RuntimeException('AI service not configured');
        }
    }

    /**
     * Get AI response for chat message
     */
    public function getAIResponse(
        string $message,
        array $history,
        array $preferences,
        array $surveyInsights,
        ?string $ragContext,
        array $surveyData,
        array $offerSuggestions,
        array $userSession
    ): string {
        $prompt = $this->buildChatPrompt(
            $message,
            $history,
            $preferences,
            $surveyInsights,
            $ragContext,
            $surveyData,
            $offerSuggestions,
            $userSession
        );

        return $this->callAIAPI($prompt, $message);
    }

    /**
     * Build the complete chat prompt
     */
    private function buildChatPrompt(
        string $message,
        array $history,
        array $preferences,
        array $surveyInsights,
        ?string $ragContext,
        array $surveyData,
        array $offerSuggestions,
        array $userSession
    ): string {
        $userName = $surveyData['name'] ?? 'User';
        $tier = $this->getUserAccessTier($userSession);
        $isFreeTier = ($tier === 'free');

        $prompt = $this->buildSystemPrompt($userName, $isFreeTier);

        // Add survey insights
        if (!empty($surveyInsights)) {
            $prompt .= "\n=== USER SURVEY INSIGHTS ===\n";
            foreach ($surveyInsights as $insight) {
                $prompt .= "- $insight\n";
            }
            $prompt .= "\n";
        }

        // Add RAG context
        if ($ragContext) {
            $prompt .= "=== KNOWLEDGE BASE CONTEXT ===\n";
            $prompt .= "$ragContext\n\n";
            $prompt .= "Use this context to provide accurate information.\n\n";
        }

        // Add offer suggestions
        if (!empty($offerSuggestions)) {
            $prompt .= $this->buildOfferSection($offerSuggestions[0], $isFreeTier);
        } else {
            $prompt .= "=== NO OFFERS AVAILABLE ===\n";
            $prompt .= "Direct user to premium upgrade: https://deepleafai.com/billing\n\n";
        }

        // Add conversation history
        if (!empty($history)) {
            $prompt .= $this->buildHistorySection($history);
        }

        return $prompt;
    }

    /**
     * Build the system prompt
     */
    private function buildSystemPrompt(string $userName, bool $isFreeTier): string {
        $prompt = "You are DeepLeaf's Survey Conversion Agent. Your job:\n"
            . "1) Answer questions accurately using provided context.\n"
            . "2) Personalize every reply with the user's survey insights.\n"
            . "3) Drive revenue by guiding them to recommended monetized offers.\n\n"
            . "Response rules:\n"
            . "- 3 sentences maximum.\n"
            . "- Sentence 1: precise answer with facts.\n"
            . "- Sentence 2: tie answer to user's profile from '$userName'.\n";

        if ($isFreeTier) {
            $prompt .= "- Sentence 3: TEASER MODE - Give summary, require offer completion for full details.\n";
        } else {
            $prompt .= "- Sentence 3: Include natural offer CTA if available.\n";
        }

        $prompt .= "- Tone: confident, actionable, conversion-focused.\n"
            . "- Never invent offers; only mention provided offers.\n\n";

        return $prompt;
    }

    /**
     * Build offer section
     */
    private function buildOfferSection(array $offer, bool $isFreeTier): string {
        $section = "=== RECOMMENDED OFFER ===\n";
        $section .= "Title: {$offer['title']}\n";
        $section .= "CTA: {$offer['cta']}\n";

        if (isset($offer['points'])) {
            $section .= "Bonus: {$offer['points']} sweepstakes entries\n";
        }

        if ($isFreeTier) {
            $section .= "Strategy: Position as required 'key' to unlock full response.\n\n";
        } else {
            $section .= "Strategy: Present as valuable enhancement opportunity.\n\n";
        }

        return $section;
    }

    /**
     * Build conversation history section
     */
    private function buildHistorySection(array $history): string {
        $section = "=== CONVERSATION HISTORY ===\n";
        $recentHistory = array_slice($history, -3);

        foreach ($recentHistory as $msg) {
            if (isset($msg['user_message'])) {
                $section .= 'USER: ' . $msg['user_message'] . "\n";
                $section .= 'AI: ' . $msg['ai_response'] . "\n";
            } else {
                $role = ($msg['sender'] ?? 'user') === 'user' ? 'USER' : 'AI';
                $section .= $role . ': ' . ($msg['text'] ?? $msg['content'] ?? '') . "\n";
            }
        }

        $section .= "\n";
        return $section;
    }

    /**
     * Call the AI API
     */
    private function callAIAPI(string $systemPrompt, string $userMessage): string {
        $payload = [
            "messages" => [
                ["role" => "system", "content" => $systemPrompt],
                ["role" => "user", "content" => $userMessage]
            ],
            "model" => $this->model,
            "stream" => false,
            "temperature" => 0.7
        ];

        $ch = curl_init($this->apiUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $this->apiKey
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($httpCode !== 200) {
            throw new RuntimeException("AI service error: HTTP $httpCode");
        }

        $data = json_decode($response, true);
        return $data['choices'][0]['message']['content'] ?? 'I apologize, but I encountered an error processing your request.';
    }

    /**
     * Get API key from environment
     */
    private function getApiKey(): ?string {
        // Try Grok API key first (since we're using Grok), then OpenAI as fallback
        $apiKey = $_ENV['GROK_API_KEY'] ?? $_ENV['OPENAI_API_KEY'] ?? null;

        if (!$apiKey) {
            // Fallback to config files (legacy support)
            $configFiles = [
                APP_ROOT . '/.env',
                APP_ROOT . '/config.live.env',
                APP_ROOT . '/config.env'
            ];

            foreach ($configFiles as $configFile) {
                if (file_exists($configFile)) {
                    $content = file_get_contents($configFile);
                    if (preg_match('/(?:OPENAI_API_KEY|GROK_API_KEY)=(.+)/', $content, $matches)) {
                        $apiKey = trim($matches[1], '"\'');
                        break;
                    }
                }
            }
        }

        return $apiKey;
    }

    /**
     * Get user access tier
     */
    private function getUserAccessTier(array $userSession): string {
        $tokenCount = $userSession['token_count'] ?? 0;
        $completedOffers = count($userSession['completed_offers'] ?? []);

        if ($completedOffers >= 3) {
            return 'premium';
        } elseif ($completedOffers >= 1) {
            return 'engaged';
        } else {
            return 'free';
        }
    }
}
