<?php

/**
 * Simple HTTP Router
 * Handles route registration and dispatching
 */

class Router {

    private array $routes = [];
    private Container $container;

    public function __construct(Container $container) {
        $this->container = $container;
    }

    /**
     * Register a GET route
     */
    public function get(string $path, callable|array $handler): void {
        $this->addRoute('GET', $path, $handler);
    }

    /**
     * Register a POST route
     */
    public function post(string $path, callable|array $handler): void {
        $this->addRoute('POST', $path, $handler);
    }

    /**
     * Register a PUT route
     */
    public function put(string $path, callable|array $handler): void {
        $this->addRoute('PUT', $path, $handler);
    }

    /**
     * Register a DELETE route
     */
    public function delete(string $path, callable|array $handler): void {
        $this->addRoute('DELETE', $path, $handler);
    }

    /**
     * Add a route to the routing table
     */
    private function addRoute(string $method, string $path, callable|array $handler): void {
        $this->routes[$method][$path] = $handler;
    }

    /**
     * Dispatch the current request
     */
    public function dispatch(): void {
        $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
        $uri = parse_url($_SERVER['REQUEST_URI'] ?? '', PHP_URL_PATH);

        // Remove query string if present
        if (($pos = strpos($uri, '?')) !== false) {
            $uri = substr($uri, 0, $pos);
        }

        // Remove trailing slash
        $uri = rtrim($uri, '/');

        // Handle root path
        if ($uri === '') {
            $uri = '/';
        }

        // Check for exact route match
        if (isset($this->routes[$method][$uri])) {
            $this->callHandler($this->routes[$method][$uri]);
            return;
        }

        // Check for wildcard routes (for legacy compatibility)
        foreach ($this->routes[$method] as $route => $handler) {
            if ($this->matchesRoute($route, $uri)) {
                $this->callHandler($handler);
                return;
            }
        }

        // No route found
        $this->handleNotFound();
    }

    /**
     * Check if a route matches the current URI
     */
    private function matchesRoute(string $route, string $uri): bool {
        // Simple wildcard matching for legacy routes
        if (strpos($route, '*.php') !== false) {
            return strpos($uri, '.php') !== false;
        }

        return false;
    }

    /**
     * Call the route handler
     */
    private function callHandler(callable|array $handler): void {
        if (is_array($handler)) {
            [$className, $method] = $handler;
            $instance = $this->container->resolve($className);
            $instance->$method();
        } elseif (is_callable($handler)) {
            $handler();
        } else {
            throw new RuntimeException("Invalid route handler");
        }
    }

    /**
     * Handle 404 - route not found
     */
    private function handleNotFound(): void {
        http_response_code(404);

        if ($this->isApiRequest()) {
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'error' => 'Route not found'
            ]);
        } else {
            echo "<h1>404 - Page Not Found</h1>";
        }
    }

    /**
     * Check if this is an API request
     */
    private function isApiRequest(): bool {
        $uri = $_SERVER['REQUEST_URI'] ?? '';
        return strpos($uri, '/api/') === 0;
    }

    /**
     * Get all registered routes (for debugging)
     */
    public function getRoutes(): array {
        return $this->routes;
    }
}
