<?php

/**
 * Global Error Handler
 * Comprehensive error handling and logging system
 */

class ErrorHandler {

    private Logger $logger;
    private bool $displayErrors;

    public function __construct(Logger $logger, bool $displayErrors = false) {
        $this->logger = $logger;
        $this->displayErrors = $displayErrors;
    }

    /**
     * Register all error handlers
     */
    public function register(): void {
        // Set error reporting level
        error_reporting(E_ALL);

        // Register error handler
        set_error_handler([$this, 'handleError']);

        // Register exception handler
        set_exception_handler([$this, 'handleException']);

        // Register shutdown handler for fatal errors
        register_shutdown_function([$this, 'handleShutdown']);

        // Set PHP ini settings
        ini_set('display_errors', $this->displayErrors ? '1' : '0');
        ini_set('log_errors', '1');
        ini_set('error_log', $this->logger->getLogPath());
    }

    /**
     * Handle PHP errors
     */
    public function handleError(
        int $errno,
        string $errstr,
        string $errfile = '',
        int $errline = 0,
        array $errcontext = []
    ): bool {
        // Convert error level to string
        $errorLevel = match($errno) {
            E_ERROR, E_CORE_ERROR, E_COMPILE_ERROR, E_USER_ERROR => Logger::LEVEL_CRITICAL,
            E_WARNING, E_CORE_WARNING, E_COMPILE_WARNING, E_USER_WARNING => Logger::LEVEL_WARNING,
            E_NOTICE, E_USER_NOTICE, E_STRICT, E_DEPRECATED, E_USER_DEPRECATED => Logger::LEVEL_INFO,
            default => Logger::LEVEL_ERROR
        };

        $message = sprintf(
            'PHP %s: %s in %s on line %d',
            $this->getErrorType($errno),
            $errstr,
            $errfile,
            $errline
        );

        $context = [
            'error_code' => $errno,
            'file' => $errfile,
            'line' => $errline,
            'trace' => debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS)
        ];

        $this->logger->log($errorLevel, $message, $context);

        // Don't execute PHP's internal error handler
        return true;
    }

    /**
     * Handle uncaught exceptions
     */
    public function handleException(Throwable $exception): void {
        $this->logger->exception($exception, 'Uncaught exception');

        if ($this->displayErrors) {
            $this->renderExceptionPage($exception);
        } else {
            $this->renderGenericErrorPage();
        }
    }

    /**
     * Handle shutdown for fatal errors
     */
    public function handleShutdown(): void {
        $error = error_get_last();

        if ($error && in_array($error['type'], [E_ERROR, E_CORE_ERROR, E_COMPILE_ERROR, E_USER_ERROR])) {
            $this->logger->critical('Fatal error occurred', [
                'type' => $error['type'],
                'message' => $error['message'],
                'file' => $error['file'],
                'line' => $error['line']
            ]);

            if ($this->displayErrors) {
                $this->renderFatalErrorPage($error);
            } else {
                $this->renderGenericErrorPage();
            }
        }
    }

    /**
     * Render exception page for debugging
     */
    private function renderExceptionPage(Throwable $exception): void {
        if ($this->isApiRequest()) {
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'error' => 'Internal server error',
                'message' => $exception->getMessage(),
                'debug' => [
                    'file' => $exception->getFile(),
                    'line' => $exception->getLine(),
                    'trace' => $exception->getTraceAsString()
                ]
            ], JSON_PRETTY_PRINT);
        } else {
            http_response_code(500);
            echo $this->getExceptionHtml($exception);
        }
    }

    /**
     * Render generic error page
     */
    private function renderGenericErrorPage(): void {
        if ($this->isApiRequest()) {
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'error' => 'Internal server error',
                'message' => 'Something went wrong. Please try again later.'
            ]);
        } else {
            http_response_code(500);
            echo $this->getGenericErrorHtml();
        }
    }

    /**
     * Render fatal error page
     */
    private function renderFatalErrorPage(array $error): void {
        if ($this->isApiRequest()) {
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'error' => 'Fatal error',
                'message' => $error['message']
            ], JSON_PRETTY_PRINT);
        } else {
            http_response_code(500);
            echo $this->getFatalErrorHtml($error);
        }
    }

    /**
     * Check if this is an API request
     */
    private function isApiRequest(): bool {
        return isset($_SERVER['REQUEST_URI']) &&
               strpos($_SERVER['REQUEST_URI'], '/api/') === 0;
    }

    /**
     * Get error type string
     */
    private function getErrorType(int $errno): string {
        return match($errno) {
            E_ERROR => 'E_ERROR',
            E_WARNING => 'E_WARNING',
            E_NOTICE => 'E_NOTICE',
            E_USER_ERROR => 'E_USER_ERROR',
            E_USER_WARNING => 'E_USER_WARNING',
            E_USER_NOTICE => 'E_USER_NOTICE',
            E_STRICT => 'E_STRICT',
            E_RECOVERABLE_ERROR => 'E_RECOVERABLE_ERROR',
            E_DEPRECATED => 'E_DEPRECATED',
            E_USER_DEPRECATED => 'E_USER_DEPRECATED',
            E_CORE_ERROR => 'E_CORE_ERROR',
            E_CORE_WARNING => 'E_CORE_WARNING',
            E_COMPILE_ERROR => 'E_COMPILE_ERROR',
            E_COMPILE_WARNING => 'E_COMPILE_WARNING',
            E_PARSE => 'E_PARSE',
            default => 'UNKNOWN_ERROR'
        };
    }

    /**
     * Get HTML for exception display
     */
    private function getExceptionHtml(Throwable $exception): string {
        return <<<HTML
<!DOCTYPE html>
<html>
<head>
    <title>Application Error</title>
    <style>
        body { font-family: monospace; margin: 20px; background: #f5f5f5; }
        .error { background: #ffebee; border: 1px solid #f44336; padding: 15px; border-radius: 4px; }
        .trace { background: #fff; border: 1px solid #ddd; padding: 10px; margin-top: 10px; overflow-x: auto; }
        h1 { color: #d32f2f; }
        h2 { color: #1976d2; margin-top: 20px; }
    </style>
</head>
<body>
    <div class="error">
        <h1>🚨 Application Error</h1>
        <p><strong>Type:</strong> {$exception::class}</p>
        <p><strong>Message:</strong> {$exception->getMessage()}</p>
        <p><strong>File:</strong> {$exception->getFile()}</p>
        <p><strong>Line:</strong> {$exception->getLine()}</p>
    </div>

    <h2>Stack Trace</h2>
    <div class="trace">
        <pre>{$exception->getTraceAsString()}</pre>
    </div>
</body>
</html>
HTML;
    }

    /**
     * Get generic error HTML
     */
    private function getGenericErrorHtml(): string {
        return <<<HTML
<!DOCTYPE html>
<html>
<head>
    <title>Service Unavailable</title>
    <style>
        body { font-family: Arial, sans-serif; text-align: center; padding: 50px; background: #f5f5f5; }
        .container { max-width: 600px; margin: 0 auto; background: white; padding: 40px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        h1 { color: #d32f2f; margin-bottom: 20px; }
        p { color: #666; line-height: 1.6; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🚧 Service Temporarily Unavailable</h1>
        <p>We're experiencing some technical difficulties. Our team has been notified and is working to resolve the issue.</p>
        <p>Please try again in a few minutes.</p>
    </div>
</body>
</html>
HTML;
    }

    /**
     * Get fatal error HTML
     */
    private function getFatalErrorHtml(array $error): string {
        return <<<HTML
<!DOCTYPE html>
<html>
<head>
    <title>Fatal Error</title>
    <style>
        body { font-family: monospace; margin: 20px; background: #f5f5f5; }
        .error { background: #ffebee; border: 1px solid #f44336; padding: 15px; border-radius: 4px; }
        h1 { color: #d32f2f; }
    </style>
</head>
<body>
    <div class="error">
        <h1>💀 Fatal Error</h1>
        <p><strong>Message:</strong> {$error['message']}</p>
        <p><strong>File:</strong> {$error['file']}</p>
        <p><strong>Line:</strong> {$error['line']}</p>
    </div>
</body>
</html>
HTML;
    }
}
