<?php

/**
 * Health Controller
 * Provides health check endpoints for monitoring
 */

class HealthController {

    /**
     * Basic health check
     */
    public function check(): void {
        $status = $this->getSystemStatus();

        header('Content-Type: application/json');

        if ($status['overall'] === 'healthy') {
            http_response_code(200);
        } else {
            http_response_code(503); // Service Unavailable
        }

        echo json_encode([
            'status' => $status['overall'],
            'timestamp' => date('c'),
            'checks' => $status['checks'],
            'version' => $this->getVersion()
        ]);
    }

    /**
     * Get comprehensive system status
     */
    private function getSystemStatus(): array {
        $checks = [
            'database' => $this->checkDatabase(),
            'file_system' => $this->checkFileSystem(),
            'memory' => $this->checkMemory(),
            'ai_service' => $this->checkAIService(),
            'rag_service' => $this->checkRAGService()
        ];

        $healthy = array_filter($checks, function($check) {
            return $check['status'] === 'healthy';
        });

        return [
            'overall' => count($healthy) === count($checks) ? 'healthy' : 'degraded',
            'checks' => $checks
        ];
    }

    /**
     * Check database connectivity
     */
    private function checkDatabase(): array {
        try {
            // Simple database connection test
            $pdo = new PDO(
                'mysql:host=' . (getenv('DB_HOST') ?: 'localhost') . ';dbname=' . (getenv('DB_NAME') ?: 'test'),
                getenv('DB_USER') ?: 'root',
                getenv('DB_PASS') ?: '',
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );

            $pdo->query('SELECT 1');
            return ['status' => 'healthy', 'message' => 'Database connection OK'];

        } catch (Exception $e) {
            return ['status' => 'unhealthy', 'message' => 'Database connection failed: ' . $e->getMessage()];
        }
    }

    /**
     * Check file system permissions
     */
    private function checkFileSystem(): array {
        $paths = [
            APP_ROOT . '/storage',
            APP_ROOT . '/storage/logs',
            APP_ROOT . '/storage/cache'
        ];

        foreach ($paths as $path) {
            if (!is_writable($path)) {
                return ['status' => 'unhealthy', 'message' => "Path not writable: $path"];
            }
        }

        return ['status' => 'healthy', 'message' => 'File system OK'];
    }

    /**
     * Check memory usage
     */
    private function checkMemory(): array {
        $memoryUsage = memory_get_peak_usage(true);
        $memoryLimit = $this->getMemoryLimit();

        if ($memoryUsage > $memoryLimit * 0.9) {
            return ['status' => 'degraded', 'message' => 'High memory usage'];
        }

        return ['status' => 'healthy', 'message' => 'Memory usage OK'];
    }

    /**
     * Check AI service availability
     */
    private function checkAIService(): array {
        try {
            $apiKey = getenv('OPENAI_API_KEY') ?: getenv('GROK_API_KEY');
            if (!$apiKey) {
                return ['status' => 'unhealthy', 'message' => 'No AI API key configured'];
            }

            // Simple API key validation (basic check)
            if (strlen($apiKey) < 20) {
                return ['status' => 'unhealthy', 'message' => 'Invalid AI API key format'];
            }

            return ['status' => 'healthy', 'message' => 'AI service configured'];

        } catch (Exception $e) {
            return ['status' => 'unhealthy', 'message' => 'AI service check failed: ' . $e->getMessage()];
        }
    }

    /**
     * Check RAG service availability
     */
    private function checkRAGService(): array {
        try {
            $ch = curl_init('http://localhost:5000/health');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 5);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($httpCode === 200) {
                return ['status' => 'healthy', 'message' => 'RAG service available'];
            } else {
                return ['status' => 'degraded', 'message' => 'RAG service returned HTTP ' . $httpCode];
            }

        } catch (Exception $e) {
            return ['status' => 'unhealthy', 'message' => 'RAG service unreachable'];
        }
    }

    /**
     * Get PHP memory limit in bytes
     */
    private function getMemoryLimit(): int {
        $limit = ini_get('memory_limit');
        if (preg_match('/^(\d+)(.)$/', $limit, $matches)) {
            $value = (int) $matches[1];
            $unit = $matches[2];

            switch (strtoupper($unit)) {
                case 'G': return $value * 1024 * 1024 * 1024;
                case 'M': return $value * 1024 * 1024;
                case 'K': return $value * 1024;
                default: return $value;
            }
        }

        return 128 * 1024 * 1024; // Default 128MB
    }

    /**
     * Get application version
     */
    private function getVersion(): string {
        return getenv('APP_VERSION') ?: '2.0.0';
    }
}
