<?php

/**
 * Chat Controller
 * Handles AI chat requests and responses
 * Refactored from the original 1,201-line openai_chat.php
 */

class ChatController {

    private ChatService $chatService;
    private OfferService $offerService;
    private SessionManager $sessionManager;

    public function __construct() {
        $this->chatService = new ChatService();
        $this->offerService = new OfferService();
        $this->sessionManager = new SessionManager();
    }

    /**
     * Handle chat API requests
     */
    public function chat(): void {
        try {
            // Set headers
            header('Content-Type: application/json');
            header('Access-Control-Allow-Origin: *');
            header('Access-Control-Allow-Methods: POST');
            header('Access-Control-Allow-Headers: Content-Type');

            // Get and validate input
            $input = $this->getValidatedInput();

            // Process the chat request
            $response = $this->processChatRequest($input);

            // Send response
            echo json_encode($response);

        } catch (Exception $e) {
            error_log("Chat API Error: " . $e->getMessage());
            $this->sendErrorResponse($e->getMessage());
        }
    }

    /**
     * Get and validate input data
     */
    private function getValidatedInput(): array {
        $input = json_decode(file_get_contents('php://input'), true);

        if (!$input || !isset($input['message'])) {
            throw new InvalidArgumentException('Message is required');
        }

        return [
            'message' => trim($input['message']),
            'history' => $input['history'] ?? [],
            'preferences' => $input['preferences'] ?? [],
            'survey' => $input['survey'] ?? []
        ];
    }

    /**
     * Process the chat request through the service layer
     */
    private function processChatRequest(array $input): array {
        // Get user session and profile
        $userId = $this->sessionManager->getUserSessionId($input['survey']);
        $userSession = $this->sessionManager->getUserSession($userId, $input['survey']);

        // Merge conversation histories
        $serverHistory = $this->sessionManager->getConversationHistory($userId, 5);
        $combinedHistory = array_merge($serverHistory, $input['history']);
        $combinedHistory = array_slice($combinedHistory, -8); // Limit history

        // Analyze for offers
        $offerSuggestions = $this->offerService->analyzeForOffers(
            $input['message'],
            $input['preferences'],
            $combinedHistory,
            $input['survey'],
            $userSession
        );

        // Get RAG context
        $ragContext = $this->getRAGContext($input['message'], $combinedHistory, $input['preferences']);

        // Extract survey insights
        $surveyInsights = $this->extractSurveyInsights($input['survey']);

        // Build and send prompt to AI
        $aiResponse = $this->chatService->getAIResponse(
            $input['message'],
            $combinedHistory,
            $input['preferences'],
            $surveyInsights,
            $ragContext,
            $input['survey'],
            $offerSuggestions,
            $userSession
        );

        // Store conversation
        $this->sessionManager->addConversationTurn($userId, $input['message'], $aiResponse, $offerSuggestions);

        // Record offer analytics
        if (!empty($offerSuggestions)) {
            $this->recordOfferAnalytics($userId, $offerSuggestions);
        }

        return [
            'success' => true,
            'response' => $aiResponse,
            'suggestions' => [],
            'offers' => $offerSuggestions,
            'user_id' => $userId,
            'rag_used' => !empty($ragContext)
        ];
    }

    /**
     * Get RAG context from the RAG service
     */
    private function getRAGContext(string $message, array $history, array $preferences): ?string {
        try {
            $ragService = new RAGService();
            $result = $ragService->query($message, $history, $preferences);
            return $result['context'] ?? null;
        } catch (Exception $e) {
            error_log("RAG context error: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Extract survey insights
     */
    private function extractSurveyInsights(array $survey): array {
        if (empty($survey['answers'])) {
            return [];
        }

        $insights = [];
        $answers = $survey['answers'];

        // Business owner insights
        if (isset($answers['10']) && strpos(strtolower($answers['10']), 'own') !== false) {
            $insights[] = "BUSINESS OWNER: Prime candidate for B2B offers and scaling opportunities.";
        }

        // Financial insights
        if (isset($answers['04'])) {
            $financial = strtolower($answers['04']);
            if (strpos($financial, 'debt') !== false) {
                $insights[] = "DEBT CONCERNS: Interested in relief solutions and income opportunities.";
            }
        }

        return $insights;
    }

    /**
     * Record offer analytics for monetization tracking
     */
    private function recordOfferAnalytics(string $userId, array $offers): void {
        try {
            foreach ($offers as $offer) {
                // Record offer impressions
                error_log("Offer shown to user $userId: " . ($offer['title'] ?? 'Unknown'));
            }
        } catch (Exception $e) {
            error_log("Analytics error: " . $e->getMessage());
        }
    }

    /**
     * Send error response
     */
    private function sendErrorResponse(string $message): void {
        echo json_encode([
            'success' => false,
            'error' => $message,
            'response' => "I'm currently experiencing some technical difficulties. Please try again later.",
            'suggestions' => []
        ]);
    }
}
