<?php

/**
 * Dependency Injection Container
 * Simple implementation for managing service dependencies
 */

class Container {

    private array $bindings = [];
    private array $instances = [];

    /**
     * Bind a service to the container
     */
    public function bind(string $abstract, callable $concrete): void {
        $this->bindings[$abstract] = $concrete;
    }

    /**
     * Resolve a service from the container
     */
    public function resolve(string $abstract): mixed {
        if (isset($this->instances[$abstract])) {
            return $this->instances[$abstract];
        }

        if (isset($this->bindings[$abstract])) {
            $concrete = $this->bindings[$abstract];
            $instance = $concrete($this);

            if (is_object($instance)) {
                $this->instances[$abstract] = $instance;
            }

            return $instance;
        }

        // Try to instantiate directly
        if (class_exists($abstract)) {
            return $this->instances[$abstract] = new $abstract();
        }

        throw new RuntimeException("Cannot resolve service: $abstract");
    }

    /**
     * Get a service instance
     */
    public function get(string $abstract): mixed {
        return $this->resolve($abstract);
    }

    /**
     * Check if a service is bound
     */
    public function has(string $abstract): bool {
        return isset($this->bindings[$abstract]) || class_exists($abstract);
    }

    /**
     * Register a singleton instance
     */
    public function singleton(string $abstract, mixed $instance): void {
        $this->instances[$abstract] = $instance;
    }
}
