<?php

/**
 * Main Application Class
 * Handles routing and application bootstrap
 * Implements clean architecture pattern
 */

class Application {

    private Router $router;
    private Container $container;

    public function __construct() {
        $this->container = new Container();
        $this->router = new Router($this->container);

        // Initialize error handling first
        $this->initializeErrorHandling();

        $this->registerRoutes();
        $this->registerServices();
    }

    /**
     * Run the application
     */
    public function run(): void {
        try {
            $this->router->dispatch();
        } catch (Throwable $e) {
            $this->handleException($e);
        }
    }

    /**
     * Initialize error handling system
     */
    private function initializeErrorHandling(): void {
        $logger = new Logger();
        $errorHandler = new ErrorHandler($logger, defined('APP_DEBUG') && APP_DEBUG);

        $errorHandler->register();

        // Store logger in container for other services
        $this->container->singleton(Logger::class, $logger);
    }

    /**
     * Register application routes
     */
    private function registerRoutes(): void {
        // Web routes
        $this->router->get('/', [SurveyController::class, 'index']);
        $this->router->get('/survey', [SurveyController::class, 'index']);

        // API routes
        $this->router->post('/api/chat', [ChatController::class, 'chat']);
        $this->router->get('/api/health', [HealthController::class, 'check']);

        // Legacy routes (for backward compatibility)
        $this->router->post('/api/openai_chat.php', [ChatController::class, 'chat']);
    }

    /**
     * Register services in the container
     */
    private function registerServices(): void {
        $this->container->bind(ChatService::class, function() {
            return new ChatService();
        });

        $this->container->bind(OfferService::class, function() {
            return new OfferService();
        });

        $this->container->bind(RAGService::class, function() {
            return new RAGService();
        });

        $this->container->bind(SessionManager::class, function() {
            return new SessionManager();
        });

        // Controllers
        $this->container->bind(SurveyController::class, function($container) {
            return new SurveyController();
        });

        $this->container->bind(ChatController::class, function($container) {
            return new ChatController();
        });
    }

    /**
     * Handle application exceptions
     */
    private function handleException(Throwable $e): void {
        // Get logger from container
        $logger = $this->container->get(Logger::class) ?? new Logger();
        $logger->exception($e, 'Application exception');

        // The ErrorHandler will handle the response rendering
        // This is just a fallback
        if (!headers_sent()) {
            $isApiRequest = $this->isApiRequest();

            if ($isApiRequest) {
                header('Content-Type: application/json');
                http_response_code(500);
                echo json_encode([
                    'success' => false,
                    'error' => 'Internal server error',
                    'message' => defined('APP_DEBUG') && APP_DEBUG ? $e->getMessage() : 'Something went wrong'
                ]);
            } else {
                http_response_code(500);
                if (defined('APP_DEBUG') && APP_DEBUG) {
                    echo "<h1>Error</h1><pre>{$e->getMessage()}</pre><pre>{$e->getTraceAsString()}</pre>";
                } else {
                    echo "<h1>Internal Server Error</h1><p>Something went wrong. Please try again later.</p>";
                }
            }
        }
    }

    /**
     * Check if this is an API request
     */
    private function isApiRequest(): bool {
        $uri = $_SERVER['REQUEST_URI'] ?? '';
        return strpos($uri, '/api/') === 0;
    }
}
