# Docker Deployment Guide

## Overview

This guide covers deploying DeepLeaf AI using Docker and Docker Compose for production environments.

## Prerequisites

- Docker Engine 20.10+
- Docker Compose 2.0+
- At least 4GB RAM available
- 20GB free disk space
- Domain name (optional but recommended)

## Quick Start

1. **Clone and setup environment:**
   ```bash
   git clone <repository-url> deepleaf-ai
   cd deepleaf-ai

   # Copy environment template
   cp env.docker.example .env

   # Edit with your actual values
   nano .env
   ```

2. **Start the application:**
   ```bash
   # Build and start all services
   docker-compose up -d

   # View logs
   docker-compose logs -f app
   ```

3. **Verify deployment:**
   ```bash
   # Check health endpoint
   curl http://localhost:8080/api/health

   # Access application
   open http://localhost:8080
   ```

## Environment Configuration

### Required Environment Variables

Edit your `.env` file with actual values:

```bash
# Database
MYSQL_ROOT_PASSWORD=your_secure_root_password
MYSQL_PASSWORD=your_secure_app_password

# AI Services
OPENAI_API_KEY=sk-your-actual-openai-key
GROK_API_KEY=your-actual-grok-key

# Application
APP_URL=https://your-domain.com
APP_KEY=your-generated-app-key
```

### Optional Environment Variables

```bash
# Custom ports
APP_PORT=8080
MYSQL_PORT=3306

# Debug mode (development only)
APP_DEBUG=false

# Redis (if using external Redis)
REDIS_HOST=redis
REDIS_PORT=6379
```

## Production Deployment

### 1. Domain Configuration

Update your DNS to point to your server, then configure SSL:

```bash
# Install certbot for SSL
sudo apt install certbot

# Get SSL certificate
sudo certbot certonly --standalone -d your-domain.com

# Update nginx configuration for SSL
```

### 2. Production Optimizations

```bash
# Use production Docker Compose override
docker-compose -f docker-compose.yml -f docker-compose.prod.yml up -d

# Enable log rotation
docker-compose exec app logrotate /etc/logrotate.d/app

# Setup automated backups
0 2 * * * /path/to/scripts/deploy/deploy.sh production backup
```

### 3. Monitoring Setup

Access monitoring dashboards:
- **Grafana**: http://localhost:3000 (admin/admin_password)
- **Prometheus**: http://localhost:9090

### 4. Scaling

```bash
# Scale application instances
docker-compose up -d --scale app=3

# Scale database read replicas (advanced)
docker-compose up -d --scale db-replica=2
```

## Troubleshooting

### Common Issues

**Application won't start:**
```bash
# Check logs
docker-compose logs app

# Check if ports are available
netstat -tlnp | grep :8080
```

**Database connection failed:**
```bash
# Check database logs
docker-compose logs db

# Verify environment variables
docker-compose exec app env | grep DB_
```

**AI services not working:**
```bash
# Verify API keys
docker-compose exec app env | grep API_KEY

# Test API connectivity
docker-compose exec app curl -H "Authorization: Bearer $OPENAI_API_KEY" https://api.openai.com/v1/models
```

### Health Checks

```bash
# Manual health check
curl -f http://localhost:8080/api/health

# Automated health monitoring
./scripts/monitoring/health-check.sh --watch
```

## Backup and Recovery

### Automated Backups

```bash
# Run backup
./scripts/deploy/deploy.sh production backup

# List available backups
ls -la /opt/backups/deepleaf-ai/
```

### Manual Restore

```bash
# Stop application
docker-compose down

# Restore from backup
./scripts/deploy/deploy.sh production rollback 20231201_143000

# Restart application
docker-compose up -d
```

## Security Considerations

### Production Hardening

1. **Change default passwords** in `.env`
2. **Use strong database passwords**
3. **Enable SSL/TLS** for all connections
4. **Configure firewall** rules
5. **Regular security updates**:
   ```bash
   docker-compose pull
   docker-compose up -d
   ```

### Network Security

```bash
# Don't expose database ports in production
# Remove or comment out MYSQL_PORT in docker-compose.yml

# Use internal networks only
docker network ls
```

## Performance Optimization

### Resource Allocation

```yaml
# docker-compose.prod.yml
services:
  app:
    deploy:
      resources:
        limits:
          cpus: '2.0'
          memory: 2G
        reservations:
          cpus: '0.5'
          memory: 512M
```

### Database Optimization

```bash
# MySQL performance tuning
docker-compose exec db mysql -u root -p

# Inside MySQL:
SET GLOBAL innodb_buffer_pool_size = 1073741824; -- 1GB
SET GLOBAL max_connections = 200;
```

## Monitoring and Alerts

### Setting up Alerts

1. **Slack Integration:**
   ```bash
   export SLACK_WEBHOOK=https://hooks.slack.com/services/...
   ./scripts/monitoring/health-check.sh
   ```

2. **Email Alerts:**
   ```bash
   export EMAIL_ALERTS=admin@yourcompany.com
   ./scripts/monitoring/health-check.sh
   ```

### Log Management

```bash
# View application logs
docker-compose logs -f app

# View all service logs
docker-compose logs

# Export logs for analysis
docker-compose logs app > app_logs_$(date +%Y%m%d).log
```

## Updating the Application

### Rolling Updates

```bash
# Pull latest changes
git pull origin main

# Build and deploy without downtime
docker-compose up -d --build

# Verify deployment
curl http://localhost:8080/api/health
```

### Blue-Green Deployment

```bash
# Create blue environment
docker-compose -p blue up -d

# Test blue environment
curl http://localhost:8081/api/health

# Switch traffic (update load balancer)
# ...

# Remove old environment
docker-compose -p green down
```

## Support

For issues and questions:
- Check logs: `docker-compose logs`
- Health check: `curl http://localhost:8080/api/health`
- Documentation: `/docs/api/index.html`
- Support: support@deepleafai.com
