# LeadGen AI Chat System

A sophisticated AI-powered survey chat system with RAG (Retrieval-Augmented Generation) integration, powered by Grok Fast Reasoning Model.

## 🚀 System Overview

This system combines:
- **Survey-based personalization** - Collects user data through surveys
- **RAG knowledge base** - Admin-uploaded documents for custom knowledge
- **Grok Fast Reasoning** - Advanced AI responses using `grok-4-fast-reasoning`
- **PHP Backend** - Robust API for chat interactions
- **Python RAG System** - Document processing and retrieval

## 📁 Project Structure

```
leadgen/
├── api/
│   ├── openai_chat.php      # Main chat API endpoint
│   └── rag_bridge.php        # Bridge to Python RAG system
├── airagagent/
│   ├── app.py                # Flask RAG API server
│   ├── main.py               # RAG system core
│   ├── grok_api.py           # Grok API integration
│   ├── pdf_directory/        # Upload PDFs here for RAG
│   └── processed/            # Processed documents (auto-generated)
├── config/
│   └── database.php          # Database configuration
└── tests/
    └── ai-chat-flow.spec.js  # E2E tests
```

## 🛠️ Setup Instructions

### Prerequisites

- PHP 7.4+
- Python 3.12+
- MySQL/MariaDB
- Composer (for PHP dependencies)
- Flask, sentence-transformers, faiss-cpu (for RAG)

### Installation

1. **Clone/Download the project**
   ```bash
   cd /var/www/html/leadgen
   ```

2. **Install PHP dependencies** (if any)
   ```bash
   composer install
   ```

3. **Install Python RAG dependencies**
   ```bash
   cd airagagent
   python3 -m pip install flask requests langchain-text-splitters sentence-transformers faiss-cpu numpy rapidfuzz --break-system-packages
   ```

4. **Configure database**
   - Update `config/database.php` with your database credentials
   - Ensure database tables exist (tbl_survey, tbl_survey_answers, etc.)

5. **Start RAG Flask server**
   ```bash
   cd airagagent
   nohup python3 app.py > /tmp/rag_app.log 2>&1 &
   ```
   The server runs on `http://localhost:5000`

6. **Configure web server**
   - Point Apache/Nginx to `/var/www/html/leadgen`
   - Ensure PHP is enabled
   - Set proper file permissions

## 📚 Adding Books/Documents to RAG

To add knowledge to the chat system:

1. **Upload PDF files**
   ```bash
   # Copy PDFs to the pdf_directory
   cp your_book.pdf /var/www/html/leadgen/airagagent/pdf_directory/
   ```

2. **Process documents**
   - The RAG system automatically processes new PDFs
   - Or trigger manually via: `curl http://localhost:5000/status`

3. **Verify processing**
   ```bash
   curl http://localhost:5000/status
   # Check "total_documents" count increased
   ```

## 🔧 Running the System

### Start RAG Flask Server

```bash
cd /var/www/html/leadgen/airagagent
python3 app.py
```

Or run in background:
```bash
nohup python3 app.py > /tmp/rag_app.log 2>&1 &
```

### Check RAG System Status

```bash
curl http://localhost:5000/status
```

### Test Chat API

```bash
curl -X POST https://yourdomain.com/api/openai_chat.php \
  -H "Content-Type: application/json" \
  -d '{
    "message": "Hello",
    "survey": {
      "name": "Test User",
      "answers": {"10": "Yes, I own a business"}
    }
  }'
```

## 🔑 API Configuration

### Grok API Key

The Grok API key is configured in:
- `api/openai_chat.php` (line 798)
- `airagagent/grok_api.py` (line 12)

Current model: `grok-4-fast-reasoning`

### RAG System Configuration

- RAG Flask server: `http://localhost:5000`
- Vector store: `airagagent/processed/embeddings/`
- Documents: `airagagent/pdf_directory/`

## 📊 System Status Checks

### Check if RAG Flask is running
```bash
ps aux | grep "python3 app.py" | grep -v grep
```

### Check RAG documents
```bash
curl http://localhost:5000/status | python3 -m json.tool
```

### Test full integration
```bash
curl -X POST https://yourdomain.com/api/openai_chat.php \
  -H "Content-Type: application/json" \
  -d '{"message":"What is cannabis?","survey":{"name":"Test","answers":{}}}'
```

## 🧪 Testing

Run E2E tests:
```bash
cd tests
npm test ai-chat-flow.spec.js
```

## 🔄 Maintenance

### Restart RAG Flask Server
```bash
pkill -f "python3 app.py"
cd /var/www/html/leadgen/airagagent
nohup python3 app.py > /tmp/rag_app.log 2>&1 &
```

### View RAG Logs
```bash
tail -f /tmp/rag_app.log
```

### Add New Documents
1. Copy PDFs to `airagagent/pdf_directory/`
2. System auto-processes on next request
3. Or call `/status` endpoint to trigger processing

## 📝 Notes

- **RAG System**: Processes documents automatically, stores embeddings in FAISS
- **Chat API**: Uses Grok Fast Reasoning for responses
- **Personalization**: Combines survey data + RAG context + user questions
- **Response Style**: Short, direct, aggressive (configured in prompts)

## 🆘 Troubleshooting

### RAG Flask not responding
```bash
# Check if running
ps aux | grep "python3 app.py"

# Check logs
tail -50 /tmp/rag_app.log

# Restart
pkill -f "python3 app.py"
cd /var/www/html/leadgen/airagagent && python3 app.py
```

### Chat API errors
- Check PHP error logs
- Verify database connection
- Ensure RAG Flask is running on port 5000

### Documents not processing
- Check file permissions in `pdf_directory/`
- Verify Python dependencies installed
- Check RAG Flask logs for errors

## 📞 Support

For issues or questions, check:
- RAG Flask logs: `/tmp/rag_app.log`
- PHP error logs: `/var/log/apache2/error.log` (or nginx equivalent)
- System status: `curl http://localhost:5000/status`
