# 📚 API Documentation

Complete API documentation for the LoudCannabis Lead Generation Platform.

## 🔐 Authentication

All state-changing operations (POST, PUT, DELETE) require:
- Admin session authentication (via `$_SESSION['UserName']`)
- CSRF token in request body or `X-CSRF-Token` header
- Rate limiting applies to all endpoints

### Getting CSRF Token

```
GET /api/sponsors.php?action=csrf_token
GET /api/affiliates.php?action=csrf_token
```

**Response:**
```json
{
  "success": true,
  "csrf_token": "abc123..."
}
```

---

## 👥 Affiliates API

**Base URL:** `/api/affiliates.php`

### List Affiliates

```
GET /api/affiliates.php?action=list
```

**Query Parameters:**
- `status` - Filter by status (active, pending, suspended)
- `search` - Search by company name, email, or phone
- `date_from` - Filter by creation date (YYYY-MM-DD)
- `date_to` - Filter by creation date (YYYY-MM-DD)
- `min_revenue` - Minimum revenue filter
- `max_revenue` - Maximum revenue filter
- `sort_by` - Sort field (created_at, email, company_name, status, revenue, conversions, total_clicks)
- `sort_order` - Sort direction (ASC, DESC)
- `page` - Page number (default: 1)
- `per_page` - Items per page (default: 25, max: 100)

**Response:**
```json
{
  "success": true,
  "affiliates": [
    {
      "id": 1,
      "email": "affiliate@example.com",
      "company_name": "Example Company",
      "phone": "+1234567890",
      "status": "active",
      "payment_method": "paypal",
      "payment_details": "paypal@example.com",
      "total_clicks": 1250,
      "conversions": 45,
      "revenue": 2250.00,
      "created_at": "2024-01-01 10:00:00"
    }
  ],
  "pagination": {
    "page": 1,
    "per_page": 25,
    "total": 100,
    "total_pages": 4
  }
}
```

### Get Affiliate Detail

```
GET /api/affiliates.php?action=detail&id={affiliateId}
```

**Response:**
```json
{
  "success": true,
  "affiliate": {
    "id": 1,
    "email": "affiliate@example.com",
    "company_name": "Example Company",
    "phone": "+1234567890",
    "status": "active",
    "payment_method": "paypal",
    "payment_details": "paypal@example.com",
    "total_clicks": 1250,
    "conversions": 45,
    "revenue": 2250.00,
    "created_at": "2024-01-01 10:00:00"
  }
}
```

### Update Affiliate

```
PUT /api/affiliates.php?action=update
POST /api/affiliates.php?action=update
```

**Request Body:**
```json
{
  "affiliateId": 1,
  "email": "newemail@example.com",
  "company_name": "New Company Name",
  "phone": "+1234567890",
  "status": "active",
  "payment_method": "paypal",
  "payment_details": "paypal@example.com",
  "status_reason": "Optional reason for status change"
}
```

**Response:**
```json
{
  "success": true,
  "message": "Affiliate updated successfully"
}
```

### Delete Affiliate (Soft Delete)

```
DELETE /api/affiliates.php?action=delete
```

**Request Body:**
```json
{
  "affiliateId": 1,
  "reason": "Optional reason for deletion"
}
```

**Response:**
```json
{
  "success": true,
  "message": "Affiliate deactivated successfully"
}
```

### Get Affiliate Status History

```
GET /api/affiliates.php?action=status_history&id={affiliateId}
```

**Response:**
```json
{
  "success": true,
  "history": [
    {
      "old_status": "pending",
      "new_status": "active",
      "reason": "Email verified and account approved",
      "changed_by": "admin",
      "changed_at": "2024-01-15 10:30:00"
    }
  ],
  "count": 1
}
```

### Get Top Performing Affiliates

```
GET /api/affiliates.php?action=top&range={range}
```

**Query Parameters:**
- `range` - Time range (7days, 30days, 90days, 1year)

**Response:**
```json
{
  "success": true,
  "affiliates": [
    {
      "company_name": "Top Affiliate",
      "clicks": 5000,
      "conversions": 200,
      "revenue": 10000.00
    }
  ]
}
```

---

## 🏢 Sponsors API

**Base URL:** `/api/sponsors.php`

### List Sponsors

```
GET /api/sponsors.php?action=sponsors
GET /api/sponsors.php?action=list
```

**Query Parameters:**
- `status` - Filter by status (active, inactive)
- `search` - Search by name, email, or industry
- `date_from` - Filter by creation date (YYYY-MM-DD)
- `date_to` - Filter by creation date (YYYY-MM-DD)
- `min_revenue` - Minimum revenue filter
- `max_revenue` - Maximum revenue filter
- `sort_by` - Sort field
- `sort_order` - Sort direction (ASC, DESC)
- `page` - Page number (default: 1)
- `per_page` - Items per page (default: 25, max: 100)

**Response:**
```json
{
  "success": true,
  "sponsors": [
    {
      "id": 1,
      "name": "Example Sponsor",
      "contact_email": "sponsor@example.com",
      "industry": "cannabis",
      "status": "active",
      "total_revenue": 5000.00,
      "total_offers": 5,
      "created_at": "2024-01-01 10:00:00"
    }
  ],
  "pagination": {
    "page": 1,
    "per_page": 25,
    "total": 50,
    "total_pages": 2
  }
}
```

### Get Sponsor Detail

```
GET /api/sponsors.php?action=detail&id={sponsorId}
GET /api/sponsors.php?action=sponsor&id={sponsorId}
```

**Response:**
```json
{
  "success": true,
  "sponsor": {
    "id": 1,
    "name": "Example Sponsor",
    "contact_email": "sponsor@example.com",
    "industry": "cannabis",
    "status": "active",
    "offer_count": 5,
    "active_offer_count": 3,
    "offers": [
      {
        "id": 1,
        "title": "Example Offer",
        "description": "Offer description",
        "offer_type": "link",
        "payout_amount": 50.00,
        "status": "active",
        "total_clicks": 100,
        "conversions": 10,
        "total_revenue": 500.00
      }
    ],
    "created_at": "2024-01-01 10:00:00"
  }
}
```

### Create Sponsor

```
POST /api/sponsors.php?action=sponsor
```

**Request Body:**
```json
{
  "name": "New Sponsor",
  "contact_email": "sponsor@example.com",
  "industry": "cannabis",
  "status": "active"
}
```

**Response:**
```json
{
  "success": true,
  "sponsor_id": 123,
  "message": "Sponsor created"
}
```

### Update Sponsor

```
PUT /api/sponsors.php?action=sponsor
POST /api/sponsors.php?action=sponsor_update
```

**Request Body:**
```json
{
  "id": 1,
  "name": "Updated Sponsor Name",
  "contact_email": "newemail@example.com",
  "industry": "cannabis",
  "status": "active",
  "status_reason": "Optional reason for status change"
}
```

**Response:**
```json
{
  "success": true,
  "message": "Sponsor updated",
  "sponsor": { ... }
}
```

### Delete Sponsor (Soft Delete)

```
DELETE /api/sponsors.php?action=sponsor
```

**Request Body:**
```json
{
  "id": 1,
  "reason": "Optional reason for deletion"
}
```

**Response:**
```json
{
  "success": true,
  "message": "Sponsor deactivated"
}
```

### Get Sponsor Status History

```
GET /api/sponsors.php?action=status_history&id={sponsorId}
```

**Response:**
```json
{
  "success": true,
  "history": [
    {
      "old_status": "active",
      "new_status": "inactive",
      "reason": "Contract expired",
      "changed_by": "admin",
      "changed_at": "2024-01-15 10:30:00"
    }
  ],
  "count": 1
}
```

### Get Sponsor Offer Count

```
GET /api/sponsors.php?action=offer_count&id={sponsorId}
```

**Response:**
```json
{
  "success": true,
  "offer_count": 5
}
```

---

## 🎯 Offers API

**Base URL:** `/api/sponsors.php` (offers are managed through sponsors API)

### List Offers

```
GET /api/sponsors.php?action=offers
```

**Query Parameters:**
- `sponsor_id` - Filter by sponsor ID
- `status` - Filter by status (active, inactive)
- `type` - Filter by type (link, call)
- `search` - Search by title or description
- `date_from` - Filter by creation date (YYYY-MM-DD)
- `date_to` - Filter by creation date (YYYY-MM-DD)
- `min_payout` - Minimum payout filter
- `max_payout` - Maximum payout filter
- `sort_by` - Sort field
- `sort_order` - Sort direction (ASC, DESC)
- `page` - Page number (default: 1)
- `per_page` - Items per page (default: 25, max: 100)

**Response:**
```json
{
  "success": true,
  "offers": [
    {
      "id": 1,
      "title": "Example Offer",
      "description": "Offer description",
      "sponsor_id": 1,
      "sponsor_name": "Example Sponsor",
      "offer_type": "link",
      "payout_amount": 50.00,
      "status": "active",
      "total_clicks": 100,
      "total_conversions": 10,
      "total_revenue": 500.00,
      "created_at": "2024-01-01 10:00:00"
    }
  ],
  "pagination": {
    "page": 1,
    "per_page": 25,
    "total": 50,
    "total_pages": 2
  }
}
```

### Get Offer Detail

```
GET /api/sponsors.php?action=offer_detail&id={offerId}
```

**Response:**
```json
{
  "success": true,
  "offer": {
    "id": 1,
    "title": "Example Offer",
    "description": "Offer description",
    "sponsor_id": 1,
    "sponsor_name": "Example Sponsor",
    "offer_type": "link",
    "payout_amount": 50.00,
    "status": "active",
    "total_clicks": 100,
    "conversions": 10,
    "total_revenue": 500.00,
    "created_at": "2024-01-01 10:00:00"
  }
}
```

### Create Offer

```
POST /api/sponsors.php?action=offer
```

**Request Body:**
```json
{
  "sponsor_id": 1,
  "title": "New Offer",
  "description": "Offer description",
  "offer_type": "link",
  "payout_amount": 50.00,
  "status": "active",
  "qualification_criteria": {
    "min_age": 21,
    "location": "US"
  }
}
```

**Response:**
```json
{
  "success": true,
  "offer_id": 123,
  "message": "Offer created"
}
```

### Update Offer

```
PUT /api/sponsors.php?action=offer
POST /api/sponsors.php?action=offer_update
```

**Request Body:**
```json
{
  "id": 1,
  "title": "Updated Offer Title",
  "description": "Updated description",
  "payout_amount": 75.00,
  "status": "active",
  "status_reason": "Optional reason for status change"
}
```

**Response:**
```json
{
  "success": true,
  "message": "Offer updated",
  "offer": { ... }
}
```

### Delete Offer (Soft Delete)

```
DELETE /api/sponsors.php?action=offer
```

**Request Body:**
```json
{
  "id": 1,
  "reason": "Optional reason for deletion"
}
```

**Response:**
```json
{
  "success": true,
  "message": "Offer deactivated"
}
```

### Get Offer Status History

```
GET /api/sponsors.php?action=offer_status_history&id={offerId}
```

**Response:**
```json
{
  "success": true,
  "history": [
    {
      "old_status": "active",
      "new_status": "inactive",
      "reason": "Low performance",
      "changed_by": "admin",
      "changed_at": "2024-01-15 10:30:00"
    }
  ],
  "count": 1
}
```

### Get Offer Performance Metrics

```
GET /api/sponsors.php?action=offer_performance&id={offerId}&range={range}
```

**Query Parameters:**
- `id` - Offer ID (required)
- `range` - Time range (7days, 30days, 90days, 1year) - default: 30days

**Response:**
```json
{
  "success": true,
  "performance": {
    "total_clicks": 1250,
    "conversions": 45,
    "total_revenue": 2250.00,
    "unique_affiliates": 12,
    "avg_conversion_value": 50.00,
    "conversion_rate": 3.6
  },
  "range": "30days"
}
```

### Get Top Performing Offers

```
GET /api/sponsors.php?action=top_offers&range={range}
```

**Query Parameters:**
- `range` - Time range (7days, 30days, 90days, 1year) - default: 30days

**Response:**
```json
{
  "success": true,
  "offers": [
    {
      "title": "Top Offer",
      "clicks": 5000,
      "conversions": 200,
      "revenue": 10000.00
    }
  ]
}
```

---

## 📊 Tracking API

**Base URL:** `/api/tracking.php`

### Get Tracking Events

```
GET /api/tracking.php
```

**Query Parameters:**
- `affiliate_id` - Filter by affiliate ID
- `sponsor_id` - Filter by sponsor ID
- `offer_id` - Filter by offer ID
- `event_type` - Filter by event type (click, conversion, call_qualified)
- `date_from` - Filter by date (YYYY-MM-DD)
- `date_to` - Filter by date (YYYY-MM-DD)
- `daily` - Return daily aggregated data (true/false)
- `group_by_date` - Group results by date (true/false)
- `page` - Page number (default: 1)
- `per_page` - Items per page (default: 25, max: 100)

**Response:**
```json
{
  "success": true,
  "events": [
    {
      "id": 1,
      "affiliate_id": 1,
      "affiliate_name": "Example Affiliate",
      "offer_id": 1,
      "offer_title": "Example Offer",
      "sponsor_id": 1,
      "sponsor_name": "Example Sponsor",
      "event_type": "conversion",
      "revenue": 50.00,
      "created_at": "2024-01-15 10:30:00"
    }
  ],
  "pagination": {
    "page": 1,
    "per_page": 25,
    "total": 100,
    "total_pages": 4
  }
}
```

---

## 📈 Reports & Analytics API

**Base URL:** `/api/admin_stats.php` and `/api/realtime.php`

### Get Overview Statistics

```
GET /api/admin_stats.php
```

**Response:**
```json
{
  "success": true,
  "stats": {
    "total_affiliates": 100,
    "total_sponsors": 50,
    "total_offers": 200,
    "total_conversions": 5000,
    "today_clicks": 250,
    "total_revenue": 50000.00
  }
}
```

### Get Real-time Stats

```
GET /api/realtime.php?type=stats
```

**Response:**
```json
{
  "success": true,
  "type": "stats",
  "data": {
    "total_affiliates": 100,
    "total_sponsors": 50,
    "active_offers": 150,
    "total_conversions": 5000,
    "todays_clicks": 250,
    "total_revenue": 50000.00,
    "updated_at": 1704067200
  },
  "timestamp": 1704067200
}
```

### Get Recent Activity

```
GET /api/realtime.php?type=activity&since={timestamp}
```

**Query Parameters:**
- `type` - Must be "activity"
- `since` - Unix timestamp (optional, defaults to last 5 minutes)

**Response:**
```json
{
  "success": true,
  "type": "activity",
  "data": [
    {
      "id": 1,
      "affiliate_name": "Example Affiliate",
      "offer_title": "Example Offer",
      "sponsor_name": "Example Sponsor",
      "event_type": "conversion",
      "revenue": 50.00,
      "created_at": "2024-01-15 10:30:00"
    }
  ],
  "count": 10,
  "timestamp": 1704067200
}
```

### Get Performance Metrics

```
GET /api/realtime.php?type=metrics&range={range}
```

**Query Parameters:**
- `type` - Must be "metrics"
- `range` - Time range (1hour, 1day, 1week) - default: 1hour

**Response:**
```json
{
  "success": true,
  "type": "metrics",
  "data": {
    "clicks": 1250,
    "conversions": 45,
    "revenue": 2250.00,
    "conversion_rate": 3.6
  },
  "range": "1hour",
  "timestamp": 1704067200
}
```

### Get Notifications

```
GET /api/realtime.php?type=notifications&since={timestamp}
```

**Query Parameters:**
- `type` - Must be "notifications"
- `since` - Unix timestamp (optional, defaults to last hour)

**Response:**
```json
{
  "success": true,
  "type": "notifications",
  "data": [
    {
      "type": "info",
      "message": "5 new affiliate(s) registered",
      "timestamp": 1704067200
    },
    {
      "type": "success",
      "message": "3 high-value conversion(s) worth $500.00",
      "timestamp": 1704067200
    }
  ],
  "count": 2,
  "timestamp": 1704067200
}
```

---

## 🔒 Security Features

### Rate Limiting

All endpoints are rate-limited:
- **Default:** 100 requests per minute
- **Create operations:** 20 requests per minute
- **Update operations:** 50 requests per minute
- **Delete operations:** 10 requests per minute
- **List operations:** 200 requests per minute

**Rate Limit Headers:**
```
X-RateLimit-Limit: 100
X-RateLimit-Remaining: 95
X-RateLimit-Reset: 1704067200
```

**Rate Limit Exceeded Response (429):**
```json
{
  "success": false,
  "error": "Rate limit exceeded",
  "message": "Too many requests. Please try again in 45 seconds.",
  "retry_after": 45
}
```

### CSRF Protection

All POST, PUT, DELETE requests require a CSRF token:
- Include in request body: `{"csrf_token": "..."}`
- Or include in header: `X-CSRF-Token: ...`

**CSRF Token Validation Failed (403):**
```json
{
  "success": false,
  "error": "CSRF token validation failed. Please refresh the page and try again."
}
```

### Authentication

State-changing operations require admin authentication:
- Session-based authentication via `$_SESSION['UserName']`
- Unauthorized access attempts are logged
- Currently in logging-only mode (can be made stricter)

**Unauthorized Response (401):**
```json
{
  "success": false,
  "error": "Authentication required",
  "message": "Please log in to access this resource"
}
```

---

## 📝 Input Validation

All inputs are validated and sanitized:

### Validation Rules

- **String:** Min/max length, HTML sanitization, SQL injection prevention
- **Email:** Email format validation
- **Phone:** Phone number format validation (7-20 digits)
- **Integer:** Min/max value validation
- **Float:** Min/max value validation
- **Status:** Must be one of allowed values
- **Date:** Date format validation

### Sanitization

- Null byte removal
- Control character removal
- HTML entity encoding (XSS prevention)
- HTML/XML tag stripping
- SQL injection pattern detection
- JavaScript event handler removal
- Script tag removal

---

## 💾 Caching

All GET endpoints are cached:
- **Sponsors List:** 5 minutes
- **Sponsor Detail:** 10 minutes
- **Offers List:** 5 minutes
- **Offer Detail:** 10 minutes
- **Affiliates List:** 5 minutes
- **Affiliate Detail:** 10 minutes
- **Stats:** 1 minute (frequently updated)
- **Reports:** 5 minutes
- **Top Offers:** 10 minutes
- **Top Affiliates:** 10 minutes

Cache is automatically invalidated on:
- Create operations
- Update operations
- Delete operations

---

## 🐛 Error Handling

All endpoints return consistent error responses:

### Success Response
```json
{
  "success": true,
  "data": { ... }
}
```

### Error Response
```json
{
  "success": false,
  "error": "Error message",
  "errors": ["Detailed error 1", "Detailed error 2"]
}
```

### HTTP Status Codes

- **200 OK** - Success
- **400 Bad Request** - Invalid input
- **401 Unauthorized** - Authentication required
- **403 Forbidden** - Insufficient permissions or CSRF failure
- **404 Not Found** - Resource not found
- **429 Too Many Requests** - Rate limit exceeded
- **500 Internal Server Error** - Server error

---

## 📊 Response Format

All responses follow this structure:

### Success Response
```json
{
  "success": true,
  "data": { ... },
  "message": "Optional success message"
}
```

### Paginated Response
```json
{
  "success": true,
  "data": [ ... ],
  "pagination": {
    "page": 1,
    "per_page": 25,
    "total": 100,
    "total_pages": 4
  }
}
```

### Error Response
```json
{
  "success": false,
  "error": "Error message",
  "errors": ["Detailed error messages"]
}
```

---

## 🔄 Status Change History

All status changes are tracked in history tables:

### Tables
- `affiliate_status_history` - Affiliate status changes
- `sponsor_status_history` - Sponsor status changes
- `offer_status_history` - Offer status changes

### History Record Structure
```json
{
  "old_status": "pending",
  "new_status": "active",
  "reason": "Optional reason",
  "changed_by": "admin",
  "changed_at": "2024-01-15 10:30:00"
}
```

---

## 📈 Performance Metrics

### Offer Performance Metrics
- Total clicks
- Conversions
- Total revenue
- Unique affiliates
- Average conversion value
- Conversion rate (calculated)

### Time Ranges
- `7days` - Last 7 days
- `30days` - Last 30 days (default)
- `90days` - Last 90 days
- `1year` - Last year

---

## 🎯 Best Practices

1. **Always include CSRF token** for state-changing operations
2. **Handle rate limiting** - Check `X-RateLimit-Remaining` header
3. **Use pagination** for large datasets
4. **Cache responses** when appropriate (respect cache headers)
5. **Handle errors gracefully** - Check `success` field first
6. **Validate inputs** before sending requests
7. **Use appropriate HTTP methods** (GET for read, POST/PUT for write, DELETE for delete)

---

## 📞 Support

For API support or questions, contact the development team.

**Last Updated:** 2024-01-15
**API Version:** 1.0

