#!/bin/bash

set -e

echo "🚀 Starting DeepLeaf AI Application..."

# Create necessary directories
mkdir -p /var/www/html/storage/logs
mkdir -p /var/www/html/storage/cache
mkdir -p /var/www/html/storage/sessions
mkdir -p /var/www/html/storage/uploads

# Set proper permissions
chown -R www-data:www-data /var/www/html/storage
chmod -R 755 /var/www/html/storage

# Copy .env.example to .env if .env doesn't exist
if [ ! -f /var/www/html/.env ]; then
    echo "📋 Creating .env file from template..."
    cp /var/www/html/env.example /var/www/html/.env
    echo "⚠️  Please update the .env file with your actual configuration values!"
fi

# Run database migrations if they exist
if [ -d /var/www/html/database/migrations ] && [ "$(ls -A /var/www/html/database/migrations)" ]; then
    echo "🗄️  Running database migrations..."
    # Add migration logic here if needed
fi

# Clear and cache config (if Laravel-like commands exist)
if [ -f /var/www/html/artisan ]; then
    echo "🔧 Optimizing Laravel configuration..."
    php artisan config:cache
    php artisan route:cache
    php artisan view:cache
fi

# Install Python dependencies for RAG service
if [ -f /var/www/html/airagagent/requirements.txt ]; then
    echo "🐍 Installing Python dependencies..."
    pip3 install --no-cache-dir -r /var/www/html/airagagent/requirements.txt
fi

# Generate application key if needed
if [ -f /var/www/html/artisan ] && ! grep -q "APP_KEY=.*[^[:space:]]" /var/www/html/.env; then
    echo "🔑 Generating application key..."
    php artisan key:generate
fi

echo "✅ Application startup complete!"
echo "🌐 Application will be available at http://localhost:8080"
echo "🔍 Health check: http://localhost:8080/api/health"

# Execute the main command
exec "$@"
