#!/bin/bash

# Configuration Extraction Script for Apts R US
# Run this BEFORE deleting the old website to extract important settings

echo "🔍 Extracting configuration information from current website..."
echo "================================================================="

# Find current website directory
CURRENT_SITE="/var/www/html"
if [ ! -d "$CURRENT_SITE" ]; then
    echo "❌ Current website directory not found at $CURRENT_SITE"
    echo "Please specify the correct path:"
    read -p "Enter current website directory: " CURRENT_SITE
fi

echo "📂 Scanning directory: $CURRENT_SITE"
echo ""

# Extract email/SMTP configuration
echo "📧 EMAIL & SMTP CONFIGURATION:"
echo "------------------------------"
grep -r "smtp\|mail\|email\|gmail\|outlook\|yahoo" "$CURRENT_SITE" \
    --include="*.php" \
    --include="*.js" \
    --include="*.py" \
    --include="*.config" \
    --include=".env*" \
    2>/dev/null || echo "No email configuration found"

echo ""

# Extract database configuration
echo "🗄️ DATABASE CONFIGURATION:"
echo "--------------------------"
grep -r "mysql\|database\|db_\|pdo\|mysqli\|host\|port\|username\|password" "$CURRENT_SITE" \
    --include="*.php" \
    --include="*.config" \
    --include=".env*" \
    2>/dev/null || echo "No database configuration found"

echo ""

# Extract payment/API keys
echo "💳 PAYMENT & API CONFIGURATION:"
echo "-------------------------------"
grep -r "stripe\|paypal\|api\|key\|secret\|token\|webhook" "$CURRENT_SITE" \
    --include="*.php" \
    --include="*.js" \
    --include="*.config" \
    --include=".env*" \
    2>/dev/null | grep -v "public" || echo "No payment/API configuration found"

echo ""

# Extract contact information
echo "📞 CONTACT INFORMATION:"
echo "-----------------------"
grep -r "phone\|contact\|support\|info@\|admin@" "$CURRENT_SITE" \
    --include="*.php" \
    --include="*.html" \
    --include="*.js" \
    2>/dev/null || echo "No contact information found"

echo ""

# List all configuration files
echo "📋 CONFIGURATION FILES FOUND:"
echo "-----------------------------"
find "$CURRENT_SITE" -name "*.config" -o -name ".env*" -o -name "config.php" -o -name "settings.php" -o -name "wp-config.php" 2>/dev/null

echo ""

# Check for .htaccess or nginx configs
echo "🌐 SERVER CONFIGURATION:"
echo "------------------------"
find "$CURRENT_SITE" -name ".htaccess" -o -name "nginx.conf" -o -name "apache.conf" 2>/dev/null

echo ""

# Create backup summary
echo "💾 CREATING BACKUP SUMMARY..."
BACKUP_DIR="/root/website_config_backup_$(date +%Y%m%d_%H%M%S)"
sudo mkdir -p "$BACKUP_DIR"

# Backup important files
sudo find "$CURRENT_SITE" -name "*.config" -o -name ".env*" -o -name "config.php" -o -name "settings.php" | head -10 | xargs -I {} sudo cp {} "$BACKUP_DIR/" 2>/dev/null || true

echo "✅ Configuration backed up to: $BACKUP_DIR"
echo ""
echo "⚠️  IMPORTANT REMINDERS:"
echo "1. Save any SMTP credentials you found above"
echo "2. Note any API keys or payment processor settings"
echo "3. Remember database connection details if any"
echo "4. The backup directory contains copies of config files"
echo ""
echo "🗑️  Ready to proceed with website replacement?"
read -p "Press Enter when ready to continue..."

echo "✅ Configuration extraction complete!"
