import requests
import json
import csv
import datetime
from typing import Dict, List, Optional
import logging
from dataclasses import dataclass, asdict
import os
import smtplib
from email.mime.text import MIMEText
from email.mime.multipart import MIMEMultipart
from flask import Flask, render_template, request, jsonify, session
from dotenv import load_dotenv

# Load environment variables
load_dotenv()

# Configure logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

@dataclass
class ApplicationData:
    """Data structure for rental application"""
    # Property Information
    apartment_name_link: str
    lease_duration: str
    move_in_date: str
    unit_details: str

    # Contact Information (required fields first)
    full_name: str
    email: str
    phone: str

    # Occupant Information
    adults_count: int
    kids_count: int

    # Optional fields with defaults
    emergency_contact_name: Optional[str] = None
    emergency_contact_email: Optional[str] = None
    emergency_contact_phone: Optional[str] = None
    emergency_contact_relationship: Optional[str] = None

    # Parking Information
    car_make_model: Optional[str] = None
    vin_number: Optional[str] = None
    license_tag: Optional[str] = None

    # Application metadata
    application_date: str = None
    application_id: str = None
    status: str = "pending"

class AptsRUsChatAgent:
    """AI Agent for handling rental applications"""

    def __init__(self, grok_api_key: str, admin_email: str = "info@aptsrus.com"):
        self.grok_api_key = grok_api_key
        self.admin_email = admin_email
        self.grok_api_url = "https://api.x.ai/v1/chat/completions"
        self.model = "grok-beta"  # Update to grok-4-1-fast-reasoning if available
        self.current_application = None
        self.conversation_history = []

        # Enhanced System prompt for Grok
        self.system_prompt = """You are an advanced AI rental application assistant for Apts R US. Your primary role is to guide applicants through the rental application process while being friendly, professional, and extremely encouraging.

        YOUR CORE MISSION:
        - Help EVERY applicant complete their rental application successfully
        - Emphasize that we approve ALL applicants regardless of background, credit, or employment
        - Make the process feel simple, fast, and guaranteed

        KEY BUSINESS PRINCIPLES TO COMMUNICATE:
        1. ✅ ZERO REJECTIONS: We approve everyone - no credit checks, no background checks, no employment verification
        2. 💰 SIMPLE PROCESS: $500 deposit starts everything, $1000 total for approval
        3. ⚡ FAST APPROVAL: 7-10 day review period with guaranteed acceptance
        4. 🏠 BROAD NETWORK: We work with thousands of private owners nationwide
        5. 📞 EXCELLENT SUPPORT: Mon-Fri 9 AM - 6 PM at (555) 123-4567 or info@aptsrus.com

        CONVERSATION STYLE:
        - Be extremely encouraging and positive
        - Use emojis to make conversations fun and engaging
        - Repeat the "we approve everyone" guarantee frequently
        - Ask one question at a time to avoid overwhelming applicants
        - Validate information as you collect it
        - Show enthusiasm about helping them find their perfect home

        INFORMATION COLLECTION STRATEGY:
        1. PROPERTY INFO: Apartment name/link, lease duration, move-in date, unit details
        2. OCCUPANT INFO: Number of adults and children
        3. EMERGENCY CONTACT: Optional but recommended
        4. PARKING INFO: Only if they mention needing parking
        5. CONTACT DETAILS: Full name, email, phone (required)

        VALIDATION RULES:
        - Email must contain @ symbol
        - Phone should be 10 digits or with formatting
        - Dates should be realistic (not in the past for move-in)
        - Be flexible with formatting but ensure data quality

        HANDLING OBJECTIONS:
        - "Bad credit? No problem! We don't check credit."
        - "Criminal background? Doesn't matter to us."
        - "Unemployed? We don't verify employment."
        - "Low income? Our private owners understand all situations."

        CLOSING THE APPLICATION:
        - Once all info is collected, encourage immediate submission
        - Remind them of the $500 deposit to start the process
        - Emphasize the speed (7-10 days) and guarantee of approval
        - Provide clear next steps and contact information

        REMEMBER: Your goal is to collect complete, accurate information while making every applicant feel excited and confident about their approval. We truly do approve everyone!
        """

    def chat_with_grok(self, user_message: str) -> str:
        """Send message to Grok API and get response"""

        headers = {
            "Authorization": f"Bearer {self.grok_api_key}",
            "Content-Type": "application/json"
        }

        # Prepare messages including history
        messages = [
            {"role": "system", "content": self.system_prompt}
        ]

        # Add conversation history
        messages.extend(self.conversation_history)

        # Add current user message
        messages.append({"role": "user", "content": user_message})

        payload = {
            "model": self.model,
            "messages": messages,
            "max_tokens": 500,
            "temperature": 0.7
        }

        try:
            response = requests.post(self.grok_api_url, headers=headers, json=payload)
            response.raise_for_status()

            ai_response = response.json()["choices"][0]["message"]["content"]

            # Update conversation history
            self.conversation_history.append({"role": "user", "content": user_message})
            self.conversation_history.append({"role": "assistant", "content": ai_response})

            return ai_response

        except Exception as e:
            logger.error(f"Error calling Grok API: {e}")
            return "I apologize, I'm having trouble connecting right now. Please contact us directly at (555) 123-4567 or info@aptsrus.com"

    def start_application(self) -> str:
        """Start a new rental application"""
        self.current_application = ApplicationData(
            application_date=datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
            application_id=f"APT{datetime.datetime.now().strftime('%Y%m%d%H%M%S')}"
        )

        welcome_message = """🏠 Welcome to Apts R US Rental Application! 🏠

I'm your AI assistant here to help you apply for your perfect rental. Here's what you need to know:

✅ WE APPROVE EVERYONE: No credit check, no criminal background check, no employment verification
💰 $500 Deposit: Starts your application process
📋 Simple Process: Just answer a few questions
🏢 All Properties: Apartments, houses, condos everywhere
✅ Guaranteed Approval for all qualified applicants

Let's get started! First, I need some information about the property you're interested in:

1. What's the apartment name or website link for the property?
"""
        self.conversation_history.append({"role": "assistant", "content": welcome_message})
        return welcome_message

    def extract_application_data(self, user_messages: List[str]) -> bool:
        """Extract structured data from conversation history using pattern matching"""
        try:
            if not self.current_application:
                return False

            # Combine all user messages for analysis
            full_conversation = ' '.join(user_messages).lower()

            # Extract data using pattern matching
            # This is a simplified implementation - in production, you'd use NLP or Grok's function calling

            # Property Information
            if not self.current_application.apartment_name_link:
                # Look for apartment names or links
                for msg in user_messages:
                    if 'apartment' in msg.lower() or 'apt' in msg.lower() or 'http' in msg.lower():
                        self.current_application.apartment_name_link = msg.strip()
                        break

            # Lease duration
            duration_keywords = ['month', 'year', '6 months', '12 months', '1 year']
            for duration in duration_keywords:
                if duration in full_conversation:
                    self.current_application.lease_duration = duration
                    break

            # Move-in date
            date_patterns = [
                r'\b\d{1,2}/\d{1,2}/\d{4}\b',  # MM/DD/YYYY
                r'\b\d{1,2}-\d{1,2}-\d{4}\b',  # MM-DD-YYYY
                r'\b(jan|feb|mar|apr|may|jun|jul|aug|sep|oct|nov|dec)\w*\s+\d{1,2},?\s+\d{4}\b'
            ]
            import re
            for pattern in date_patterns:
                match = re.search(pattern, full_conversation, re.IGNORECASE)
                if match:
                    self.current_application.move_in_date = match.group()
                    break

            # Unit details
            unit_keywords = ['bedroom', 'bathroom', 'sq ft', 'square feet', 'studio', '1br', '2br', '3br']
            for msg in user_messages:
                if any(keyword in msg.lower() for keyword in unit_keywords):
                    self.current_application.unit_details = msg.strip()
                    break

            # Contact Information (most important)
            # Extract email
            email_pattern = r'\b[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\.[A-Z|a-z]{2,}\b'
            email_match = re.search(email_pattern, full_conversation)
            if email_match:
                self.current_application.email = email_match.group()

            # Extract phone
            phone_pattern = r'\b\d{3}[-.]?\d{3}[-.]?\d{4}\b'
            phone_match = re.search(phone_pattern, full_conversation)
            if phone_match:
                self.current_application.phone = phone_match.group()

            # Extract name (simple heuristic - look for capitalized words)
            name_pattern = r'\b[A-Z][a-z]+\s+[A-Z][a-z]+\b'
            name_match = re.search(name_pattern, full_conversation)
            if name_match:
                self.current_application.full_name = name_match.group()

            # Occupant counts (look for numbers)
            adult_patterns = [r'(\d+)\s+adult', r'(\d+)\s+person', r'(\d+)\s+people']
            for pattern in adult_patterns:
                match = re.search(pattern, full_conversation)
                if match:
                    self.current_application.adults_count = int(match.group(1))
                    break

            kid_patterns = [r'(\d+)\s+(?:child|kid|children)']
            for pattern in kid_patterns:
                match = re.search(pattern, full_conversation)
                if match:
                    self.current_application.kids_count = int(match.group(1))
                    break

            # Emergency contact
            emergency_keywords = ['emergency', 'contact', 'backup']
            for msg in user_messages:
                if any(keyword in msg.lower() for keyword in emergency_keywords):
                    # Simple extraction - in production use better NLP
                    self.current_application.emergency_contact_name = msg.strip()
                    break

            # Parking info
            parking_keywords = ['car', 'vehicle', 'parking', 'vin', 'license']
            for msg in user_messages:
                if any(keyword in msg.lower() for keyword in parking_keywords):
                    self.current_application.car_make_model = msg.strip()
                    break

            logger.info("Data extraction completed for application")
            return True

        except Exception as e:
            logger.error(f"Error extracting application data: {e}")
            return False

    def send_to_admin(self) -> bool:
        """Send completed application to admin"""
        if not self.current_application:
            return False

        try:
            # Prepare application summary
            app_data = asdict(self.current_application)
            app_summary = self._format_application_summary(app_data)

            # In production, you would:
            # 1. Send email to admin
            # 2. Save to database
            # 3. Send notification

            # Mock email sending (replace with actual email service)
            email_sent = self._send_email_notification(app_summary)

            # Also log to file/database
            self._save_application(app_data)

            return email_sent

        except Exception as e:
            logger.error(f"Error sending to admin: {e}")
            return False

    def _format_application_summary(self, app_data: Dict) -> str:
        """Format application data for admin review"""
        summary = f"""

        ⚡ NEW RENTAL APPLICATION - Apts R US ⚡
        Application ID: {app_data.get('application_id', 'N/A')}
        Date: {app_data.get('application_date', 'N/A')}

        === PROPERTY INFORMATION ===
        Property: {app_data.get('apartment_name_link', 'Not provided')}
        Lease Duration: {app_data.get('lease_duration', 'Not provided')}
        Move-in Date: {app_data.get('move_in_date', 'Not provided')}
        Unit Details: {app_data.get('unit_details', 'Not provided')}

        === OCCUPANT INFORMATION ===
        Adults: {app_data.get('adults_count', 0)}
        Children: {app_data.get('kids_count', 0)}
        Emergency Contact: {app_data.get('emergency_contact_name', 'Not provided')}

        === CONTACT INFORMATION ===
        Name: {app_data.get('full_name', 'Not provided')}
        Email: {app_data.get('email', 'Not provided')}
        Phone: {app_data.get('phone', 'Not provided')}

        === ADDITIONAL INFO ===
        Parking Needed: {'Yes' if app_data.get('car_make_model') else 'No'}

        ⭐ REMINDER: This applicant is pre-approved per our guarantee!
        ✅ No credit/background/employment checks needed
        """
        return summary

    def _send_email_notification(self, content: str) -> bool:
        """Send email notification to admin"""
        try:
            # Email configuration from environment variables
            smtp_server = os.environ.get('SMTP_SERVER', 'smtp.gmail.com')
            smtp_port = int(os.environ.get('SMTP_PORT', '587'))
            smtp_username = os.environ.get('SMTP_USERNAME', '')
            smtp_password = os.environ.get('SMTP_PASSWORD', '')
            from_email = os.environ.get('FROM_EMAIL', 'applications@aptsrus.com')

            # If no SMTP credentials provided, log and return success for demo
            if not smtp_username or not smtp_password:
                logger.info(f"SMTP credentials not configured. Would send email to {self.admin_email}")
                logger.info(f"Email content:\n{content}")
                return True

            # Create message
            msg = MIMEMultipart()
            msg['From'] = from_email
            msg['To'] = self.admin_email
            msg['Subject'] = f'🏠 NEW RENTAL APPLICATION - {self.current_application.application_id}'

            # Add body
            msg.attach(MIMEText(content, 'plain'))

            # Send email
            with smtplib.SMTP(smtp_server, smtp_port) as server:
                server.starttls()
                server.login(smtp_username, smtp_password)
                server.send_message(msg)

            logger.info(f"Email sent successfully to {self.admin_email}")
            return True

        except Exception as e:
            logger.error(f"Error sending email: {e}")
            # Log the content anyway for manual processing
            logger.info(f"Email content that failed to send:\n{content}")
            return False

    def _save_application(self, app_data: Dict):
        """Save application to CSV file"""
        try:
            # Define CSV headers
            csv_headers = [
                'application_id', 'application_date', 'status',
                'apartment_name_link', 'lease_duration', 'move_in_date', 'unit_details',
                'full_name', 'email', 'phone',
                'adults_count', 'kids_count',
                'emergency_contact_name', 'emergency_contact_email', 'emergency_contact_phone', 'emergency_contact_relationship',
                'car_make_model', 'vin_number', 'license_tag'
            ]

            # Ensure applications directory exists
            os.makedirs('applications', exist_ok=True)

            # Path to master CSV file
            csv_file = 'applications/all_applications.csv'

            # Check if file exists to determine if we need headers
            file_exists = os.path.exists(csv_file)

            # Prepare row data
            row_data = {header: app_data.get(header, '') for header in csv_headers}

            with open(csv_file, 'a', newline='', encoding='utf-8') as f:
                writer = csv.DictWriter(f, fieldnames=csv_headers)

                # Write headers if file doesn't exist
                if not file_exists:
                    writer.writeheader()

                # Write application data
                writer.writerow(row_data)

            # Also save individual JSON file for backup/reference
            json_filename = f"applications/{app_data.get('application_id', 'temp')}.json"
            with open(json_filename, 'w') as f:
                json.dump(app_data, f, indent=2)

            logger.info(f"Application saved to CSV: {csv_file} and JSON backup: {json_filename}")

        except Exception as e:
            logger.error(f"Error saving application: {e}")

    def complete_application(self) -> str:
        """Finalize and submit application"""
        if not self.current_application:
            return "No application in progress. Please start a new application."

        # Extract data from conversation
        success = self.extract_application_data(
            [msg["content"] for msg in self.conversation_history if msg["role"] == "user"]
        )

        if success:
            # Send to admin
            admin_notified = self.send_to_admin()

            completion_message = f"""

            🎉 APPLICATION SUBMITTED SUCCESSFULLY! 🎉

            Thank you for completing your Apts R US rental application!

            Application ID: {self.current_application.application_id}

            📋 NEXT STEPS:
            1. Our team will review your application (7-10 days)
            2. We'll match you with property owners in our network
            3. You'll be contacted for lease signing
            4. Pay remaining $500 upon approval
            5. Move into your new home!

            ⚡ REMEMBER: Your $500 deposit starts the process!
            ✅ GUARANTEE: You ARE approved through our network!

            For questions, contact us:
            📧 {self.admin_email}
            📞 (555) 123-4567
            🕒 Mon-Fri 9 AM - 6 PM

            Welcome to the Apts R US family! 🏠
            """

            # Reset for next application
            self.current_application = None
            self.conversation_history = []

            return completion_message
        else:
            return "I need a bit more information to complete your application. Please provide the missing details."

app = Flask(__name__)
app.secret_key = os.environ.get('FLASK_SECRET_KEY', os.urandom(24))

@app.route('/')
def home():
    return render_template('index.html')

@app.route('/start-chat', methods=['POST'])
def start_chat():
    if 'agent' not in session:
        # Initialize with your Grok API key
        grok_api_key = os.environ.get('GROK_API_KEY', 'your-api-key-here')
        session['agent'] = AptsRUsChatAgent(grok_api_key)

    agent = session['agent']
    welcome = agent.start_application()

    return jsonify({
        'response': welcome,
        'application_started': True
    })

@app.route('/chat', methods=['POST'])
def chat():
    user_message = request.json.get('message', '')

    if 'agent' not in session:
        return jsonify({'response': 'Please start a new application first.'})

    agent = session['agent']
    response = agent.chat_with_grok(user_message)

    return jsonify({'response': response})

@app.route('/submit-application', methods=['POST'])
def submit_application():
    if 'agent' not in session:
        return jsonify({'response': 'No application in progress.'})

    agent = session['agent']
    completion = agent.complete_application()

    # Clear session
    session.pop('agent', None)

    return jsonify({'response': completion})

if __name__ == '__main__':
    # Create applications directory
    os.makedirs('applications', exist_ok=True)
    app.run(debug=True, port=5000)
