#!/bin/bash

# Apts R US AI Chat Application - Deployment Script
# Run this on the server after uploading the application

echo "🚀 Starting Apts R US AI Chat Application Deployment"

# Update system packages
echo "📦 Updating system packages..."
sudo apt update && sudo apt upgrade -y

# Install Python and pip if not installed
echo "🐍 Installing Python and pip..."
sudo apt install -y python3 python3-pip python3-venv

# Install required system packages
echo "🔧 Installing system dependencies..."
sudo apt install -y nginx supervisor

# Create application directory
echo "📁 Setting up application directory..."
sudo mkdir -p /var/www/aptsrus-ai
sudo chown $USER:$USER /var/www/aptsrus-ai

# Copy application files (assuming they're uploaded to /tmp)
echo "📋 Copying application files..."
cp -r * /var/www/aptsrus-ai/

# Set up Python virtual environment
echo "🌐 Setting up Python virtual environment..."
cd /var/www/aptsrus-ai
python3 -m venv venv
source venv/bin/activate

# Install Python dependencies
echo "📦 Installing Python dependencies..."
pip install -r requirements.txt

# Configure environment variables
echo "🔐 Setting up environment variables..."
cp env_template.txt .env
# EDIT .env file with your actual credentials before continuing!

# Create applications directory
mkdir -p applications

# Set up Gunicorn configuration
echo "⚙️ Creating Gunicorn configuration..."
cat > gunicorn.conf.py << EOF
bind = "127.0.0.1:8000"
workers = 3
worker_class = "sync"
worker_connections = 1000
timeout = 30
keepalive = 2
user = "$USER"
group = "$USER"
tmp_upload_dir = None
EOF

# Set up Supervisor configuration
echo "👮 Setting up Supervisor for process management..."
sudo tee /etc/supervisor/conf.d/aptsrus-ai.conf > /dev/null << EOF
[program:aptsrus-ai]
directory=/var/www/aptsrus-ai
command=/var/www/aptsrus-ai/venv/bin/gunicorn --config gunicorn.conf.py app:app
autostart=true
autorestart=true
redirect_stderr=true
stdout_logfile=/var/log/aptsrus-ai.log
stderr_logfile=/var/log/aptsrus-ai-error.log
environment=PATH="/var/www/aptsrus-ai/venv/bin"
user=$USER
EOF

# Set up Nginx configuration
echo "🌐 Setting up Nginx reverse proxy..."
sudo tee /etc/nginx/sites-available/aptsrus-ai > /dev/null << EOF
server {
    listen 80;
    server_name aptsr-us.com www.aptsr-us.com;

    location = /favicon.ico { access_log off; log_not_found off; }

    location / {
        include proxy_params;
        proxy_pass http://127.0.0.1:8000;
        proxy_set_header X-Forwarded-For \$proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto \$scheme;
        proxy_set_header X-Real-IP \$remote_addr;
        proxy_set_header Host \$http_host;
    }

    location /static {
        alias /var/www/aptsrus-ai/static;
        expires 1y;
        add_header Cache-Control "public, immutable";
    }
}
EOF

# Enable Nginx site
sudo ln -sf /etc/nginx/sites-available/aptsrus-ai /etc/nginx/sites-enabled/
sudo rm -f /etc/nginx/sites-enabled/default

# Test configurations
echo "🧪 Testing configurations..."
sudo nginx -t
sudo supervisorctl reread
sudo supervisorctl update

# Start services
echo "▶️ Starting services..."
sudo systemctl restart nginx
sudo supervisorctl start aptsrus-ai

# Check if services are running
echo "🔍 Checking service status..."
sudo systemctl status nginx --no-pager
sudo supervisorctl status aptsrus-ai

echo "✅ Deployment completed!"
echo "🌐 Your application should be available at https://aptsr-us.com"
echo ""
echo "📋 Next steps:"
echo "1. Edit /var/www/aptsrus-ai/.env with your actual API keys and email settings"
echo "2. Test the application at https://aptsr-us.com"
echo "3. Set up SSL certificate with Let's Encrypt if needed"
echo ""
echo "🔧 Useful commands:"
echo "sudo supervisorctl restart aptsrus-ai  # Restart application"
echo "sudo systemctl reload nginx          # Reload nginx config"
echo "tail -f /var/log/aptsrus-ai.log      # View application logs"
