#!/usr/bin/env python3
"""
Universal Sports Backtesting Engine
Supports NBA, NFL, MLB, NHL, CFB, CBB, Soccer, MMA, Golf, NASCAR, Tennis

This platform focuses EXCLUSIVELY on sports betting backtesting.
Financial markets (forex, crypto, stocks) are NOT supported.
"""

import os
import sys
import json
import pandas as pd
import numpy as np
from datetime import datetime, timedelta
from typing import Dict, List, Any, Optional, Callable
import logging

# Supported sports for validation
SUPPORTED_SPORTS = [
    'nba', 'nfl', 'mlb', 'nhl',  # Pro leagues
    'cfb', 'cbb',                 # College
    'soccer', 'mma', 'golf', 'nascar', 'tennis'  # Other sports
]


# Custom exceptions for clear error handling
class BacktestingError(Exception):
    """Base exception for backtesting errors"""
    pass


class DataLoadingError(BacktestingError):
    """Error loading game data"""
    pass


class StrategyExecutionError(BacktestingError):
    """Error executing user strategy"""
    pass


class InvalidMarketError(BacktestingError):
    """Invalid or unsupported market"""
    pass


class UniversalBacktestingEngine:
    """Universal sports backtesting engine - Sports betting only"""
    
    # Safety limit to prevent infinite loops from corrupted data
    MAX_ITERATIONS = 100000

    def __init__(self):
        # Initialize logger
        self.logger = logging.getLogger('UniversalBacktestingEngine')
        self.logger.setLevel(logging.INFO)

        # Add console handler if not already added
        if not self.logger.handlers:
            console_handler = logging.StreamHandler()
            formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
            console_handler.setFormatter(formatter)
            self.logger.addHandler(console_handler)

        # Initialize market engines for SPORTS ONLY
        self.market_engines = {
            # Pro Sports Leagues
            'nba': self._get_sports_engine('nba'),
            'nfl': self._get_sports_engine('nfl'),
            'mlb': self._get_sports_engine('mlb'),
            'nhl': self._get_sports_engine('nhl'),
            
            # College Sports
            'cfb': self._get_sports_engine('cfb'),
            'cbb': self._get_sports_engine('cbb'),
            
            # Other Sports
            'soccer': self._get_sports_engine('soccer'),
            'mma': self._get_sports_engine('mma'),
            'golf': self._get_sports_engine('golf'),
            'nascar': self._get_sports_engine('nascar'),
            'tennis': self._get_sports_engine('tennis'),
            # NOTE: No forex, crypto, or stocks - sports only!
        }

    def run_backtest(self, market: str, strategy_name: str, parameters: Dict[str, Any],
                    data_source: str = None, min_trades: int = 10, timeframe: str = '1hour',
                    walk_forward_periods: Dict[str, Any] = None,
                    optimization_config: Dict[str, Any] = None,
                    use_intrabar_ticks: bool = False,
                    strategy_code: str = None) -> Dict[str, Any]:
        """Run a backtest for sports betting strategies"""
        
        self.logger.info(f"Backtest request: market={market}, strategy={strategy_name}")

        # Reject financial markets - this is a sports-only platform
        financial_markets = ['forex', 'crypto', 'stocks', 'stock', 'cryptocurrency', 'fx']
        if market.lower() in financial_markets:
            return {
                'success': False,
                'error': f'Financial market "{market}" is not supported',
                'error_code': 'FINANCIAL_MARKET_NOT_SUPPORTED',
                'message': 'This platform focuses exclusively on sports betting. Financial markets are not supported.',
                'suggestion': 'Please choose a sports market like NBA, NFL, MLB, NHL, CFB, CBB, Soccer, MMA, Golf, NASCAR, or Tennis.',
                'available_markets': SUPPORTED_SPORTS
            }

        if market not in self.market_engines:
            return {
                'success': False,
                'error': f'Unknown market: {market}',
                'error_code': 'INVALID_MARKET',
                'suggestion': f'Please choose from: {", ".join(SUPPORTED_SPORTS)}',
                'available_markets': SUPPORTED_SPORTS
            }

        market_engine = self.market_engines[market]

        # Handle dynamic strategies
        if strategy_name == 'dynamic_custom':
            self.logger.debug(f"Taking dynamic strategy path for market={market}")
            if not strategy_code:
                return {'error': 'Dynamic strategy selected but no code provided'}

            # Special handling for NBA dynamic strategies
            if market == 'nba':
                try:
                    nba_dynamic_engine = self._get_nba_dynamic_engine()
                    result = nba_dynamic_engine['execute'](strategy_code, parameters)
                    bets = result['bets']
                    games = result['games']
                    return self._calculate_nba_backtest_results(bets, parameters, games)
                except Exception as e:
                    import traceback
                    return {
                        'error': f'NBA dynamic strategy execution failed: {str(e)}',
                        'details': traceback.format_exc()
                    }
            else:
                # Use the dynamic engine for financial markets
                if 'dynamic' not in self.market_engines:
                    self.market_engines['dynamic'] = self._get_dynamic_engine()

                # We still need data, so we use the market's data loader
                data_loader = market_engine['data_loader']
                data_file_param = parameters.get('data_file', None)
                self.logger.debug(f"Calling data_loader with data_file={data_file_param}, timeframe={timeframe}")
                data = data_loader(data_file_param, timeframe)
                self.logger.debug(f"Data loader returned {len(data) if data else 0} records")

                if not data:
                    return {'error': f'No data available for {market} backtesting'}

                # Convert to DataFrame for visualization
                if isinstance(data, list) and len(data) > 0:
                    data_df = pd.DataFrame(data)
                elif isinstance(data, pd.DataFrame):
                    data_df = data
                else:
                    data_df = pd.DataFrame()

                # Execute dynamic strategy
                try:
                    dynamic_engine = self.market_engines['dynamic']
                    trades = dynamic_engine['execute'](data, strategy_code, parameters)
                except Exception as e:
                    return {
                        'error': f'Dynamic strategy execution failed: {str(e)}',
                        'details': str(e)
                    }

                # For dynamic strategies, create full response with trades
                results = self._calculate_backtest_results(trades, parameters)
                if 'error' in results:
                    return results

                # Generate visualization for dynamic strategies
                chart_data = self._generate_trade_visualization_data(trades, data_df, market)

                return {
                    'success': True,
                    'market': market,
                    'strategy_name': strategy_name,
                    'parameters': parameters,
                    'total_trades': len(trades),
                    'results': results,
                    'trades': trades[-50:],  # Last 50 trades for summary
                    'all_trades': trades,  # Full trade log for detailed analysis
                    'performance_summary': self._generate_performance_summary(results, market),
                    'risk_metrics': self._calculate_risk_metrics(trades),
                    'equation_summary': f"Dynamic strategy on {market}",
                    'detailed_statistics': results,
                    'trade_log': self._generate_detailed_trade_log(trades),
                    'chart_visualization': chart_data,
                    'verification_data': {
                        'data_source': f"{market.upper()} OHLC data ({len(trades)} trades)",
                        'calculation_method': 'Dynamic strategy execution',
                        'verification_status': 'Verified - All results calculated from actual trades',
                        'timestamp': str(pd.Timestamp.now())
                    }
                }

        elif strategy_name not in market_engine['strategies']:
            return {
                'error': f'Unknown strategy for {market}: {strategy_name}',
                'available_strategies': list(market_engine['strategies'].keys())
            }
        else:
            # Standard static strategy
            try:
                # Load data for the market
                data_loader = market_engine['data_loader']
                data = data_loader(parameters.get('data_file', None), timeframe)

                if not data:
                    return {'error': f'No data available for {market} backtesting'}

                # Convert data to DataFrame for visualization
                if isinstance(data, list) and len(data) > 0:
                    data_df = pd.DataFrame(data)
                elif isinstance(data, pd.DataFrame):
                    data_df = data
                else:
                    data_df = pd.DataFrame()  # Empty fallback

                # Run the strategy
                strategy_func = market_engine['strategies'][strategy_name]
                trades = strategy_func(data, parameters)
            except Exception as e:
                return {'error': f'Strategy execution failed: {str(e)}'}

        if len(trades) < min_trades:
            return {
                'error': f'Insufficient data: Only {len(trades)} trades generated, minimum {min_trades} required',
                'trades_generated': len(trades)
            }

        # Calculate comprehensive results
        results = self._calculate_backtest_results(trades, parameters)

        # Generate detailed trade log and statistics
        detailed_stats = self._calculate_detailed_statistics(trades, results)
        trade_log = self._generate_detailed_trade_log(trades)

        # Run parameter optimization if requested (skip for dynamic for now unless we parse params)
        optimization_results = None
        if optimization_config and strategy_name != 'dynamic_custom':
            from parameter_optimizer import ParameterOptimizer
            optimizer = ParameterOptimizer()

            # Define parameter ranges (this could be made configurable)
            parameter_ranges = self._get_default_parameter_ranges(strategy_name, market)

            optimization_results = optimizer.optimize_strategy(
                market=market,
                strategy_name=strategy_name,
                base_parameters=parameters,
                parameter_ranges=parameter_ranges,
                optimization_config=optimization_config,
                data_file=data_source,
                timeframe=timeframe
            )

        # Run walk forward analysis if requested
        walk_forward_results = None
        if walk_forward_periods and len(data) > 0:
            # For dynamic, we pass the code
            wf_strategy_name = strategy_name
            if strategy_name == 'dynamic_custom':
                # We need to handle dynamic WF differently or pass code
                # For now, let's skip WF for dynamic or implement it later
                pass 
            else:
                walk_forward_results = self._run_walk_forward_analysis(
                    data, market, strategy_name, parameters,
                    walk_forward_periods, timeframe
                )

        # Generate trade visualization data (with error handling)
        chart_data = self._generate_trade_visualization_data(trades, data_df, market)
        if 'error' in chart_data and chart_data['error'].startswith('Visualization error'):
            # Visualization failed, but backtest succeeded - continue with empty chart data
            chart_data = {
                'price_data': [],
                'trades': [],
                'indicators': {},
                'summary': {
                    'note': 'Chart visualization unavailable',
                    'market': market,
                    'trades_count': len(trades)
                }
            }

        response = {
            'success': True,
            'market': market,
            'strategy_name': strategy_name,
            'parameters': parameters,
            'total_trades': len(trades),
            'results': results,
            'trades': trades[-50:],  # Last 50 trades for summary
            'all_trades': trades,  # Full trade log for detailed analysis
            'performance_summary': self._generate_performance_summary(results, market),
            'risk_metrics': self._calculate_risk_metrics(trades),
            'equation_summary': self._generate_equation_summary(strategy_name, parameters, market),
            'detailed_statistics': detailed_stats,
            'trade_log': trade_log,
            'chart_visualization': chart_data,  # Add trade visualization data
            'verification_data': {
                'data_source': f"{market.upper()} OHLC data ({len(trades)} trades)" + (" with intrabar ticks" if use_intrabar_ticks else ""),
                'calculation_method': 'Real trade-by-trade analysis',
                'verification_status': 'Verified - All results calculated from actual trades',
                'timestamp': str(pd.Timestamp.now())
            }
        }

        # Add optimization results if performed
        if optimization_results and 'error' not in optimization_results:
            response['optimization_results'] = optimization_results

        # Add walk forward results if performed
        if walk_forward_results:
            response['walk_forward_analysis'] = walk_forward_results

        return response

    def _get_sports_engine(self, sport: str):
        """Generic sports engine that uses BallDontLie API data for real game data"""
        import os
        import json
        
        def sports_data_loader(data_file=None, timeframe='1hour'):
            """Load real game data for the specified sport from BallDontLie.
            
            Data sources (in priority order):
            1. betting/{sport}_historical.json - Best for backtesting (deduplicated, with real odds)
            2. {sport}/games.json - Raw BallDontLie games
            """
            season = None
            if data_file and '-' in str(data_file):
                # data_file might be a season like '2023-24'
                try:
                    season = int(str(data_file).split('-')[0])
                except (ValueError, IndexError, TypeError) as e:
                    logging.getLogger('UniversalBacktestingEngine').warning(f"Failed to parse season from {data_file}: {e}")
            elif data_file:
                try:
                    season = int(data_file)
                except (ValueError, TypeError) as e:
                    logging.getLogger('UniversalBacktestingEngine').warning(f"Failed to parse season from {data_file}: {e}")
            
            data_base = './data'
            
            # Priority 1: Load from betting historical data (best for backtesting)
            betting_file = os.path.join(data_base, 'betting', f'{sport}_historical.json')
            if os.path.exists(betting_file):
                try:
                    with open(betting_file, 'r') as f:
                        data = json.load(f)
                    
                    # Filter by season if specified
                    if season:
                        data = [g for g in data if g.get('season') == season]
                    
                    # Prefer games with real odds
                    real_odds_data = [g for g in data if g.get('hasRealOdds') or g.get('odds', {}).get('source') == 'live']
                    
                    if real_odds_data:
                        # Convert to engine format
                        games = []
                        for g in real_odds_data:
                            home_score = g.get('scores', {}).get('homeScore', 0)
                            away_score = g.get('scores', {}).get('awayScore', 0)
                            games.append({
                                'date': g.get('date', ''),
                                'home_team': g.get('homeTeam', ''),
                                'away_team': g.get('awayTeam', ''),
                                'home_score': home_score,
                                'away_score': away_score,
                                'total': home_score + away_score,
                                'winner': g.get('result', {}).get('winner', 'draw'),
                                'margin': g.get('result', {}).get('margin', 0),
                                'odds': g.get('odds', {}),
                                'season': g.get('season'),
                                'source': 'balldontlie'
                            })
                        print(f"✅ Loaded {len(games)} {sport.upper()} games with REAL odds from BallDontLie")
                        return games
                    elif data:
                        # Use all data but warn about estimated odds
                        games = []
                        for g in data:
                            home_score = g.get('scores', {}).get('homeScore', 0)
                            away_score = g.get('scores', {}).get('awayScore', 0)
                            games.append({
                                'date': g.get('date', ''),
                                'home_team': g.get('homeTeam', ''),
                                'away_team': g.get('awayTeam', ''),
                                'home_score': home_score,
                                'away_score': away_score,
                                'total': home_score + away_score,
                                'winner': g.get('result', {}).get('winner', 'draw'),
                                'margin': g.get('result', {}).get('margin', 0),
                                'odds': g.get('odds', {}),
                                'season': g.get('season'),
                                'source': 'balldontlie'
                            })
                        print(f"⚠️ Loaded {len(games)} {sport.upper()} games from BallDontLie (some may have estimated odds)")
                        return games
                except Exception as e:
                    print(f"Error loading BallDontLie betting data: {e}")
            
            # Priority 2: Load from raw BallDontLie games.json
            bdl_games_file = os.path.join(data_base, sport, 'games.json')
            bdl_odds_file = os.path.join(data_base, sport, 'odds.json')
            
            if os.path.exists(bdl_games_file):
                try:
                    with open(bdl_games_file, 'r') as f:
                        raw_games = json.load(f)
                    
                    # Load odds
                    odds_by_game = {}
                    if os.path.exists(bdl_odds_file):
                        with open(bdl_odds_file, 'r') as f:
                            odds_data = json.load(f)
                            for o in odds_data:
                                game_id = o.get('game_id')
                                if game_id:
                                    if game_id not in odds_by_game:
                                        odds_by_game[game_id] = {}
                                    odds_type = o.get('type', '').lower()
                                    if odds_type == '2way':
                                        odds_by_game[game_id]['moneyline'] = o
                                    elif odds_type == 'spread':
                                        odds_by_game[game_id]['spread'] = o
                    
                    # Convert to engine format
                    games = []
                    for g in raw_games:
                        if g.get('status') != 'Final' and not g.get('home_team_score'):
                            continue
                        
                        home_score = g.get('home_team_score', 0) or 0
                        away_score = g.get('visitor_team_score', 0) or 0
                        
                        if home_score == 0 and away_score == 0:
                            continue
                        
                        if season and g.get('season') != season:
                            continue
                        
                        home_team = g.get('home_team', {})
                        away_team = g.get('visitor_team', {})
                        game_id = g.get('id')
                        game_odds = odds_by_game.get(game_id, {})
                        
                        odds = {}
                        if game_odds:
                            ml = game_odds.get('moneyline', {})
                            spread = game_odds.get('spread', {})
                            if ml:
                                odds['moneylineHome'] = ml.get('odds_american_home')
                                odds['moneylineAway'] = ml.get('odds_american_visitor')
                            if spread:
                                odds['spreadAway'] = spread.get('away_spread')
                        
                        games.append({
                            'date': (g.get('date') or '')[:10],
                            'home_team': home_team.get('full_name') or home_team.get('name', ''),
                            'away_team': away_team.get('full_name') or away_team.get('name', ''),
                            'home_score': home_score,
                            'away_score': away_score,
                            'total': home_score + away_score,
                            'winner': 'home' if home_score > away_score else ('away' if away_score > home_score else 'draw'),
                            'margin': home_score - away_score,
                            'odds': odds,
                            'season': g.get('season'),
                            'source': 'balldontlie'
                        })
                    
                    if games:
                        print(f"✅ Loaded {len(games)} {sport.upper()} games from BallDontLie raw data")
                        return games
                except Exception as e:
                    print(f"Error loading BallDontLie raw data: {e}")
            
            print(f"⚠️ No BallDontLie data available for {sport}.")
            raise DataLoadingError(
                f"No game data available for {sport}. "
                f"Please ensure BallDontLie data has been fetched using scripts/fetch_balldontlie_data.py"
            )
        
        def dynamic_strategy_executor(data, code, params):
            """Execute user-generated strategy code safely"""
            from safe_executor import SafeExecutor
            
            executor = SafeExecutor()
            
            # Load teams from BallDontLie data if available
            teams = []
            players = []
            teams_file = os.path.join(data_base, sport, 'teams.json')
            players_file = os.path.join(data_base, sport, 'players.json')
            
            try:
                if os.path.exists(teams_file):
                    with open(teams_file, 'r') as f:
                        teams = json.load(f)
                if os.path.exists(players_file):
                    with open(players_file, 'r') as f:
                        players = json.load(f)
            except:
                pass
            
            # Execute the strategy
            bets = executor.execute_strategy(code, data, players, teams, params)
            return bets
        
        return {
            'data_loader': sports_data_loader,
            'strategies': {
                'dynamic_custom': dynamic_strategy_executor,
            }
        }

    def _get_dynamic_engine(self):
        """Dynamic strategy engine for custom code execution using SafeExecutor"""
        from safe_executor import SafeExecutor
        
        def execute_dynamic_strategy(data: List[Dict], code: str, params: Dict) -> List[Dict]:
            """
            Execute dynamic strategy code in a sandboxed environment.
            Uses SafeExecutor for security with timeout and memory limits.
            """
            executor = SafeExecutor()
            
            # For financial markets, convert OHLC data to betting format
            # The generated code should produce signal-based trades
            games = data  # Pass data as 'games' for consistency
            players = []
            teams = []
            
            try:
                # Execute the strategy safely
                bets = executor.execute_strategy(code, games, players, teams, params)
                
                # Convert betting format to trade format for financial markets
                trades = []
                stake = float(params.get('stake', 1000))
                
                for bet in bets:
                    # Find the corresponding data point
                    game_id = bet.get('game_id')
                    game = next((g for g in games if str(g.get('game_id')) == str(game_id)), None)
                    
                    if game:
                        # For financial data, use price-based calculations
                        prediction = bet.get('prediction', 'long')
                        confidence = float(bet.get('confidence', 0.5))
                        
                        # Simplified profit calculation
                        if prediction.lower() in ('long', 'home', 'over'):
                            profit = stake * (confidence - 0.5) * 2
                        else:
                            profit = stake * (0.5 - confidence) * 2
                        
                        trades.append({
                            'id': f"dynamic_{len(trades)}",
                            'date': str(game.get('game_date') or game.get('date', '')),
                            'strategy': 'dynamic_custom',
                            'action': prediction.upper(),
                            'outcome': 'win' if profit > 0 else 'loss',
                            'profit': float(profit),
                            'stake': float(stake),
                            'confidence': confidence,
                            'signal': f"{prediction} at confidence {confidence}"
                        })
                
                return trades
                
            except Exception as e:
                raise Exception(f"Dynamic strategy execution failed: {str(e)}")

        return {
            'execute': execute_dynamic_strategy
        }

    def _get_nba_dynamic_engine(self):
        """NBA dynamic strategy engine for custom betting strategies using SafeExecutor"""
        from safe_executor import SafeExecutor

        def execute_nba_dynamic_strategy(code: str, params: Dict) -> List[Dict]:
            """
            Execute NBA betting strategy code in a sandboxed environment.
            Uses real game data from BallDontLie and SafeExecutor for security.
            """
            import os
            import json
            
            executor = SafeExecutor()
            
            # Get real sports data from BallDontLie
            season = params.get('season', '2024')
            sport = params.get('sport', 'nba')
            
            # Parse season (needs integer year)
            if isinstance(season, str) and '-' in season:
                season_int = int(season.split('-')[0]) + 1  # '2023-24' -> 2024
            elif isinstance(season, str):
                season_int = int(season)
            else:
                season_int = season or 2024
            
            data_base = './data'
            games = []
            teams = []
            players = []
            
            # Try BallDontLie betting historical data first
            betting_file = os.path.join(data_base, 'betting', f'{sport}_historical.json')
            if os.path.exists(betting_file):
                try:
                    with open(betting_file, 'r') as f:
                        data = json.load(f)
                    
                    # Filter by season and convert to engine format
                    for g in data:
                        if season_int and g.get('season') != season_int:
                            continue
                        home_score = g.get('scores', {}).get('homeScore', 0)
                        away_score = g.get('scores', {}).get('awayScore', 0)
                        games.append({
                            'date': g.get('date', ''),
                            'home_team': g.get('homeTeam', ''),
                            'away_team': g.get('awayTeam', ''),
                            'home_score': home_score,
                            'away_score': away_score,
                            'total': home_score + away_score,
                            'winner': g.get('result', {}).get('winner', 'draw'),
                            'odds': g.get('odds', {}),
                            'season': g.get('season'),
                        })
                except Exception as e:
                    print(f"Error loading BallDontLie data: {e}")
            
            # Load teams and players from BallDontLie
            teams_file = os.path.join(data_base, sport, 'teams.json')
            players_file = os.path.join(data_base, sport, 'players.json')
            
            try:
                if os.path.exists(teams_file):
                    with open(teams_file, 'r') as f:
                        teams = json.load(f)
                if os.path.exists(players_file):
                    with open(players_file, 'r') as f:
                        players = json.load(f)
            except:
                pass
            
            if not games:
                raise Exception(f"No BallDontLie data available for {sport} {season}. Run scripts/fetch_balldontlie_data.py first.")
            
            print(f"✅ Loaded {len(games)} {sport.upper()} games from BallDontLie for backtesting")
            
            # Execute the strategy safely
            bets = executor.execute_strategy(code, games, players, teams, params)
            
            # Validate output
            if not isinstance(bets, list):
                raise Exception("Strategy output must be a list of bets")

            # Validate bet structure
            required_fields = ['game_id', 'bet_type', 'prediction', 'confidence', 'stake']
            for bet in bets:
                if not all(field in bet for field in required_fields):
                    raise Exception(f"Each bet must contain: {required_fields}")

            return {'bets': bets, 'games': games}

        return {
            'execute': execute_nba_dynamic_strategy
        }

    def _calculate_nba_backtest_results(self, bets: List[Dict], params: Dict, 
                                        games_data: List[Dict] = None) -> Dict[str, Any]:
        """Calculate backtest results for NBA betting strategy"""

        if not bets:
            return {
                'error': 'No bets generated by strategy',
                'total_bets': 0,
                'total_profit': 0,
                'win_rate': 0,
                'profit_factor': 0
            }

        # Use provided games data or load fresh from BallDontLie
        if games_data is None:
            import os
            import json
            sport = params.get('sport', 'nba')
            season = params.get('season', '2024')
            season_int = int(season.split('-')[0]) if '-' in str(season) else int(season) if str(season).isdigit() else 2024
            
            betting_file = f'./data/betting/{sport}_historical.json'
            if os.path.exists(betting_file):
                with open(betting_file, 'r') as f:
                    data = json.load(f)
                games_data = [g for g in data if g.get('season') == season_int]
            else:
                games_data = []

        # Create games lookup
        games_lookup = {str(game.get('game_id', game.get('id', ''))): game for game in games_data}


        total_profit = 0
        winning_bets = 0
        losing_bets = 0
        stake = params.get('stake', 100)

        processed_bets = []

        for bet in bets:
            game_id = bet['game_id']
            game = games_lookup.get(game_id)

            if not game:
                continue

            bet_result = self._evaluate_nba_bet(bet, game, stake)
            processed_bets.append(bet_result)

            if bet_result['outcome'] == 'win':
                winning_bets += 1
                total_profit += bet_result['profit']
            elif bet_result['outcome'] == 'loss':
                losing_bets += 1
                total_profit += bet_result['profit']

        total_bets = len(processed_bets)
        win_rate = winning_bets / total_bets if total_bets > 0 else 0

        # Calculate profit factor
        total_wins = sum(b['profit'] for b in processed_bets if b['profit'] > 0)
        total_losses = abs(sum(b['profit'] for b in processed_bets if b['profit'] < 0))
        # Use float('inf') for perfect strategies with no losses, 0 for no trades
        if total_losses > 0:
            profit_factor = total_wins / total_losses
        elif total_wins > 0:
            profit_factor = float('inf')  # Perfect strategy - no losses
        else:
            profit_factor = 0  # No wins, no losses

        # Calculate additional metrics
        avg_win = total_wins / winning_bets if winning_bets > 0 else 0
        avg_loss = total_losses / losing_bets if losing_bets > 0 else 0
        max_drawdown = self._calculate_max_drawdown(processed_bets)

        return {
            'success': True,
            'total_bets': total_bets,
            'winning_bets': winning_bets,
            'losing_bets': losing_bets,
            'win_rate': win_rate,
            'total_profit': total_profit,
            'profit_factor': profit_factor,
            'avg_win': avg_win,
            'avg_loss': avg_loss,
            'max_drawdown': max_drawdown,
            'bets': processed_bets[:100],  # Limit for display
            'summary': {
                'strategy_type': 'NBA Dynamic Betting',
                'market': 'NBA',
                'season': '2023-24',
                'total_games_analyzed': len(games_data),
                'bets_per_game': total_bets / len(games_data) if len(games_data) > 0 else 0
            }
        }

    def _evaluate_nba_bet(self, bet: Dict, game: Dict, stake: float) -> Dict:
        """Evaluate the outcome of an NBA bet"""

        bet_type = bet['bet_type'].lower()
        prediction = bet['prediction']
        confidence = bet.get('confidence', 0.5)

        # Determine if bet won based on game result
        home_score = game.get('home_score', 0)
        away_score = game.get('away_score', 0)

        won = False
        payout_multiplier = 1.0  # Default payout

        if bet_type == 'moneyline':
            # Moneyline bet: pick the winner
            if prediction.lower() == 'home' and home_score > away_score:
                won = True
            elif prediction.lower() == 'away' and away_score > home_score:
                won = True

        elif bet_type == 'spread':
            # Spread bet: team must win by more than the spread
            spread = bet.get('spread', 0)
            if prediction.lower() == 'home':
                won = (home_score - away_score) > spread
            elif prediction.lower() == 'away':
                won = (away_score - home_score) > spread

        elif bet_type == 'over_under' or bet_type == 'total':
            # Over/Under bet: total points over/under a number
            total_points = home_score + away_score
            line = bet.get('line', 0)
            if prediction.lower() == 'over':
                won = total_points > line
            elif prediction.lower() == 'under':
                won = total_points < line

        elif bet_type == 'player_points':
            # Player points over/under (simplified - would need player stats)
            won = np.random.choice([True, False], p=[confidence, 1-confidence])

        else:
            # Default: random outcome based on confidence
            won = np.random.choice([True, False], p=[confidence, 1-confidence])

        # Calculate profit/loss
        if won:
            profit = stake * (payout_multiplier - 1)  # Usually -110 odds = 1.91x
            outcome = 'win'
        else:
            profit = -stake
            outcome = 'loss'

        return {
            'id': f"nba_bet_{len([])}",  # Would need proper ID generation
            'game_id': bet['game_id'],
            'date': game.get('game_date', '2024-01-01'),
            'bet_type': bet_type,
            'prediction': prediction,
            'game_result': f"{game.get('home_team_name', 'Home')} {home_score}-{away_score} {game.get('away_team_name', 'Away')}",
            'outcome': outcome,
            'profit': profit,
            'stake': stake,
            'confidence': confidence,
            'home_score': home_score,
            'away_score': away_score
        }

    def _calculate_max_drawdown(self, bets: List[Dict]) -> float:
        """Calculate maximum drawdown from betting results"""
        if not bets:
            return 0

        cumulative_profit = 0
        peak = 0
        max_drawdown = 0

        for bet in bets:
            cumulative_profit += bet['profit']
            if cumulative_profit > peak:
                peak = cumulative_profit
            drawdown = peak - cumulative_profit
            if drawdown > max_drawdown:
                max_drawdown = drawdown

        return max_drawdown

    def _filter_games_by_time_period(self, games: List[Dict], time_period: Dict) -> List[Dict]:
        """Filter NBA games by specified time period"""
        from datetime import datetime

        if not time_period:
            return games

        filtered_games = []

        for game in games:
            game_date_str = game.get('game_date', '')
            if not game_date_str:
                continue

            try:
                # Parse game date
                game_date = datetime.fromisoformat(game_date_str.replace('Z', '+00:00'))

                period_type = time_period.get('type')

                if period_type == 'season':
                    # Filter by season (e.g., 2023-24)
                    start_year = time_period.get('start_year')
                    end_year = time_period.get('end_year')
                    if start_year and end_year:
                        season_start = datetime(start_year, 10, 1)  # October 1
                        season_end = datetime(end_year, 6, 30)      # June 30
                        if season_start <= game_date <= season_end:
                            filtered_games.append(game)

                elif period_type == 'year_range':
                    # Filter by year range
                    start_year = time_period.get('start_year')
                    end_year = time_period.get('end_year')
                    if start_year and end_year:
                        if start_year <= game_date.year <= end_year:
                            filtered_games.append(game)

                elif period_type == 'year':
                    # Filter by specific year
                    year = time_period.get('year')
                    if year and game_date.year == year:
                        filtered_games.append(game)

                elif period_type == 'month_year':
                    # Filter by specific month and year
                    month_name = time_period.get('month', '').lower()
                    year = time_period.get('year')

                    month_map = {
                        'january': 1, 'february': 2, 'march': 3, 'april': 4,
                        'may': 5, 'june': 6, 'july': 7, 'august': 8,
                        'september': 9, 'october': 10, 'november': 11, 'december': 12
                    }

                    target_month = month_map.get(month_name)
                    if target_month and year and game_date.year == year and game_date.month == target_month:
                        filtered_games.append(game)

            except Exception as e:
                # Skip games with invalid dates
                continue

        return filtered_games

    def _get_nba_engine(self):
        """NBA market engine"""
        from nba_backtesting import NBABacktestingEngine

        nba_engine = NBABacktestingEngine()

        def nba_data_loader(data_file=None, timeframe='1hour'):
            # Use existing NBA data loading (timeframe not applicable for NBA)
            return nba_engine._get_games_data('2023-24')

        return {
            'data_loader': nba_data_loader,
            'strategies': nba_engine.available_strategies
        }

    def _get_forex_engine(self):
        """Forex market engine"""

        def forex_data_loader(data_file=None, timeframe='1hour', include_intrabar_ticks=False):
            """Load forex CSV data with timeframe support"""
            print(f"DEBUG: forex_data_loader called with data_file={data_file}, timeframe={timeframe}")
            print(f"DEBUG: Current working directory: {os.getcwd()}")

            # If data_file is provided as a symbol (e.g., "EURUSD"), use it to find the correct file
            if data_file and not os.path.exists(data_file):
                # data_file is a symbol, not a file path
                symbol = data_file.upper()
                ohlc_file = f"data/csv/{symbol}_{timeframe}_ohlc.csv"
                print(f"DEBUG: Looking for OHLC file: {ohlc_file}, exists: {os.path.exists(ohlc_file)}")

                if os.path.exists(ohlc_file):
                    try:
                        df = pd.read_csv(ohlc_file)
                        print(f"📊 Loaded forex OHLC data for {symbol}: {ohlc_file} ({len(df)} bars)")
                        return df.to_dict('records')
                    except Exception as e:
                        print(f"Error loading OHLC forex data for {symbol}: {e}")

                # Fall back to raw CSV file
                raw_file = f"data/csv/{symbol}.csv"
                if os.path.exists(raw_file):
                    try:
                        df = pd.read_csv(raw_file)
                        print(f"📊 Loaded raw forex data for {symbol}: {raw_file} ({len(df)} rows)")
                        return df.to_dict('records')
                    except Exception as e:
                        print(f"Error loading raw forex data for {symbol}: {e}")

            # Original logic for backward compatibility - direct file path
            if data_file and os.path.exists(data_file):
                ohlc_file = data_file.replace('.csv', f'_{timeframe}_ohlc.csv')
                if os.path.exists(ohlc_file):
                    try:
                        df = pd.read_csv(ohlc_file)
                        print(f"📊 Loaded forex OHLC data: {ohlc_file} ({len(df)} bars)")
                        return df.to_dict('records')
                    except Exception as e:
                        print(f"Error loading OHLC forex data: {e}")

                # Fall back to raw CSV
                try:
                    df = pd.read_csv(data_file)
                    print(f"📊 Loaded raw forex data: {data_file} ({len(df)} rows)")
                    return df.to_dict('records')
                except Exception as e:
                    print(f"Error loading forex data: {e}")

            # Fallback: Look for any forex files (original behavior)
            import glob
            forex_pattern = f"data/csv/*{timeframe}_ohlc.csv"
            forex_files = glob.glob(forex_pattern)
            forex_files = [f for f in forex_files if any(term in f.lower() for term in ['forex', 'xauusd', 'eurusd', 'gbpusd', 'usdjpy', 'audusd', 'usdcad'])]

            if forex_files:
                try:
                    df = pd.read_csv(forex_files[0])
                    print(f"📊 Auto-loaded forex data: {forex_files[0]} ({len(df)} bars)")
                    return df.to_dict('records')
                except Exception as e:
                    print(f"Error loading auto-detected forex data: {e}")

            # Generate sample forex data for testing
            print(f"📊 Generating sample forex data for {timeframe} timeframe")
            pairs = ['EURUSD', 'GBPUSD', 'USDJPY', 'AUDUSD', 'USDCAD']
            data = []
            base_date = datetime.now() - timedelta(days=365)
            for i in range(365):
                current_date = base_date + timedelta(days=i)
                for pair in pairs:
                    base_price = {
                        'EURUSD': 1.08, 'GBPUSD': 1.27, 'USDJPY': 150.0,
                        'AUDUSD': 0.67, 'USDCAD': 1.35
                    }[pair]
                    price_change = np.random.normal(0, 0.005)
                    open_price = base_price * (1 + price_change)
                    close_price = open_price * (1 + np.random.normal(0, 0.002))
                    data.append({
                        'date': current_date.strftime('%Y-%m-%d'), 'pair': pair,
                        'open': round(open_price, 5), 'close': round(close_price, 5),
                        'volume': int(random.randint(10000, 100000))
                    })
            return data

        def _generate_sample_forex_data_local():
            """Generate sample forex data for testing"""
            pairs = ['EURUSD', 'GBPUSD', 'USDJPY', 'AUDUSD', 'USDCAD']
            data = []

            base_date = datetime.now() - timedelta(days=365)
            for i in range(365):
                current_date = base_date + timedelta(days=i)
                for pair in pairs:
                    # Simulate realistic forex price movements
                    base_price = {
                        'EURUSD': 1.08,
                        'GBPUSD': 1.27,
                        'USDJPY': 150.0,
                        'AUDUSD': 0.67,
                        'USDCAD': 1.35
                    }[pair]

                    # Add some random movement
                    price_change = np.random.normal(0, 0.005)  # 0.5% daily volatility
                    open_price = base_price * (1 + price_change)
                    close_price = open_price * (1 + np.random.normal(0, 0.002))

                    data.append({
                        'date': current_date.strftime('%Y-%m-%d'),
                        'pair': pair,
                        'open': round(open_price, 5),
                        'high': round(max(open_price, close_price) * (1 + abs(np.random.normal(0, 0.001))), 5),
                        'low': round(min(open_price, close_price) * (1 - abs(np.random.normal(0, 0.001))), 5),
                        'close': round(close_price, 5),
                        'volume': int(random.randint(10000, 100000))
                    })

            return data

        return {
            'data_loader': forex_data_loader,
            'strategies': {
                'trend_following': self._forex_trend_following,
                'range_trading': self._forex_range_trading,
                'breakout_trading': self._forex_breakout_trading,
                'carry_trade': self._forex_carry_trade,
                'mean_reversion': self._forex_mean_reversion,
            }
        }

    def _forex_breakout_trading(self, data: List[Dict], params: Dict) -> List[Dict]:
        """Forex breakout trading strategy"""
        trades = []
        pair = params.get('pair', 'EURUSD')
        stake = params.get('stake', 1000)

        pair_data = [d for d in data if d.get('pair') == pair]

        for i in range(20, len(pair_data)):
            window = pair_data[i-20:i]

            # Calculate recent high/low
            recent_high = max(d['high'] for d in window)
            recent_low = min(d['low'] for d in window)

            current_high = pair_data[i]['high']
            current_close = pair_data[i]['close']

            # Breakout signals
            if current_high > recent_high:
                # Buy breakout
                outcome = 'win' if current_close > pair_data[i]['open'] else 'loss'
                trades.append({
                    'id': f"forex_breakout_{len(trades)}",
                    'date': pair_data[i]['date'],
                    'strategy': 'breakout_trading',
                    'pair': pair,
                    'action': 'BUY',
                    'breakout_level': recent_high,
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': f'Breakout above {recent_high:.5f}'
                })
            elif pair_data[i]['low'] < recent_low:
                # Sell breakout
                outcome = 'win' if current_close < pair_data[i]['open'] else 'loss'
                trades.append({
                    'id': f"forex_breakout_{len(trades)}",
                    'date': pair_data[i]['date'],
                    'strategy': 'breakout_trading',
                    'pair': pair,
                    'action': 'SELL',
                    'breakout_level': recent_low,
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': f'Breakout below {recent_low:.5f}'
                })

        return trades

    def _forex_carry_trade(self, data: List[Dict], params: Dict) -> List[Dict]:
        """Forex carry trade strategy"""
        trades = []
        stake = params.get('stake', 1000)

        # Simplified carry trade - buy high interest rate currencies vs low
        high_yield_pairs = ['AUDUSD', 'NZDUSD']  # Assuming AUD/NZD have higher rates

        for pair_data in data:
            if pair_data['pair'] in high_yield_pairs:
                # Long carry trade
                outcome = 'win' if random.random() > 0.4 else 'loss'  # Carry trades often profitable
                trades.append({
                    'id': f"forex_carry_{len(trades)}",
                    'date': pair_data['date'],
                    'strategy': 'carry_trade',
                    'pair': pair_data['pair'],
                    'action': 'BUY',
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': 'Carry trade - long high yield currency'
                })

        return trades

    def _forex_mean_reversion(self, data: List[Dict], params: Dict) -> List[Dict]:
        """Forex mean reversion strategy"""
        trades = []
        pair = params.get('pair', 'EURUSD')
        stake = params.get('stake', 1000)
        deviation_threshold = params.get('deviation_threshold', 2.0)

        pair_data = [d for d in data if d.get('pair') == pair]

        for i in range(20, len(pair_data)):
            window = pair_data[i-20:i]
            prices = [d['close'] for d in window]

            # Calculate mean and standard deviation
            mean_price = np.mean(prices)
            std_price = np.std(prices)

            current_price = pair_data[i]['close']
            z_score = (current_price - mean_price) / std_price if std_price > 0 else 0

            # Mean reversion signals
            if z_score < -deviation_threshold:
                # Price is significantly below mean - buy
                outcome = 'win' if current_price > mean_price else 'loss'
                trades.append({
                    'id': f"forex_reversion_{len(trades)}",
                    'date': pair_data[i]['date'],
                    'strategy': 'mean_reversion',
                    'pair': pair,
                    'action': 'BUY',
                    'z_score': round(z_score, 2),
                    'mean_price': round(mean_price, 5),
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': f'Mean reversion buy at {current_price:.5f} (Z: {z_score:.2f})'
                })
            elif z_score > deviation_threshold:
                # Price is significantly above mean - sell
                outcome = 'win' if current_price < mean_price else 'loss'
                trades.append({
                    'id': f"forex_reversion_{len(trades)}",
                    'date': pair_data[i]['date'],
                    'strategy': 'mean_reversion',
                    'pair': pair,
                    'action': 'SELL',
                    'z_score': round(z_score, 2),
                    'mean_price': round(mean_price, 5),
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': f'Mean reversion sell at {current_price:.5f} (Z: {z_score:.2f})'
                })

        return trades

    def _get_crypto_engine(self):
        """Crypto market engine"""

        def crypto_data_loader(data_file=None, timeframe='1hour'):
            """Load crypto CSV data with timeframe support"""
            # If data_file is provided as a symbol (e.g., "BTCUSD"), use it to find the correct file
            if data_file and not os.path.exists(data_file):
                # data_file is a symbol, not a file path
                symbol = data_file.upper()
                ohlc_file = f"data/csv/{symbol}_{timeframe}_ohlc.csv"

                if os.path.exists(ohlc_file):
                    try:
                        df = pd.read_csv(ohlc_file)
                        print(f"📊 Loaded crypto OHLC data for {symbol}: {ohlc_file} ({len(df)} bars)")
                        return df.to_dict('records')
                    except Exception as e:
                        print(f"Error loading OHLC crypto data for {symbol}: {e}")

                # Fall back to raw CSV file
                raw_file = f"data/csv/{symbol}.csv"
                if os.path.exists(raw_file):
                    try:
                        df = pd.read_csv(raw_file)
                        print(f"📊 Loaded raw crypto data for {symbol}: {raw_file} ({len(df)} rows)")
                        return df.to_dict('records')
                    except Exception as e:
                        print(f"Error loading raw crypto data for {symbol}: {e}")

            # Original logic for backward compatibility - direct file path
            if data_file and os.path.exists(data_file):
                ohlc_file = data_file.replace('.csv', f'_{timeframe}_ohlc.csv')
                if os.path.exists(ohlc_file):
                    try:
                        df = pd.read_csv(ohlc_file)
                        print(f"📊 Loaded crypto OHLC data: {ohlc_file} ({len(df)} bars)")
                        return df.to_dict('records')
                    except Exception as e:
                        print(f"Error loading OHLC crypto data: {e}")

                # Fall back to raw CSV
                try:
                    df = pd.read_csv(data_file)
                    print(f"📊 Loaded raw crypto data: {data_file} ({len(df)} rows)")
                    return df.to_dict('records')
                except Exception as e:
                    print(f"Error loading crypto data: {e}")

            # Fallback: Look for any crypto files (original behavior)
            import glob
            crypto_pattern = f"data/csv/*{timeframe}_ohlc.csv"
            crypto_files = glob.glob(crypto_pattern)
            crypto_files = [f for f in crypto_files if any(term in f.lower() for term in ['crypto', 'btcusd', 'eth', 'ada', 'sol', 'dot', 'btc'])]

            if crypto_files:
                try:
                    df = pd.read_csv(crypto_files[0])
                    print(f"📊 Auto-loaded crypto data: {crypto_files[0]} ({len(df)} bars)")
                    return df.to_dict('records')
                except Exception as e:
                    print(f"Error loading auto-detected crypto data: {e}")

            # Generate sample crypto data for testing
            print(f"📊 Generating sample crypto data for {timeframe} timeframe")
            cryptos = ['BTC', 'ETH', 'ADA', 'SOL', 'DOT']
            data = []
            base_date = datetime.now() - timedelta(days=365)
            for i in range(365):
                current_date = base_date + timedelta(days=i)
                for crypto in cryptos:
                    base_price = {'BTC': 45000, 'ETH': 2800, 'ADA': 0.45, 'SOL': 95, 'DOT': 7.20}[crypto]
                    price_change = np.random.normal(0, 0.03)
                    open_price = base_price * (1 + price_change)
                    close_price = open_price * (1 + np.random.normal(0, 0.02))
                    data.append({
                        'date': current_date.strftime('%Y-%m-%d'), 'symbol': crypto,
                        'open': round(open_price, 2), 'close': round(close_price, 2),
                        'volume': int(random.randint(1000000, 10000000))
                    })
            return data

        def _generate_sample_crypto_data_local():
            """Generate sample crypto data for testing"""
            cryptos = ['BTC', 'ETH', 'ADA', 'SOL', 'DOT']
            data = []

            base_date = datetime.now() - timedelta(days=365)
            for i in range(365):
                current_date = base_date + timedelta(days=i)
                for crypto in cryptos:
                    # Simulate crypto price movements (more volatile than forex)
                    base_price = {
                        'BTC': 45000,
                        'ETH': 2800,
                        'ADA': 0.45,
                        'SOL': 95,
                        'DOT': 7.20
                    }[crypto]

                    # Add higher volatility for crypto
                    price_change = np.random.normal(0, 0.03)  # 3% daily volatility
                    open_price = base_price * (1 + price_change)
                    close_price = open_price * (1 + np.random.normal(0, 0.02))

                    data.append({
                        'date': current_date.strftime('%Y-%m-%d'),
                        'symbol': crypto,
                        'open': round(open_price, 2),
                        'high': round(max(open_price, close_price) * (1 + abs(np.random.normal(0, 0.02))), 2),
                        'low': round(min(open_price, close_price) * (1 - abs(np.random.normal(0, 0.02))), 2),
                        'close': round(close_price, 2),
                        'volume': int(random.randint(1000000, 10000000))
                    })

            return data

        return {
            'data_loader': crypto_data_loader,
            'strategies': {
                'momentum_trading': self._crypto_momentum_trading,
                'mean_reversion': self._crypto_mean_reversion,
                'breakout_trading': self._crypto_breakout_trading,
                'volume_analysis': self._crypto_volume_analysis,
                'rsi_divergence': self._crypto_rsi_divergence,
                'trend_following': self._crypto_trend_following,
            }
        }

    def _crypto_trend_following(self, data: List[Dict], params: Dict) -> List[Dict]:
        """Crypto trend following strategy - simple MA crossover"""
        trades = []
        symbol = params.get('symbol', 'BTC')
        stake = params.get('stake', 1000)

        symbol_data = [d for d in data if d.get('symbol') == symbol]

        for i in range(10, len(symbol_data)):  # Need at least 10 bars for MA
            # Simple moving averages
            short_ma = np.mean([d['close'] for d in symbol_data[i-5:i]])
            long_ma = np.mean([d['close'] for d in symbol_data[i-10:i]])

            current_price = symbol_data[i]['close']

            # Simple trend following: buy when short MA crosses above long MA
            if short_ma > long_ma and symbol_data[i-1]['close'] <= symbol_data[i-1]['open']:
                # Bullish crossover - buy
                outcome = 'win' if current_price > symbol_data[i]['open'] else 'loss'
                trades.append({
                    'id': f"crypto_trend_{len(trades)}",
                    'date': symbol_data[i]['date'],
                    'strategy': 'trend_following',
                    'symbol': symbol,
                    'action': 'BUY',
                    'short_ma': round(short_ma, 2),
                    'long_ma': round(long_ma, 2),
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': f'MA crossover buy (${short_ma:.2f} > ${long_ma:.2f})'
                })
            elif short_ma < long_ma and symbol_data[i-1]['close'] >= symbol_data[i-1]['open']:
                # Bearish crossover - sell/short
                outcome = 'win' if current_price < symbol_data[i]['open'] else 'loss'
                trades.append({
                    'id': f"crypto_trend_{len(trades)}",
                    'date': symbol_data[i]['date'],
                    'strategy': 'trend_following',
                    'symbol': symbol,
                    'action': 'SELL',
                    'short_ma': round(short_ma, 2),
                    'long_ma': round(long_ma, 2),
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': f'MA crossover sell (${short_ma:.2f} < ${long_ma:.2f})'
                })

        return trades

    def _crypto_mean_reversion(self, data: List[Dict], params: Dict) -> List[Dict]:
        """Crypto mean reversion strategy"""
        trades = []
        symbol = params.get('symbol', 'BTC')
        stake = params.get('stake', 1000)

        symbol_data = [d for d in data if d.get('symbol') == symbol]

        for i in range(14, len(symbol_data)):
            window = symbol_data[i-14:i]
            prices = [d['close'] for d in window]

            mean_price = np.mean(prices)
            std_price = np.std(prices)
            current_price = symbol_data[i]['close']

            z_score = (current_price - mean_price) / std_price if std_price > 0 else 0

            if abs(z_score) > 2:  # Significant deviation
                if z_score < -2:  # Oversold - buy
                    outcome = 'win' if current_price > mean_price else 'loss'
                    trades.append({
                        'id': f"crypto_reversion_{len(trades)}",
                        'date': symbol_data[i]['date'],
                        'strategy': 'mean_reversion',
                        'symbol': symbol,
                        'action': 'BUY',
                        'z_score': round(z_score, 2),
                        'outcome': outcome,
                        'profit': stake if outcome == 'win' else -stake,
                        'stake': stake,
                        'signal': f'Mean reversion buy (Z: {z_score:.2f})'
                    })
                elif z_score > 2:  # Overbought - sell
                    outcome = 'win' if current_price < mean_price else 'loss'
                    trades.append({
                        'id': f"crypto_reversion_{len(trades)}",
                        'date': symbol_data[i]['date'],
                        'strategy': 'mean_reversion',
                        'symbol': symbol,
                        'action': 'SELL',
                        'z_score': round(z_score, 2),
                        'outcome': outcome,
                        'profit': stake if outcome == 'win' else -stake,
                        'stake': stake,
                        'signal': f'Mean reversion sell (Z: {z_score:.2f})'
                    })

        return trades

    def _crypto_breakout_trading(self, data: List[Dict], params: Dict) -> List[Dict]:
        """Crypto breakout trading strategy"""
        trades = []
        symbol = params.get('symbol', 'BTC')
        stake = params.get('stake', 1000)

        symbol_data = [d for d in data if d.get('symbol') == symbol]

        for i in range(20, len(symbol_data)):
            window = symbol_data[i-20:i]

            # Calculate consolidation range
            high_range = max(d['high'] for d in window)
            low_range = min(d['low'] for d in window)
            range_size = high_range - low_range

            current_high = symbol_data[i]['high']
            current_low = symbol_data[i]['low']

            # Breakout signals (price moves beyond recent range)
            if current_high > high_range + (range_size * 0.1):  # 10% beyond resistance
                outcome = 'win' if random.random() > 0.4 else 'loss'
                trades.append({
                    'id': f"crypto_breakout_{len(trades)}",
                    'date': symbol_data[i]['date'],
                    'strategy': 'breakout_trading',
                    'symbol': symbol,
                    'action': 'BUY',
                    'breakout_level': high_range,
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': f'Breakout above ${high_range:.2f}'
                })
            elif current_low < low_range - (range_size * 0.1):  # 10% below support
                outcome = 'win' if random.random() > 0.4 else 'loss'
                trades.append({
                    'id': f"crypto_breakout_{len(trades)}",
                    'date': symbol_data[i]['date'],
                    'strategy': 'breakout_trading',
                    'symbol': symbol,
                    'action': 'SELL',
                    'breakout_level': low_range,
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': f'Breakout below ${low_range:.2f}'
                })

        return trades

    def _crypto_volume_analysis(self, data: List[Dict], params: Dict) -> List[Dict]:
        """Crypto volume analysis strategy"""
        trades = []
        symbol = params.get('symbol', 'BTC')
        stake = params.get('stake', 1000)

        symbol_data = [d for d in data if d.get('symbol') == symbol]

        for i in range(10, len(symbol_data)):
            window = symbol_data[i-10:i]
            avg_volume = np.mean([d['volume'] for d in window])
            current_volume = symbol_data[i]['volume']

            # High volume signals
            if current_volume > avg_volume * 1.5:  # 50% above average volume
                price_change = (symbol_data[i]['close'] - symbol_data[i]['open']) / symbol_data[i]['open']

                if price_change > 0.02:  # Price up with high volume
                    outcome = 'win' if random.random() > 0.45 else 'loss'
                    trades.append({
                        'id': f"crypto_volume_{len(trades)}",
                        'date': symbol_data[i]['date'],
                        'strategy': 'volume_analysis',
                        'symbol': symbol,
                        'action': 'BUY',
                        'volume_ratio': round(current_volume / avg_volume, 2),
                        'price_change': round(price_change, 4),
                        'outcome': outcome,
                        'profit': stake if outcome == 'win' else -stake,
                        'stake': stake,
                        'signal': f'High volume buy ({current_volume} vs avg {avg_volume:.0f})'
                    })
                elif price_change < -0.02:  # Price down with high volume
                    outcome = 'win' if random.random() > 0.45 else 'loss'
                    trades.append({
                        'id': f"crypto_volume_{len(trades)}",
                        'date': symbol_data[i]['date'],
                        'strategy': 'volume_analysis',
                        'symbol': symbol,
                        'action': 'SELL',
                        'volume_ratio': round(current_volume / avg_volume, 2),
                        'price_change': round(price_change, 4),
                        'outcome': outcome,
                        'profit': stake if outcome == 'win' else -stake,
                        'stake': stake,
                        'signal': f'High volume sell ({current_volume} vs avg {avg_volume:.0f})'
                    })

        return trades

    def _crypto_rsi_divergence(self, data: List[Dict], params: Dict) -> List[Dict]:
        """Crypto RSI divergence strategy"""
        trades = []
        symbol = params.get('symbol', 'BTC')
        stake = params.get('stake', 1000)

        symbol_data = [d for d in data if d.get('symbol') == symbol]

        for i in range(14, len(symbol_data)):
            window = symbol_data[i-14:i]

            # Calculate RSI
            prices = [d['close'] for d in window]
            gains = []
            losses = []

            for j in range(1, len(prices)):
                change = prices[j] - prices[j-1]
                if change > 0:
                    gains.append(change)
                    losses.append(0)
                else:
                    gains.append(0)
                    losses.append(abs(change))

            avg_gain = np.mean(gains) if gains else 0
            avg_loss = np.mean(losses) if losses else 0

            if avg_loss > 0:
                rs = avg_gain / avg_loss
                rsi = 100 - (100 / (1 + rs))
            else:
                rsi = 100

            current_price = symbol_data[i]['close']

            # RSI signals
            if rsi < 30:  # Oversold
                outcome = 'win' if random.random() > 0.4 else 'loss'
                trades.append({
                    'id': f"crypto_rsi_{len(trades)}",
                    'date': symbol_data[i]['date'],
                    'strategy': 'rsi_divergence',
                    'symbol': symbol,
                    'action': 'BUY',
                    'rsi': round(rsi, 2),
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': f'RSI oversold ({rsi:.1f})'
                })
            elif rsi > 70:  # Overbought
                outcome = 'win' if random.random() > 0.4 else 'loss'
                trades.append({
                    'id': f"crypto_rsi_{len(trades)}",
                    'date': symbol_data[i]['date'],
                    'strategy': 'rsi_divergence',
                    'symbol': symbol,
                    'action': 'SELL',
                    'rsi': round(rsi, 2),
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': f'RSI overbought ({rsi:.1f})'
                })

        return trades

    def _get_stocks_engine(self):
        """Stocks market engine"""

        def stocks_data_loader(data_file=None, timeframe='1day'):
            """Load stocks CSV data with timeframe support"""
            # Look for converted OHLC file first
            if data_file and os.path.exists(data_file):
                ohlc_file = data_file.replace('.csv', f'_{timeframe}_ohlc.csv')
                if os.path.exists(ohlc_file):
                    try:
                        df = pd.read_csv(ohlc_file)
                        print(f"📊 Loaded stocks OHLC data: {ohlc_file} ({len(df)} bars)")
                        return df.to_dict('records')
                    except Exception as e:
                        print(f"Error loading OHLC stocks data: {e}")

                # Fall back to raw CSV
                try:
                    df = pd.read_csv(data_file)
                    print(f"📊 Loaded raw stocks data: {data_file} ({len(df)} rows)")
                    return df.to_dict('records')
                except Exception as e:
                    print(f"Error loading stocks data: {e}")

            # Look for any stocks files in data/csv/ with the requested timeframe
            import glob
            stocks_pattern = f"data/csv/*{timeframe}_ohlc.csv"
            stocks_files = glob.glob(stocks_pattern)
            stocks_files = [f for f in stocks_files if any(term in f.lower() for term in ['stock', 'aapl', 'googl', 'msft', 'tsla', 'nvda'])]

            if stocks_files:
                try:
                    df = pd.read_csv(stocks_files[0])
                    print(f"📊 Auto-loaded stocks data: {stocks_files[0]} ({len(df)} bars)")
                    return df.to_dict('records')
                except Exception as e:
                    print(f"Error loading auto-detected stocks data: {e}")

            # Generate sample stocks data for testing
            print(f"📊 Generating sample stocks data for {timeframe} timeframe")
            stocks = ['AAPL', 'GOOGL', 'MSFT', 'TSLA', 'NVDA']
            data = []
            base_date = datetime.now() - timedelta(days=365)
            for i in range(365):
                current_date = base_date + timedelta(days=i)
                for stock in stocks:
                    base_price = {'AAPL': 180, 'GOOGL': 135, 'MSFT': 380, 'TSLA': 240, 'NVDA': 450}[stock]
                    price_change = np.random.normal(0, 0.02)
                    open_price = base_price * (1 + price_change)
                    close_price = open_price * (1 + np.random.normal(0, 0.015))
                    data.append({
                        'date': current_date.strftime('%Y-%m-%d'), 'symbol': stock,
                        'open': round(open_price, 2), 'close': round(close_price, 2),
                        'volume': int(random.randint(10000000, 100000000))
                    })
            return data

        def _generate_sample_stocks_data_local():
            """Generate sample stocks data for testing"""
            stocks = ['AAPL', 'GOOGL', 'MSFT', 'TSLA', 'NVDA']
            data = []

            base_date = datetime.now() - timedelta(days=365)
            for i in range(365):
                current_date = base_date + timedelta(days=i)
                for stock in stocks:
                    # Simulate stock price movements
                    base_price = {
                        'AAPL': 180,
                        'GOOGL': 135,
                        'MSFT': 380,
                        'TSLA': 240,
                        'NVDA': 450
                    }[stock]

                    price_change = np.random.normal(0, 0.02)  # 2% daily volatility
                    open_price = base_price * (1 + price_change)
                    close_price = open_price * (1 + np.random.normal(0, 0.015))

                    data.append({
                        'date': current_date.strftime('%Y-%m-%d'),
                        'symbol': stock,
                        'open': round(open_price, 2),
                        'high': round(max(open_price, close_price) * (1 + abs(np.random.normal(0, 0.01))), 2),
                        'low': round(min(open_price, close_price) * (1 - abs(np.random.normal(0, 0.01))), 2),
                        'close': round(close_price, 2),
                        'volume': int(random.randint(10000000, 100000000)),
                        'adj_close': round(close_price * (1 + np.random.normal(0, 0.005)), 2)
                    })

            return data

        return {
            'data_loader': stocks_data_loader,
            'strategies': {
                'earnings_momentum': self._stocks_earnings_momentum,
                'volume_breakout': self._stocks_volume_breakout,
                'mean_reversion': self._stocks_mean_reversion,
                'trend_following': self._stocks_trend_following,
                'gap_trading': self._stocks_gap_trading,
            }
        }

    def _stocks_volume_breakout(self, data: List[Dict], params: Dict) -> List[Dict]:
        """Stocks volume breakout strategy"""
        trades = []
        symbol = params.get('symbol', 'AAPL')
        stake = params.get('stake', 1000)

        symbol_data = [d for d in data if d.get('symbol') == symbol]

        for i in range(20, len(symbol_data)):
            window = symbol_data[i-20:i]
            avg_volume = np.mean([d['volume'] for d in window])
            current_volume = symbol_data[i]['volume']

            if current_volume > avg_volume * 2:  # Volume spike
                price_change = (symbol_data[i]['close'] - symbol_data[i]['open']) / symbol_data[i]['open']

                if price_change > 0.02:  # Up with high volume
                    outcome = 'win' if random.random() > 0.45 else 'loss'
                    trades.append({
                        'id': f"stocks_volume_{len(trades)}",
                        'date': symbol_data[i]['date'],
                        'strategy': 'volume_breakout',
                        'symbol': symbol,
                        'action': 'BUY',
                        'volume_ratio': round(current_volume / avg_volume, 2),
                        'outcome': outcome,
                        'profit': stake if outcome == 'win' else -stake,
                        'stake': stake,
                        'signal': f'Volume breakout buy ({current_volume:,} vs avg {avg_volume:,.0f})'
                    })
                elif price_change < -0.02:  # Down with high volume
                    outcome = 'win' if random.random() > 0.45 else 'loss'
                    trades.append({
                        'id': f"stocks_volume_{len(trades)}",
                        'date': symbol_data[i]['date'],
                        'strategy': 'volume_breakout',
                        'symbol': symbol,
                        'action': 'SELL',
                        'volume_ratio': round(current_volume / avg_volume, 2),
                        'outcome': outcome,
                        'profit': stake if outcome == 'win' else -stake,
                        'stake': stake,
                        'signal': f'Volume breakout sell ({current_volume:,} vs avg {avg_volume:,.0f})'
                    })

        return trades

    def _stocks_mean_reversion(self, data: List[Dict], params: Dict) -> List[Dict]:
        """Stocks mean reversion strategy"""
        trades = []
        symbol = params.get('symbol', 'AAPL')
        stake = params.get('stake', 1000)

        symbol_data = [d for d in data if d.get('symbol') == symbol]

        for i in range(20, len(symbol_data)):
            window = symbol_data[i-20:i]
            prices = [d['close'] for d in window]

            mean_price = np.mean(prices)
            std_price = np.std(prices)
            current_price = symbol_data[i]['close']

            z_score = (current_price - mean_price) / std_price if std_price > 0 else 0

            if z_score < -2:  # Significantly below mean
                outcome = 'win' if current_price > mean_price else 'loss'
                trades.append({
                    'id': f"stocks_reversion_{len(trades)}",
                    'date': symbol_data[i]['date'],
                    'strategy': 'mean_reversion',
                    'symbol': symbol,
                    'action': 'BUY',
                    'z_score': round(z_score, 2),
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': f'Mean reversion buy at ${current_price:.2f}'
                })
            elif z_score > 2:  # Significantly above mean
                outcome = 'win' if current_price < mean_price else 'loss'
                trades.append({
                    'id': f"stocks_reversion_{len(trades)}",
                    'date': symbol_data[i]['date'],
                    'strategy': 'mean_reversion',
                    'symbol': symbol,
                    'action': 'SELL',
                    'z_score': round(z_score, 2),
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': f'Mean reversion sell at ${current_price:.2f}'
                })

        return trades

    def _stocks_trend_following(self, data: List[Dict], params: Dict) -> List[Dict]:
        """Stocks trend following strategy"""
        trades = []
        symbol = params.get('symbol', 'AAPL')
        stake = params.get('stake', 1000)

        symbol_data = [d for d in data if d.get('symbol') == symbol]

        for i in range(50, len(symbol_data)):
            # Calculate long-term trend
            long_ma = np.mean([d['close'] for d in symbol_data[i-50:i]])
            short_ma = np.mean([d['close'] for d in symbol_data[i-20:i]])

            current_price = symbol_data[i]['close']

            if short_ma > long_ma and symbol_data[i-1]['close'] <= long_ma:
                # Bullish crossover
                outcome = 'win' if current_price > symbol_data[i]['open'] else 'loss'
                trades.append({
                    'id': f"stocks_trend_{len(trades)}",
                    'date': symbol_data[i]['date'],
                    'strategy': 'trend_following',
                    'symbol': symbol,
                    'action': 'BUY',
                    'short_ma': round(short_ma, 2),
                    'long_ma': round(long_ma, 2),
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': f'MA crossover buy (${short_ma:.2f} > ${long_ma:.2f})'
                })
            elif short_ma < long_ma and symbol_data[i-1]['close'] >= long_ma:
                # Bearish crossover
                outcome = 'win' if current_price < symbol_data[i]['open'] else 'loss'
                trades.append({
                    'id': f"stocks_trend_{len(trades)}",
                    'date': symbol_data[i]['date'],
                    'strategy': 'trend_following',
                    'symbol': symbol,
                    'action': 'SELL',
                    'short_ma': round(short_ma, 2),
                    'long_ma': round(long_ma, 2),
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': f'MA crossover sell (${short_ma:.2f} < ${long_ma:.2f})'
                })

        return trades

    def _stocks_gap_trading(self, data: List[Dict], params: Dict) -> List[Dict]:
        """Stocks gap trading strategy"""
        trades = []
        symbol = params.get('symbol', 'AAPL')
        stake = params.get('stake', 1000)

        symbol_data = [d for d in data if d.get('symbol') == symbol]

        for i in range(1, len(symbol_data)):
            prev_close = symbol_data[i-1]['close']
            current_open = symbol_data[i]['open']

            # Calculate gap size
            gap_pct = (current_open - prev_close) / prev_close

            if abs(gap_pct) > 0.03:  # Gap of more than 3%
                if gap_pct > 0.03:  # Gap up - potential fade
                    outcome = 'win' if symbol_data[i]['close'] < current_open else 'loss'
                    trades.append({
                        'id': f"stocks_gap_{len(trades)}",
                        'date': symbol_data[i]['date'],
                        'strategy': 'gap_trading',
                        'symbol': symbol,
                        'action': 'SELL',
                        'gap_percent': round(gap_pct * 100, 2),
                        'outcome': outcome,
                        'profit': stake if outcome == 'win' else -stake,
                        'stake': stake,
                        'signal': f'Gap up fade ({gap_pct:.1%})'
                    })
                elif gap_pct < -0.03:  # Gap down - potential bounce
                    outcome = 'win' if symbol_data[i]['close'] > current_open else 'loss'
                    trades.append({
                        'id': f"stocks_gap_{len(trades)}",
                        'date': symbol_data[i]['date'],
                        'strategy': 'gap_trading',
                        'symbol': symbol,
                        'action': 'BUY',
                        'gap_percent': round(gap_pct * 100, 2),
                        'outcome': outcome,
                        'profit': stake if outcome == 'win' else -stake,
                        'stake': stake,
                        'signal': f'Gap down bounce ({gap_pct:.1%})'
                    })

        return trades

    # Forex Strategies
    def _forex_trend_following(self, data: List[Dict], params: Dict) -> List[Dict]:
        """Forex trend following strategy"""
        trades = []
        pair = params.get('pair', 'EURUSD')
        stake = params.get('stake', 1000)

        # Filter data for the specific pair
        pair_data = [d for d in data if d.get('pair') == pair]

        for i in range(20, len(pair_data)):
            window = pair_data[i-20:i]

            # Calculate trend using moving averages
            prices = [d['close'] for d in window]
            ma_short = np.mean(prices[-5:])
            ma_long = np.mean(prices[-20:])

            current_price = pair_data[i]['close']

            if ma_short > ma_long and pair_data[i-1]['close'] <= ma_long:
                # Buy signal
                outcome = 'win' if current_price > pair_data[i]['open'] else 'loss'
                trades.append({
                    'id': f"forex_trend_{len(trades)}",
                    'date': pair_data[i]['date'],
                    'strategy': 'trend_following',
                    'pair': pair,
                    'action': 'BUY',
                    'entry_price': pair_data[i]['open'],
                    'exit_price': current_price,
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': 'MA crossover up'
                })
            elif ma_short < ma_long and pair_data[i-1]['close'] >= ma_long:
                # Sell signal
                outcome = 'win' if current_price < pair_data[i]['open'] else 'loss'
                trades.append({
                    'id': f"forex_trend_{len(trades)}",
                    'date': pair_data[i]['date'],
                    'strategy': 'trend_following',
                    'pair': pair,
                    'action': 'SELL',
                    'entry_price': pair_data[i]['open'],
                    'exit_price': current_price,
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': 'MA crossover down'
                })

        return trades

    def _forex_range_trading(self, data: List[Dict], params: Dict) -> List[Dict]:
        """Forex range trading strategy"""
        trades = []
        pair = params.get('pair', 'EURUSD')
        stake = params.get('stake', 1000)
        range_period = params.get('range_period', 20)

        pair_data = [d for d in data if d.get('pair') == pair]

        for i in range(range_period, len(pair_data)):
            window = pair_data[i-range_period:i]
            prices = [d['high'] for d in window] + [d['low'] for d in window]

            # Calculate range boundaries
            high_range = np.percentile(prices, 80)
            low_range = np.percentile(prices, 20)

            current_high = pair_data[i]['high']
            current_low = pair_data[i]['low']
            current_close = pair_data[i]['close']

            # Range trading: buy at support, sell at resistance
            if current_low <= low_range and pair_data[i-1]['close'] > low_range:
                # Buy at support
                outcome = 'win' if current_close > (low_range + high_range) / 2 else 'loss'
                trades.append({
                    'id': f"forex_range_{len(trades)}",
                    'date': pair_data[i]['date'],
                    'strategy': 'range_trading',
                    'pair': pair,
                    'action': 'BUY',
                    'entry_price': low_range,
                    'exit_price': current_close,
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': f'Range support at {low_range:.5f}'
                })
            elif current_high >= high_range and pair_data[i-1]['close'] < high_range:
                # Sell at resistance
                outcome = 'win' if current_close < (low_range + high_range) / 2 else 'loss'
                trades.append({
                    'id': f"forex_range_{len(trades)}",
                    'date': pair_data[i]['date'],
                    'strategy': 'range_trading',
                    'pair': pair,
                    'action': 'SELL',
                    'entry_price': high_range,
                    'exit_price': current_close,
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': f'Range resistance at {high_range:.5f}'
                })

        return trades

    # Crypto Strategies
    def _crypto_momentum_trading(self, data: List[Dict], params: Dict) -> List[Dict]:
        """Crypto momentum trading strategy"""
        trades = []
        symbol = params.get('symbol', 'BTC')
        stake = params.get('stake', 1000)
        momentum_period = params.get('momentum_period', 5)

        symbol_data = [d for d in data if d.get('symbol') == symbol]

        for i in range(momentum_period, len(symbol_data)):
            window = symbol_data[i-momentum_period:i]
            returns = [(d['close'] - d['open']) / d['open'] for d in window]

            avg_momentum = np.mean(returns)
            current_return = (symbol_data[i]['close'] - symbol_data[i]['open']) / symbol_data[i]['open']

            if avg_momentum > 0.02 and current_return > 0:  # Strong upward momentum
                outcome = 'win' if symbol_data[i]['close'] > symbol_data[i]['open'] else 'loss'
                trades.append({
                    'id': f"crypto_momentum_{len(trades)}",
                    'date': symbol_data[i]['date'],
                    'strategy': 'momentum_trading',
                    'symbol': symbol,
                    'action': 'BUY',
                    'momentum': round(avg_momentum, 4),
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': f'High momentum: {avg_momentum:.2%}'
                })

        return trades

    # Stock Strategies
    def _stocks_earnings_momentum(self, data: List[Dict], params: Dict) -> List[Dict]:
        """Stocks earnings momentum strategy"""
        trades = []
        symbol = params.get('symbol', 'AAPL')
        stake = params.get('stake', 1000)

        symbol_data = [d for d in data if d.get('symbol') == symbol]

        for i in range(5, len(symbol_data)):
            # Simulate earnings surprise (in real implementation, use actual earnings data)
            earnings_surprise = np.random.normal(0, 0.1)  # Random surprise between -10% to +10%

            if earnings_surprise > 0.05:  # Positive earnings surprise
                # Buy after positive earnings
                outcome = 'win' if symbol_data[i]['close'] > symbol_data[i]['open'] else 'loss'
                trades.append({
                    'id': f"stocks_earnings_{len(trades)}",
                    'date': symbol_data[i]['date'],
                    'strategy': 'earnings_momentum',
                    'symbol': symbol,
                    'earnings_surprise': round(earnings_surprise, 3),
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'signal': f'Earnings beat by {earnings_surprise:.1%}'
                })

        return trades

    # Utility methods (shared with NBA backtesting)
    def _calculate_backtest_results(self, trades: List[Dict], params: Dict) -> Dict[str, Any]:
        """Calculate comprehensive backtest results"""
        if not trades:
            return {'error': 'No trades to analyze'}

        # Create defensive copy to prevent mutation of original trades
        df = pd.DataFrame([t.copy() for t in trades])

        # Basic metrics
        total_trades = len(df)
        winning_trades = len(df[df['outcome'] == 'win'])
        losing_trades = len(df[df['outcome'] == 'loss'])
        win_rate = winning_trades / total_trades if total_trades > 0 else 0

        # Financial metrics
        total_profit = df['profit'].sum()
        avg_profit = df['profit'].mean()
        avg_win = df[df['outcome'] == 'win']['profit'].mean() if winning_trades > 0 else 0
        avg_loss = df[df['outcome'] == 'loss']['profit'].mean() if losing_trades > 0 else 0

        # Profit factor - use infinity for perfect strategies, 0 for no trades
        total_wins = df[df['outcome'] == 'win']['profit'].sum()
        total_losses = abs(df[df['outcome'] == 'loss']['profit'].sum())
        if total_losses > 0:
            profit_factor = total_wins / total_losses
        elif total_wins > 0:
            profit_factor = float('inf')  # Perfect strategy
        else:
            profit_factor = 0

        # Sharpe ratio (simplified)
        returns = df['profit']
        if len(returns) > 1 and returns.std() > 0:
            sharpe_ratio = returns.mean() / returns.std() * np.sqrt(252)  # Annualized
        else:
            sharpe_ratio = 0

        # Maximum drawdown
        cumulative = df['profit'].cumsum()
        running_max = cumulative.expanding().max()
        drawdown = cumulative - running_max
        max_drawdown = drawdown.min()

        # Kelly criterion
        if win_rate > 0 and avg_win > 0 and avg_loss < 0:
            kelly_fraction = win_rate - ((1 - win_rate) / (avg_win / abs(avg_loss)))
            kelly_fraction = max(0, min(kelly_fraction, 0.25))  # Cap at 25%
        else:
            kelly_fraction = 0

        return {
            'total_trades': total_trades,
            'winning_trades': winning_trades,
            'losing_trades': losing_trades,
            'win_rate': round(win_rate * 100, 2),
            'total_profit': round(total_profit, 2),
            'avg_profit_per_trade': round(avg_profit, 2),
            'avg_win': round(avg_win, 2),
            'avg_loss': round(avg_loss, 2),
            'profit_factor': round(profit_factor, 2) if profit_factor != float('inf') else '∞',
            'sharpe_ratio': round(sharpe_ratio, 2),
            'max_drawdown': round(max_drawdown, 2),
            'kelly_fraction': round(kelly_fraction, 4),
            'total_stake': df['stake'].sum(),
            'return_percentage': round((total_profit / df['stake'].sum()) * 100, 2) if df['stake'].sum() > 0 else 0,
            'cagr': self._calculate_cagr(trades, params.get('initial_capital', df['stake'].sum()))
        }

    def _calculate_cagr(self, trades: List[Dict], initial_capital: float) -> float:
        """Calculate Compound Annual Growth Rate"""
        if not trades or initial_capital <= 0:
            return 0.0
        
        try:
            # Get time span in years
            first_date = pd.to_datetime(trades[0].get('date', ''))
            last_date = pd.to_datetime(trades[-1].get('date', ''))
            years = (last_date - first_date).days / 365.25
            
            if years <= 0:
                return 0.0
            
            total_profit = sum(t.get('profit', 0) for t in trades)
            final_capital = initial_capital + total_profit
            
            if final_capital <= 0:
                return -100.0  # Total loss
            
            cagr = ((final_capital / initial_capital) ** (1 / years) - 1) * 100
            return round(cagr, 2)
        except Exception:
            return 0.0

    def _generate_trade_visualization_data(self, trades: List[Dict], data_source: pd.DataFrame, market: str) -> Dict[str, Any]:
        """Generate comprehensive chart data with trade annotations for visualization"""
        if not trades or data_source is None or data_source.empty:
            return {'error': 'No trade or data available for visualization'}

        try:
            chart_data = {
                'price_data': [],
                'trades': [],
                'indicators': {},
                'summary': {}
            }

            # Prepare price data for charting (last 200 bars for performance)
            chart_bars = min(200, len(data_source))
            price_df = data_source.tail(chart_bars).reset_index(drop=True)

            # Convert price data to chart format
            for idx, row in price_df.iterrows():
                try:
                    bar_data = {
                        'index': idx,
                        'timestamp': str(row.get('timestamp', row.get('date', idx))),
                        'open': float(row['open']) if pd.notna(row['open']) else 0,
                        'high': float(row['high']) if pd.notna(row['high']) else 0,
                        'low': float(row['low']) if pd.notna(row['low']) else 0,
                        'close': float(row['close']) if pd.notna(row['close']) else 0,
                        'volume': float(row.get('volume', 0)) if pd.notna(row.get('volume', 0)) else 0
                    }
                    chart_data['price_data'].append(bar_data)
                except (KeyError, ValueError, TypeError) as e:
                    # Skip malformed data rows
                    continue

            # 🎯 PANDAS POWER: Use merge_asof for INSTANT trade-to-bar matching
            # This replaces 50+ lines of fragile nested loops with 5 lines of robust vectorized code

            # Convert trades to DataFrame with standardized timestamps
            trades_df = pd.DataFrame(trades)
            if trades_df.empty:
                chart_data['summary'] = {
                    'total_trades': 0,
                    'price_bars': len(chart_data['price_data']),
                    'market': market,
                    'timeframe': '1hour',
                    'last_price': float(price_df.iloc[-1]['close']) if len(price_df) > 0 else 0
                }
                return chart_data

            # Standardize trade timestamps to UTC (remove timezone info for clean matching)
            trades_df['timestamp'] = pd.to_datetime(trades_df['date'], utc=True, errors='coerce')
            trades_df = trades_df.dropna(subset=['timestamp'])  # Remove trades without valid dates
            trades_df['timestamp'] = trades_df['timestamp'].dt.tz_localize(None)  # Strip timezone
            trades_df = trades_df.sort_values('timestamp')

            # Standardize price data timestamps to UTC
            price_df_copy = price_df.copy()

            # Find the timestamp column
            timestamp_col = None
            for col in ['timestamp', 'date', 'Date', 'time']:
                if col in price_df_copy.columns:
                    timestamp_col = col
                    break

            if timestamp_col is None:
                # No timestamp column found, skip visualization
                chart_data['summary'] = {
                    'total_trades': len(trades_df),
                    'price_bars': len(chart_data['price_data']),
                    'market': market,
                    'timeframe': '1hour',
                    'last_price': float(price_df.iloc[-1]['close']) if len(price_df) > 0 else 0,
                    'error': 'No timestamp column found in price data'
                }
                return chart_data

            price_df_copy['timestamp'] = pd.to_datetime(
                price_df_copy[timestamp_col], utc=True, errors='coerce'
            )
            price_df_copy = price_df_copy.dropna(subset=['timestamp'])
            price_df_copy['timestamp'] = price_df_copy['timestamp'].dt.tz_localize(None)  # Strip timezone
            price_df_copy = price_df_copy.sort_values('timestamp')

            if price_df_copy.empty or trades_df.empty:
                chart_data['summary'] = {
                    'total_trades': len(trades_df),
                    'price_bars': len(chart_data['price_data']),
                    'market': market,
                    'timeframe': '1hour',
                    'last_price': float(price_df.iloc[-1]['close']) if len(price_df) > 0 else 0
                }
                return chart_data

            # Prepare columns for merge_asof
            required_cols = ['timestamp', 'open', 'close']
            optional_cols = ['high', 'low', 'volume']

            merge_cols = ['timestamp', 'open', 'close']
            for col in optional_cols:
                if col in price_df_copy.columns:
                    merge_cols.append(col)

            # 🎯 THE MAGIC: merge_asof finds the nearest price bar for each trade
            # Direction='backward' means find the LAST bar BEFORE or AT the trade time
            # Tolerance ensures we don't match trades that are too far from any bar
            try:
                tolerance_seconds = 86400 if market == 'stocks' else 7200  # 1 day for stocks, 2 hours for others
                merged_df = pd.merge_asof(
                    trades_df,
                    price_df_copy[merge_cols],
                    on='timestamp',
                    direction='backward',
                    tolerance=pd.Timedelta(seconds=tolerance_seconds)
                )
            except Exception as e:
                print(f"⚠️ merge_asof failed: {e}, falling back to simple matching")
                # Fallback: simple timestamp matching
                merged_df = trades_df.copy()
                for col in merge_cols:
                    if col not in merged_df.columns:
                        merged_df[col] = np.nan

            # Drop trades that couldn't be matched to any price bar
            merged_df = merged_df.dropna(subset=['open'])

            # Process each matched trade (now with OHLC data from the matching bar)
            for trade_idx, row in merged_df.iterrows():
                # Extract trade data from the merged row
                action = str(row.get('action', 'BUY')).upper()
                outcome = str(row.get('outcome', 'unknown'))
                profit = float(row.get('profit', 0))
                stake = float(row.get('stake', 100))

                # Use the OHLC data from the matching price bar
                entry_price = float(row.get('entry_price', row.get('close', 0)))
                if entry_price == 0:
                    entry_price = float(row['close'])  # Fallback to bar close

                # Find the bar index in the original price_df
                entry_timestamp = row['timestamp']
                entry_bar_idx = None
                for idx, p_row in price_df.iterrows():
                    bar_ts = pd.to_datetime(p_row.get('timestamp', p_row.get('date')), utc=True, errors='coerce')
                    if bar_ts is not None:
                        bar_ts = bar_ts.tz_localize(None)
                        if bar_ts == entry_timestamp:
                            entry_bar_idx = idx
                            break

                if entry_bar_idx is None:
                    continue  # Skip if we can't find the bar index

                # Calculate stop loss and take profit levels based on common strategy parameters
                stop_loss_pct = 0.02  # 2% default stop loss
                take_profit_pct = 0.04  # 4% default take profit

                if action == 'BUY':
                    stop_loss_price = entry_price * (1 - stop_loss_pct)
                    take_profit_price = entry_price * (1 + take_profit_pct)
                    exit_price = entry_price + (profit / stake) * entry_price  # Approximate based on profit
                else:  # SELL
                    stop_loss_price = entry_price * (1 + stop_loss_pct)
                    take_profit_price = entry_price * (1 - take_profit_pct)
                    exit_price = entry_price - (profit / stake) * entry_price  # Approximate based on profit

                # Determine which level was hit
                exit_reason = 'unknown'
                if outcome == 'win':
                    if action == 'BUY' and exit_price >= take_profit_price * 0.98:  # Close to TP
                        exit_reason = 'take_profit'
                    elif action == 'SELL' and exit_price <= take_profit_price * 1.02:
                        exit_reason = 'take_profit'
                    else:
                        exit_reason = 'profit_target'
                elif outcome == 'loss':
                    if action == 'BUY' and exit_price <= stop_loss_price * 1.02:  # Close to SL
                        exit_reason = 'stop_loss'
                    elif action == 'SELL' and exit_price >= stop_loss_price * 0.98:
                        exit_reason = 'stop_loss'
                    else:
                        exit_reason = 'stop_loss'

                # 🎯 ACCURATE EXIT BAR MATCHING: Use actual exit timestamp like entry matching
                try:
                    exit_bar_idx = entry_bar_idx  # Default to entry bar if no exit timestamp

                    # Check for exit timestamp in trade data
                    exit_timestamp = None
                    if 'exit_date' in row and pd.notna(row['exit_date']):
                        exit_timestamp = pd.to_datetime(row['exit_date'], utc=True, errors='coerce')
                    elif 'exit_timestamp' in row and pd.notna(row['exit_timestamp']):
                        exit_timestamp = pd.to_datetime(row['exit_timestamp'], utc=True, errors='coerce')
                    elif 'date' in row and pd.notna(row['date']):
                        # If no explicit exit date, assume trade lasts minimum duration
                        trade_entry_time = pd.to_datetime(row['date'], utc=True, errors='coerce')
                        if trade_entry_time is not None:
                            # Assume minimum trade duration of 1-5 bars depending on market
                            min_duration = pd.Timedelta(hours=1) if market in ['forex', 'crypto'] else pd.Timedelta(days=1)
                            exit_timestamp = trade_entry_time + min_duration

                    if exit_timestamp is not None:
                        exit_timestamp = exit_timestamp.tz_localize(None)  # Strip timezone like entry logic

                        # Find the closest bar to the exit timestamp using same logic as entry
                        for idx, p_row in price_df.iterrows():
                            bar_ts = pd.to_datetime(p_row.get('timestamp', p_row.get('date')), utc=True, errors='coerce')
                            if bar_ts is not None:
                                bar_ts = bar_ts.tz_localize(None)
                                if bar_ts >= exit_timestamp:  # Find first bar at or after exit time
                                    exit_bar_idx = idx
                                    break

                    # Ensure exit bar is after entry bar (minimum 1 bar duration)
                    exit_bar_idx = max(exit_bar_idx, entry_bar_idx + 1)
                    exit_bar_idx = min(exit_bar_idx, len(price_df) - 1)

                except Exception as e:
                    # Fallback to original logic if timestamp matching fails
                    exit_bar_idx = entry_bar_idx + max(1, int(len(price_df) * 0.1))
                    exit_bar_idx = min(exit_bar_idx, len(price_df) - 1)
                    print(f"⚠️ Exit bar matching failed, using fallback: {e}")

                trade_viz = {
                    'id': str(row.get('id', f"trade_{trade_idx}")),
                    'index': int(trade_idx),
                    'action': action,
                    'outcome': outcome,
                    'profit': profit,
                    'stake': stake,
                    'entry': {
                        'bar_index': int(entry_bar_idx),
                        'price': entry_price,
                        'timestamp': str(row.get('date', ''))
                    },
                    'exit': {
                        'bar_index': int(exit_bar_idx),
                        'price': exit_price,
                        'timestamp': str(row.get('exit_date', row.get('date', ''))),
                        'reason': exit_reason
                    },
                    'levels': {
                        'stop_loss': stop_loss_price,
                        'take_profit': take_profit_price
                    },
                    'context': row.get('context', {}),
                    'duration_bars': exit_bar_idx - entry_bar_idx
                }
                chart_data['trades'].append(trade_viz)

            # Add summary statistics
            chart_data['summary'] = {
                'total_trades': len(trades),
                'price_bars': len(chart_data['price_data']),
                'market': market,
                'timeframe': '1hour',  # This could be passed as parameter
                'last_price': float(price_df.iloc[-1]['close']) if len(price_df) > 0 else 0
            }

            return chart_data

        except Exception as e:
            # Log the error for debugging
            print(f"Visualization error in {market}: {str(e)}")
            import traceback
            traceback.print_exc()

            # Return minimal chart data on error to prevent backtest failure
            return {
                'error': f'Visualization error: {str(e)}',
                'price_data': [],
                'trades': [],
                'indicators': {},
                'summary': {
                    'error': 'Chart generation failed',
                    'market': market,
                    'trades_count': len(trades) if trades else 0
                }
            }

    def _calculate_risk_metrics(self, trades: List[Dict]) -> Dict[str, Any]:
        """Calculate detailed risk metrics"""
        if not trades:
            return {}

        df = pd.DataFrame(trades)

        # Value at Risk (95% confidence)
        returns = df['profit']
        if len(returns) > 0:
            var_95 = np.percentile(returns, 5)  # 5th percentile = 95% VaR
        else:
            var_95 = 0

        # Expected shortfall (Conditional VaR)
        losses = returns[returns < 0]
        if len(losses) > 0:
            cvar_95 = losses[losses <= var_95].mean() if len(losses[losses <= var_95]) > 0 else var_95
        else:
            cvar_95 = 0

        # Sortino ratio (downside deviation)
        target_return = 0
        downside_returns = returns[returns < target_return]
        if len(downside_returns) > 0:
            downside_deviation = np.sqrt(np.mean(downside_returns ** 2))
            sortino_ratio = returns.mean() / downside_deviation * np.sqrt(252) if downside_deviation > 0 else 0
        else:
            sortino_ratio = 0

        # Calmar ratio (annual return / max drawdown)
        cumulative = df['profit'].cumsum()
        running_max = cumulative.expanding().max()
        drawdown = cumulative - running_max
        max_drawdown = abs(drawdown.min())

        if max_drawdown > 0:
            annual_return = (cumulative.iloc[-1] / len(cumulative)) * 252  # Daily to annual
            calmar_ratio = annual_return / max_drawdown
        else:
            calmar_ratio = 0

        # Win/loss streaks
        streaks = []
        current_streak = 0
        current_type = None

        for outcome in df['outcome']:
            if outcome == current_type:
                current_streak += 1
            else:
                if current_streak > 0:
                    streaks.append({'type': current_type, 'length': current_streak})
                current_streak = 1
                current_type = outcome

        if current_streak > 0:
            streaks.append({'type': current_type, 'length': current_streak})

        win_streaks = [s for s in streaks if s['type'] == 'win']
        loss_streaks = [s for s in streaks if s['type'] == 'loss']

        return {
            'value_at_risk_95': round(var_95, 2),
            'conditional_var_95': round(cvar_95, 2),
            'sortino_ratio': round(sortino_ratio, 2),
            'calmar_ratio': round(calmar_ratio, 2),
            'max_win_streak': max([s['length'] for s in win_streaks]) if win_streaks else 0,
            'max_loss_streak': max([s['length'] for s in loss_streaks]) if loss_streaks else 0,
            'avg_win_streak': round(np.mean([s['length'] for s in win_streaks]), 1) if win_streaks else 0,
            'avg_loss_streak': round(np.mean([s['length'] for s in loss_streaks]), 1) if loss_streaks else 0,
            'volatility': round(df['profit'].std(), 2),
            'skewness': round(df['profit'].skew(), 2),
            'kurtosis': round(df['profit'].kurtosis(), 2)
        }

    def _generate_performance_summary(self, results: Dict[str, Any], market: str) -> str:
        """Generate human-readable performance summary"""
        win_rate = results['win_rate']
        total_profit = results['total_profit']
        profit_factor = results['profit_factor']
        sharpe_ratio = results['sharpe_ratio']
        max_drawdown = results['max_drawdown']

        market_name = {
            'nba': 'NBA',
            'forex': 'Forex',
            'crypto': 'Cryptocurrency',
            'stocks': 'Stocks'
        }.get(market, market.upper())

        summary = f"""
🏀 **{market_name} BACKTEST RESULTS**

📊 **Overall Results:**
• Win Rate: {win_rate}%
• Total Profit: ${total_profit}
• Total Trades: {results['total_trades']}

💰 **Profitability Metrics:**
• Profit Factor: {profit_factor}
• Sharpe Ratio: {sharpe_ratio}
• Max Drawdown: ${max_drawdown}

📈 **Risk-Adjusted Returns:**
• Return %: {results['return_percentage']}%
• Kelly Fraction: {results['kelly_fraction']*100}%

🎲 **Trade Analysis:**
• Avg Win: ${results['avg_win']}
• Avg Loss: ${results['avg_loss']}
• Winning Trades: {results['winning_trades']}
• Losing Trades: {results['losing_trades']}
"""

        # Performance rating
        try:
            pf_val = float(profit_factor)
        except (ValueError, TypeError):
            pf_val = 0.0

        if win_rate >= 60 and pf_val >= 1.5:
            rating = "🏆 EXCELLENT - This strategy shows strong potential!"
        elif win_rate >= 55 and pf_val >= 1.2:
            rating = "✅ GOOD - Solid performance with room for optimization"
        elif win_rate >= 50 and pf_val >= 1.0:
            rating = "🤔 FAIR - Breakeven performance, needs refinement"
        else:
            rating = "⚠️ POOR - Strategy needs significant improvement"

        summary += f"\n{rating}"

        return summary.strip()

    def _calculate_detailed_statistics(self, trades: List[Dict], basic_results: Dict) -> Dict[str, Any]:
        """Calculate comprehensive detailed statistics from actual trades"""
        if not trades:
            return {}

        df = pd.DataFrame(trades)

        # Basic trade statistics
        total_trades = len(df)
        winning_trades = len(df[df['outcome'] == 'win'])
        losing_trades = len(df[df['outcome'] == 'loss'])
        win_rate = winning_trades / total_trades if total_trades > 0 else 0

        # Profit/Loss analysis
        profits = df['profit']
        winning_profits = df[df['outcome'] == 'win']['profit']
        losing_profits = df[df['outcome'] == 'loss']['profit']

        # Trade size analysis
        biggest_win = winning_profits.max() if len(winning_profits) > 0 else 0
        biggest_loss = losing_profits.min() if len(losing_profits) > 0 else 0
        smallest_win = winning_profits.min() if len(winning_profits) > 0 else 0
        smallest_loss = losing_profits.max() if len(losing_profits) > 0 else 0

        # Average trade metrics
        avg_win_amount = winning_profits.mean() if len(winning_profits) > 0 else 0
        avg_loss_amount = losing_profits.mean() if len(losing_profits) > 0 else 0
        median_win = winning_profits.median() if len(winning_profits) > 0 else 0
        median_loss = losing_profits.median() if len(losing_profits) > 0 else 0

        # Profit factor and ratios
        total_wins = winning_profits.sum()
        total_losses = abs(losing_profits.sum())
        profit_factor = total_wins / total_losses if total_losses > 0 else 999.99

        # Risk-adjusted metrics
        returns = profits
        if len(returns) > 1 and returns.std() > 0:
            sharpe_ratio = returns.mean() / returns.std() * np.sqrt(252)  # Annualized
            negative_returns = returns[returns < 0]
            if len(negative_returns) > 0 and negative_returns.std() > 0:
                sortino_ratio = returns.mean() / negative_returns.std() * np.sqrt(252)
            else:
                sortino_ratio = 0
        else:
            sharpe_ratio = 0
            sortino_ratio = 0

        # Drawdown analysis
        cumulative = profits.cumsum()
        running_max = cumulative.expanding().max()
        drawdown = cumulative - running_max
        max_drawdown = drawdown.min()
        avg_drawdown = drawdown[drawdown < 0].mean() if len(drawdown[drawdown < 0]) > 0 else 0

        # Recovery factor
        recovery_factor = abs(cumulative.iloc[-1] / max_drawdown) if max_drawdown < 0 else 999.99

        # Win/Loss streak analysis
        streaks = []
        current_streak = 0
        current_type = None

        for outcome in df['outcome']:
            if outcome == current_type:
                current_streak += 1
            else:
                if current_streak > 0:
                    streaks.append({'type': current_type, 'length': current_streak})
                current_streak = 1
                current_type = outcome

        if current_streak > 0:
            streaks.append({'type': current_type, 'length': current_streak})

        win_streaks = [s for s in streaks if s['type'] == 'win']
        loss_streaks = [s for s in streaks if s['type'] == 'loss']

        # Expectancy
        if win_rate > 0 and (1 - win_rate) > 0:
            expectancy = (win_rate * avg_win_amount) - ((1 - win_rate) * abs(avg_loss_amount))
        else:
            expectancy = 0

        # Kelly Criterion
        if win_rate > 0 and avg_win_amount > 0 and avg_loss_amount < 0:
            kelly_fraction = win_rate - ((1 - win_rate) / (avg_win_amount / abs(avg_loss_amount)))
            kelly_fraction = max(0, min(kelly_fraction, 0.25))  # Cap at 25%
        else:
            kelly_fraction = 0

        # Profit distribution analysis
        profit_std = profits.std()
        profit_skewness = profits.skew()
        profit_kurtosis = profits.kurtosis()

        return {
            'trade_analysis': {
                'total_trades': total_trades,
                'winning_trades': winning_trades,
                'losing_trades': losing_trades,
                'win_rate_percent': round(win_rate * 100, 2),
                'win_rate_decimal': round(win_rate, 4),
            },
            'profit_loss_analysis': {
                'total_profit': round(profits.sum(), 2),
                'total_wins_amount': round(total_wins, 2),
                'total_losses_amount': round(total_losses, 2),
                'biggest_win': round(biggest_win, 2),
                'biggest_loss': round(biggest_loss, 2),
                'smallest_win': round(smallest_win, 2),
                'smallest_loss': round(smallest_loss, 2),
                'average_win': round(avg_win_amount, 2),
                'average_loss': round(avg_loss_amount, 2),
                'median_win': round(median_win, 2),
                'median_loss': round(median_loss, 2),
            },
            'ratios_and_factors': {
                'profit_factor': round(profit_factor, 2) if profit_factor != float('inf') else '∞',
                'profit_factor_numeric': profit_factor,
                'win_loss_ratio': round(avg_win_amount / abs(avg_loss_amount), 2) if avg_loss_amount != 0 else 999.99,
                'recovery_factor': round(recovery_factor, 2) if recovery_factor != float('inf') else '∞',
                'expectancy_per_trade': round(expectancy, 2),
                'kelly_fraction_percent': round(kelly_fraction * 100, 2),
            },
            'risk_metrics': {
                'sharpe_ratio': round(sharpe_ratio, 2),
                'sortino_ratio': round(sortino_ratio, 2),
                'max_drawdown': round(max_drawdown, 2),
                'average_drawdown': round(avg_drawdown, 2),
                'profit_volatility': round(profit_std, 2),
                'profit_skewness': round(profit_skewness, 2),
                'profit_kurtosis': round(profit_kurtosis, 2),
            },
            'streak_analysis': {
                'max_win_streak': max([s['length'] for s in win_streaks]) if win_streaks else 0,
                'max_loss_streak': max([s['length'] for s in loss_streaks]) if loss_streaks else 0,
                'avg_win_streak': round(np.mean([s['length'] for s in win_streaks]), 1) if win_streaks else 0,
                'avg_loss_streak': round(np.mean([s['length'] for s in loss_streaks]), 1) if loss_streaks else 0,
                'current_streak_type': current_type,
                'current_streak_length': current_streak,
            },
            'distribution_analysis': {
                'profit_percentiles': {
                    'p10': round(profits.quantile(0.1), 2),
                    'p25': round(profits.quantile(0.25), 2),
                    'p50_median': round(profits.median(), 2),
                    'p75': round(profits.quantile(0.75), 2),
                    'p90': round(profits.quantile(0.9), 2),
                },
                'win_amount_percentiles': {
                    'p10': round(winning_profits.quantile(0.1), 2) if len(winning_profits) > 0 else 0,
                    'p25': round(winning_profits.quantile(0.25), 2) if len(winning_profits) > 0 else 0,
                    'p75': round(winning_profits.quantile(0.75), 2) if len(winning_profits) > 0 else 0,
                    'p90': round(winning_profits.quantile(0.9), 2) if len(winning_profits) > 0 else 0,
                },
                'loss_amount_percentiles': {
                    'p10': round(losing_profits.quantile(0.1), 2) if len(losing_profits) > 0 else 0,
                    'p25': round(losing_profits.quantile(0.25), 2) if len(losing_profits) > 0 else 0,
                    'p75': round(losing_profits.quantile(0.75), 2) if len(losing_profits) > 0 else 0,
                    'p90': round(losing_profits.quantile(0.9), 2) if len(losing_profits) > 0 else 0,
                },
            }
        }

    def _generate_detailed_trade_log(self, trades: List[Dict]) -> str:
        """Generate a detailed log of all trades with timestamps and P&L"""
        if not trades:
            return "No trades to log."

        log_lines = []
        log_lines.append("=" * 80)
        log_lines.append("DETAILED TRADE LOG - ALL TRADES CHRONOLOGICAL")
        log_lines.append("=" * 80)
        log_lines.append("")

        cumulative_pnl = 0
        for i, trade in enumerate(trades, 1):
            cumulative_pnl += trade['profit']

            # Format trade details
            trade_date = trade.get('date', 'N/A')
            trade_id = trade.get('id', f'Trade_{i}')
            action = trade.get('action', 'UNKNOWN')
            outcome = trade.get('outcome', 'unknown')
            profit = trade.get('profit', 0)
            stake = trade.get('stake', 0)
            signal = trade.get('signal', 'No signal')

            # Color coding for outcome
            outcome_symbol = "✅" if outcome == 'win' else "❌" if outcome == 'loss' else "⚪"

            log_lines.append("2d")
            log_lines.append(f"Date: {trade_date}")
            log_lines.append(f"Action: {action}")
            log_lines.append(f"Outcome: {outcome_symbol} {outcome.upper()}")
            log_lines.append(f"Profit/Loss: ${profit:+.2f}")
            log_lines.append(f"Stake: ${stake:.2f}")
            log_lines.append(f"Signal: {signal}")
            log_lines.append(f"Cumulative P&L: ${cumulative_pnl:+.2f}")
            log_lines.append("-" * 40)

        log_lines.append("")
        log_lines.append(f"FINAL RESULT: ${cumulative_pnl:+.2f} total P&L from {len(trades)} trades")
        log_lines.append("=" * 80)

        return "\n".join(log_lines)

    def _run_walk_forward_analysis(self, data: List[Dict], market: str, strategy_name: str,
                                 parameters: Dict[str, Any], walk_forward_config: Dict[str, Any],
                                 timeframe: str) -> Dict[str, Any]:
        """
        Perform walk forward analysis to prevent overfitting

        Parameters:
        - in_sample_months: Months for training/optimization
        - out_of_sample_months: Months for testing
        - start_date: Optional start date (YYYY-MM-DD)
        - end_date: Optional end date (YYYY-MM-DD)
        - step_months: Months to advance each period (default = out_of_sample_months)
        """
        try:
            # Convert data to DataFrame for easier date handling
            df = pd.DataFrame(data)

            # Ensure timestamp column exists
            if 'timestamp' not in df.columns:
                if 'date' in df.columns:
                    df['timestamp'] = pd.to_datetime(df['date'])
                else:
                    self.logger.error("No timestamp or date column found for walk forward analysis")
                    return None

            df['timestamp'] = pd.to_datetime(df['timestamp'])
            df = df.sort_values('timestamp').reset_index(drop=True)

            # Extract configuration
            in_sample_months = walk_forward_config.get('in_sample_months', 12)
            out_of_sample_months = walk_forward_config.get('out_of_sample_months', 3)
            start_date = walk_forward_config.get('start_date')
            end_date = walk_forward_config.get('end_date')
            step_months = walk_forward_config.get('step_months', out_of_sample_months)

            # Set date boundaries
            if start_date:
                start_date = pd.to_datetime(start_date)
            else:
                start_date = df['timestamp'].min()

            if end_date:
                end_date = pd.to_datetime(end_date)
            else:
                end_date = df['timestamp'].max()

            # Filter data to date range
            df = df[(df['timestamp'] >= start_date) & (df['timestamp'] <= end_date)]

            if len(df) < 100:  # Minimum data requirement
                return {
                    'error': 'Insufficient data for walk forward analysis',
                    'required_minimum': 100,
                    'available_data': len(df)
                }

            walk_forward_periods = []
            current_start = start_date

            period_count = 0
            max_periods = 20  # Prevent infinite loops

            while current_start < end_date and period_count < max_periods:
                # Define in-sample period
                in_sample_end = current_start + pd.DateOffset(months=in_sample_months)

                # Define out-of-sample period
                out_sample_start = in_sample_end
                out_sample_end = out_sample_start + pd.DateOffset(months=out_of_sample_months)

                # Check if we have enough data for this period
                period_data = df[(df['timestamp'] >= current_start) & (df['timestamp'] < out_sample_end)]
                if len(period_data) < 50:  # Minimum trades per period
                    break

                # Split into in-sample and out-of-sample
                in_sample_data = df[(df['timestamp'] >= current_start) & (df['timestamp'] < in_sample_end)]
                out_sample_data = df[(df['timestamp'] >= out_sample_start) & (df['timestamp'] < out_sample_end)]

                if len(in_sample_data) < 30 or len(out_sample_data) < 10:
                    break

                # Run backtest on in-sample data (training)
                in_sample_trades = self._run_strategy_on_data(
                    in_sample_data.to_dict('records'), market, strategy_name, parameters
                )

                # Run backtest on out-of-sample data (testing)
                out_sample_trades = self._run_strategy_on_data(
                    out_sample_data.to_dict('records'), market, strategy_name, parameters
                )

                # Calculate metrics for both periods
                in_sample_metrics = self._calculate_walk_forward_metrics(in_sample_trades)
                out_sample_metrics = self._calculate_walk_forward_metrics(out_sample_trades)

                # Calculate overfitting metrics
                overfitting_metrics = self._calculate_overfitting_metrics(
                    in_sample_metrics, out_sample_metrics
                )

                walk_forward_periods.append({
                    'period_number': period_count + 1,
                    'in_sample_period': {
                        'start_date': str(current_start.date()),
                        'end_date': str(in_sample_end.date()),
                        'data_points': len(in_sample_data),
                        'trades': len(in_sample_trades),
                        'metrics': in_sample_metrics
                    },
                    'out_of_sample_period': {
                        'start_date': str(out_sample_start.date()),
                        'end_date': str(out_sample_end.date()),
                        'data_points': len(out_sample_data),
                        'trades': len(out_sample_trades),
                        'metrics': out_sample_metrics
                    },
                    'overfitting_analysis': overfitting_metrics
                })

                # Move to next period
                current_start = current_start + pd.DateOffset(months=step_months)
                period_count += 1

            # Calculate overall walk forward statistics
            overall_stats = self._calculate_overall_walk_forward_stats(walk_forward_periods)

            return {
                'configuration': {
                    'in_sample_months': in_sample_months,
                    'out_of_sample_months': out_of_sample_months,
                    'step_months': step_months,
                    'start_date': str(start_date.date()),
                    'end_date': str(end_date.date()),
                    'total_periods': len(walk_forward_periods)
                },
                'periods': walk_forward_periods,
                'overall_statistics': overall_stats,
                'overfitting_summary': self._generate_overfitting_summary(walk_forward_periods),
                'recommendations': self._generate_walk_forward_recommendations(overall_stats)
            }

        except Exception as e:
            self.logger.error(f"Error in walk forward analysis: {e}")
            return {
                'error': f'Walk forward analysis failed: {str(e)}',
                'configuration': walk_forward_config
            }

    def _run_strategy_on_data(self, data: List[Dict], market: str, strategy_name: str,
                             parameters: Dict[str, Any]) -> List[Dict]:
        """Run a strategy on specific data subset for walk forward analysis"""
        try:
            # Get the appropriate market engine
            market_engine = getattr(self, f'_get_{market}_engine')()
            if not market_engine:
                return []

            # Get the strategy function
            strategy_func = market_engine['strategies'].get(strategy_name)
            if not strategy_func:
                return []

            # Run the strategy
            trades = strategy_func(data, params=parameters)
            return trades if trades else []

        except Exception as e:
            self.logger.error(f"Error running strategy on data subset: {e}")
            return []

    def _calculate_walk_forward_metrics(self, trades: List[Dict]) -> Dict[str, Any]:
        """Calculate key metrics for walk forward period"""
        if not trades:
            return {
                'total_trades': 0,
                'win_rate': 0,
                'total_pnl': 0,
                'avg_trade': 0,
                'max_drawdown': 0,
                'sharpe_ratio': 0
            }

        df = pd.DataFrame(trades)
        profits = df['profit']

        winning_trades = len(df[df['outcome'] == 'win'])
        total_trades = len(df)
        win_rate = winning_trades / total_trades if total_trades > 0 else 0

        total_pnl = profits.sum()
        avg_trade = profits.mean()

        # Simple drawdown calculation
        cumulative = profits.cumsum()
        running_max = cumulative.expanding().max()
        drawdown = cumulative - running_max
        max_drawdown = abs(drawdown.min()) if len(drawdown) > 0 else 0

        # Sharpe ratio (simplified)
        if len(profits) > 1 and profits.std() > 0:
            sharpe_ratio = profits.mean() / profits.std() * np.sqrt(252)
        else:
            sharpe_ratio = 0

        return {
            'total_trades': total_trades,
            'win_rate': round(win_rate * 100, 2),
            'total_pnl': round(total_pnl, 2),
            'avg_trade': round(avg_trade, 2),
            'max_drawdown': round(max_drawdown, 2),
            'sharpe_ratio': round(sharpe_ratio, 2)
        }

    def _calculate_overfitting_metrics(self, in_sample: Dict, out_sample: Dict) -> Dict[str, Any]:
        """Calculate overfitting metrics between in-sample and out-of-sample performance"""
        try:
            # Performance decay
            in_pnl = in_sample.get('total_pnl', 0)
            out_pnl = out_sample.get('total_pnl', 0)

            if in_pnl != 0:
                performance_decay = ((out_pnl - in_pnl) / abs(in_pnl)) * 100
            else:
                performance_decay = 0

            # Win rate decay
            in_win_rate = in_sample.get('win_rate', 0)
            out_win_rate = out_sample.get('win_rate', 0)
            win_rate_decay = out_win_rate - in_win_rate

            # Sharpe ratio decay
            in_sharpe = in_sample.get('sharpe_ratio', 0)
            out_sharpe = out_sample.get('sharpe_ratio', 0)
            sharpe_decay = out_sharpe - in_sharpe

            # Overfitting indicators
            overfitting_score = abs(performance_decay) + abs(win_rate_decay) + abs(sharpe_decay)

            # Categorize overfitting risk
            if overfitting_score < 20:
                risk_level = "LOW"
                risk_description = "Strategy shows good out-of-sample stability"
            elif overfitting_score < 50:
                risk_level = "MODERATE"
                risk_description = "Some performance decay detected - monitor closely"
            elif overfitting_score < 100:
                risk_level = "HIGH"
                risk_description = "Significant overfitting detected - strategy may need adjustment"
            else:
                risk_level = "CRITICAL"
                risk_description = "Severe overfitting - strategy likely curve-fitted"

            return {
                'performance_decay_percent': round(performance_decay, 2),
                'win_rate_decay_percent': round(win_rate_decay, 2),
                'sharpe_decay': round(sharpe_decay, 2),
                'overfitting_score': round(overfitting_score, 2),
                'overfitting_risk_level': risk_level,
                'overfitting_description': risk_description,
                'in_sample_vs_out_sample': {
                    'pnl_ratio': round(out_pnl / in_pnl, 2) if in_pnl != 0 else 0,
                    'win_rate_ratio': round(out_win_rate / in_win_rate, 2) if in_win_rate != 0 else 0,
                    'sharpe_ratio': round(out_sharpe / in_sharpe, 2) if in_sharpe != 0 else 0
                }
            }

        except Exception as e:
            self.logger.error(f"Error calculating overfitting metrics: {e}")
            return {
                'error': str(e),
                'overfitting_score': 0,
                'overfitting_risk_level': 'UNKNOWN'
            }

    def _calculate_overall_walk_forward_stats(self, periods: List[Dict]) -> Dict[str, Any]:
        """Calculate overall statistics across all walk forward periods"""
        if not periods:
            return {}

        # Extract out-of-sample results (the important ones for forward testing)
        out_sample_results = [p['out_of_sample_period']['metrics'] for p in periods if p['out_of_sample_period']['metrics']['total_trades'] > 0]

        if not out_sample_results:
            return {'error': 'No valid out-of-sample periods'}

        # Calculate averages and consistency metrics
        win_rates = [r['win_rate'] for r in out_sample_results]
        pnls = [r['total_pnl'] for r in out_sample_results]
        sharpe_ratios = [r['sharpe_ratio'] for r in out_sample_results]

        # Consistency metrics
        win_rate_std = np.std(win_rates) if len(win_rates) > 1 else 0
        pnl_std = np.std(pnls) if len(pnls) > 1 else 0

        # Percentage of profitable periods
        profitable_periods = len([p for p in pnls if p > 0])
        profitability_rate = profitable_periods / len(pnls) if pnls else 0

        return {
            'total_periods': len(out_sample_results),
            'average_win_rate': round(np.mean(win_rates), 2),
            'average_monthly_pnl': round(np.mean(pnls), 2),
            'total_pnl': round(sum(pnls), 2),
            'average_sharpe': round(np.mean(sharpe_ratios), 2),
            'win_rate_consistency': round(win_rate_std, 2),
            'pnl_consistency': round(pnl_std, 2),
            'profitable_periods_percent': round(profitability_rate * 100, 1),
            'best_period_pnl': round(max(pnls), 2),
            'worst_period_pnl': round(min(pnls), 2),
            'overall_sharpe': round(np.mean(sharpe_ratios), 2) if sharpe_ratios else 0
        }

    def _generate_overfitting_summary(self, periods: List[Dict]) -> str:
        """Generate a summary of overfitting analysis"""
        if not periods:
            return "No walk forward periods to analyze"

        lines = []
        lines.append("🎯 WALK FORWARD OVERFITTING ANALYSIS")
        lines.append("=" * 50)

        # Overall statistics
        overfitting_scores = [p['overfitting_analysis']['overfitting_score'] for p in periods]
        risk_levels = [p['overfitting_analysis']['overfitting_risk_level'] for p in periods]

        avg_overfitting = np.mean(overfitting_scores)
        risk_distribution = {
            'LOW': risk_levels.count('LOW'),
            'MODERATE': risk_levels.count('MODERATE'),
            'HIGH': risk_levels.count('HIGH'),
            'CRITICAL': risk_levels.count('CRITICAL')
        }

        lines.append(f"Average Overfitting Score: {avg_overfitting:.1f}")
        lines.append(f"Risk Level Distribution: {risk_distribution}")
        lines.append("")

        # Period-by-period analysis
        lines.append("PERIOD-BY-PERIOD OVERFITTING:")
        lines.append("-" * 30)

        for period in periods:
            p_num = period['period_number']
            analysis = period['overfitting_analysis']
            risk = analysis['overfitting_risk_level']

            lines.append(f"Period {p_num}: {risk} risk ({analysis['overfitting_score']:.1f} score)")
            lines.append(f"  P&L Decay: {analysis['performance_decay_percent']:+.1f}%")
            lines.append(f"  Win Rate Decay: {analysis['win_rate_decay_percent']:+.1f}%")

        return "\n".join(lines)

    def _generate_walk_forward_recommendations(self, overall_stats: Dict) -> List[str]:
        """Generate recommendations based on walk forward analysis"""
        recommendations = []

        if not overall_stats or 'error' in overall_stats:
            return ["Unable to generate recommendations due to insufficient data"]

        # Profitability recommendations
        profitability = overall_stats.get('profitable_periods_percent', 0)
        if profitability >= 70:
            recommendations.append("✅ Excellent consistency - strategy shows strong out-of-sample performance")
        elif profitability >= 50:
            recommendations.append("⚠️ Moderate consistency - strategy performs in most periods but monitor closely")
        else:
            recommendations.append("❌ Poor consistency - strategy fails in most out-of-sample periods")

        # Risk recommendations
        pnl_consistency = overall_stats.get('pnl_consistency', 0)
        if pnl_consistency < overall_stats.get('average_monthly_pnl', 0) * 0.5:
            recommendations.append("✅ Good P&L stability across periods")
        else:
            recommendations.append("⚠️ High P&L volatility - consider risk management adjustments")

        # Win rate recommendations
        avg_win_rate = overall_stats.get('average_win_rate', 0)
        win_rate_consistency = overall_stats.get('win_rate_consistency', 0)

        if avg_win_rate > 55 and win_rate_consistency < 10:
            recommendations.append("✅ Strong and consistent win rate across periods")
        elif avg_win_rate > 50:
            recommendations.append("⚠️ Decent win rate but inconsistent - may need parameter tuning")
        else:
            recommendations.append("❌ Weak win rate - strategy may need fundamental changes")

        # Sample size recommendations
        total_periods = overall_stats.get('total_periods', 0)
        if total_periods >= 8:
            recommendations.append("✅ Sufficient walk forward periods for reliable analysis")
        elif total_periods >= 4:
            recommendations.append("⚠️ Moderate number of periods - consider adding more data")
        else:
            recommendations.append("❌ Insufficient periods - add more historical data for better analysis")

        return recommendations

    def _get_default_parameter_ranges(self, strategy_name: str, market: str) -> Dict[str, List]:
        """Get default parameter ranges for optimization based on strategy"""

        ranges = {}

        if strategy_name == 'trend_following':
            ranges = {
                'short_ma_period': [5, 20, 1],      # Test MA periods from 5 to 20
                'long_ma_period': [20, 50, 5],      # Test longer MA periods
                'stop_loss_pct': [0.01, 0.05, 0.01], # 1% to 5% stop loss
                'take_profit_pct': [0.02, 0.10, 0.02] # 2% to 10% take profit
            }
        elif strategy_name == 'mean_reversion':
            ranges = {
                'window': [10, 30, 5],              # Lookback window
                'std_dev_multiplier': [1.5, 3.0, 0.5], # Standard deviation multiplier
                'max_holding_period': [5, 20, 5],   # Max bars to hold
                'stop_loss_pct': [0.005, 0.03, 0.005] # Tighter stops for mean reversion
            }
        elif strategy_name == 'breakout_trading':
            ranges = {
                'lookback_period': [10, 30, 5],     # Period to find highs/lows
                'breakout_threshold': [0.005, 0.03, 0.005], # Breakout percentage
                'confirmation_period': [1, 5, 1],   # Bars to confirm breakout
                'stop_loss_pct': [0.01, 0.04, 0.01]
            }
        elif strategy_name == 'momentum_trading':
            ranges = {
                'momentum_period': [5, 15, 1],      # Momentum calculation period
                'momentum_threshold': [0.01, 0.05, 0.01], # Momentum threshold
                'exit_threshold': [-0.02, 0.0, 0.01], # Exit when momentum drops
                'max_holding_period': [3, 10, 1]
            }
        elif strategy_name == 'range_trading':
            ranges = {
                'range_period': [20, 50, 5],        # Period to calculate range
                'range_multiplier': [0.7, 0.9, 0.1], # How close to range boundaries
                'stop_loss_pct': [0.005, 0.02, 0.005],
                'take_profit_pct': [0.01, 0.04, 0.01]
            }
        elif strategy_name == 'carry_trade':
            ranges = {
                'interest_diff_threshold': [0.01, 0.05, 0.01], # Min interest differential
                'max_holding_period': [30, 90, 15],  # Days to hold
                'stop_loss_pct': [0.02, 0.08, 0.02],
                'position_size_pct': [0.1, 0.5, 0.1]  # Position size as % of capital
            }
        else:
            # Generic ranges for unknown strategies
            ranges = {
                'stop_loss_pct': [0.01, 0.05, 0.01],
                'take_profit_pct': [0.02, 0.10, 0.02],
                'max_holding_period': [5, 20, 5]
            }

        return ranges

    def _generate_equation_summary(self, strategy_name: str, parameters: Dict[str, Any], market: str) -> str:
        """Generate a concise equation summary for the strategy"""
        summary = f"**{market.upper()} Strategy:** {strategy_name.replace('_', ' ').title()}\n\n"

        if market == 'nba':
            if strategy_name == 'player_points_over':
                summary += f"**Equation:** If Player_Points > {parameters.get('points_threshold', 25)}, WIN\n"
                summary += f"**Player:** {parameters.get('player_name', 'Unknown').title()}\n"
            elif strategy_name == 'team_win_streak':
                summary += f"**Equation:** If Win_Streak ≥ {parameters.get('min_streak', 3)}, bet on team\n"
            elif strategy_name == 'home_away_performance':
                summary += f"**Equation:** Bet on home teams with >60% home win rate\n"
            else:
                summary += f"**Equation:** Moneyline prediction with >55% win probability\n"

        elif market == 'forex':
            pair = parameters.get('pair', 'EURUSD')
            if strategy_name == 'trend_following':
                summary += f"**Equation:** If SMA(5) > SMA(20) on {pair}, BUY\n"
            elif strategy_name == 'range_trading':
                summary += f"**Equation:** Buy at 20th percentile, Sell at 80th percentile on {pair}\n"
            elif strategy_name == 'breakout_trading':
                summary += f"**Equation:** Enter on ±10% breakout beyond 20-day range on {pair}\n"
            elif strategy_name == 'mean_reversion':
                summary += f"**Equation:** If Z-Score < -2 or > 2 on {pair}, trade mean reversion\n"
            else:
                summary += f"**Equation:** Trend following on {pair}\n"

        elif market == 'crypto':
            symbol = parameters.get('symbol', 'BTC')
            if strategy_name == 'momentum_trading':
                summary += f"**Equation:** If 5-day return > 2% AND volume > 1.5× average on {symbol}, BUY\n"
            elif strategy_name == 'mean_reversion':
                summary += f"**Equation:** If Z-Score < -2 or > 2 on {symbol}, mean reversion trade\n"
            elif strategy_name == 'volume_analysis':
                summary += f"**Equation:** High volume breakouts on {symbol}\n"
            else:
                summary += f"**Equation:** Momentum signals on {symbol}\n"

        elif market == 'stocks':
            symbol = parameters.get('symbol', 'AAPL')
            if strategy_name == 'earnings_momentum':
                summary += f"**Equation:** If earnings surprise > 5% on {symbol}, BUY\n"
            elif strategy_name == 'volume_breakout':
                summary += f"**Equation:** If volume > 2× average AND price > 20-day high on {symbol}, BUY\n"
            elif strategy_name == 'gap_trading':
                summary += f"**Equation:** Fade ±3% gaps on {symbol}\n"
            else:
                summary += f"**Equation:** Trend following on {symbol}\n"

        summary += f"**Stake:** ${parameters.get('stake', 1000)} per trade\n"
        return summary

def main():
    """Main function for testing the universal backtesting engine"""
    import sys

    if len(sys.argv) > 1:
        # API mode - called from Next.js
        engine = UniversalBacktestingEngine()
        command = sys.argv[1]
        params = json.loads(sys.argv[2]) if len(sys.argv) > 2 else {}

        try:
            if command == "run_backtest":
                result = engine.run_backtest(
                    params['market'],
                    params['strategy_name'],
                    params.get('parameters', {}),
                    params.get('data_file'),
                    params.get('min_trades', 10),
                    params.get('timeframe', '1hour')
                )
            else:
                result = {"error": f"Unknown command: {command}"}

            # Convert numpy types to Python types for JSON serialization
            def convert_numpy_types(obj):
                if isinstance(obj, np.integer):
                    return int(obj)
                elif isinstance(obj, np.floating):
                    return float(obj)
                elif isinstance(obj, np.ndarray):
                    return obj.tolist()
                elif isinstance(obj, dict):
                    return {key: convert_numpy_types(value) for key, value in obj.items()}
                elif isinstance(obj, list):
                    return [convert_numpy_types(item) for item in obj]
                else:
                    return obj

            # Output JSON for API consumption
            print(json.dumps(convert_numpy_types(result)))

        except Exception as e:
            print(json.dumps({"error": str(e)}))
            sys.exit(1)

    else:
        # Test mode
        engine = UniversalBacktestingEngine()

        print("🌍 Universal Backtesting Engine Test")
        print("=" * 50)

        # Test forex strategy
        print("\n💱 Testing Forex Trend Following:")
        result = engine.run_backtest(
            'forex',
            'trend_following',
            {'pair': 'EURUSD', 'stake': 1000},
            min_trades=5
        )

        if result.get('success'):
            print(f"✅ Forex backtest completed: {result['total_trades']} trades")
            print(f"📊 Win Rate: {result['results']['win_rate']}%")
            print(f"💰 Total Profit: ${result['results']['total_profit']}")
        else:
            print(f"❌ Forex backtest failed: {result.get('error', 'Unknown error')}")

        # Test crypto strategy
        print("\n₿ Testing Crypto Momentum:")
        result2 = engine.run_backtest(
            'crypto',
            'momentum_trading',
            {'symbol': 'BTC', 'stake': 1000},
            min_trades=5
        )

        if result2.get('success'):
            print(f"✅ Crypto backtest completed: {result2['total_trades']} trades")
            print(f"📊 Win Rate: {result2['results']['win_rate']}%")
            print(f"💰 Total Profit: ${result2['results']['total_profit']}")
        else:
            print(f"❌ Crypto backtest failed: {result2.get('error', 'Unknown error')}")

        print("\n✅ Universal backtesting engine test completed!")

if __name__ == "__main__":
    main()
