#!/usr/bin/env python3
"""
Test script to demonstrate walk forward analysis functionality
"""

import json
import sys
import os

# Add the project root to Python path
sys.path.insert(0, os.path.dirname(__file__))

from universal_backtesting import UniversalBacktestingEngine

def test_walk_forward_analysis():
    """Test walk forward analysis functionality"""

    print("🧪 Testing Walk Forward Analysis - Anti-Overfitting Protection")
    print("=" * 70)

    engine = UniversalBacktestingEngine()

    # Test walk forward on XAUUSD with shorter periods to fit available data
    walk_forward_config = {
        'in_sample_months': 1,  # 1 month in-sample (training)
        'out_of_sample_months': 1,  # 1 month out-of-sample (testing)
        'step_months': 1,  # Move forward 1 month each period
        'start_date': '2025-08-01',  # Start from available data
        'end_date': '2025-11-21'  # End with available data
    }

    result = engine.run_backtest(
        market="forex",
        strategy_name="trend_following",
        parameters={"pair": "XAUUSD", "stake": 100},
        timeframe="1hour",
        min_trades=5,
        walk_forward_periods=walk_forward_config
    )

    if not result.get('success'):
        print(f"❌ Walk forward analysis failed: {result.get('error')}")
        return

    print("✅ Walk forward analysis completed successfully!")
    print(f"📊 Market: {result['market'].upper()}")
    print(f"📈 Strategy: Trend Following")
    print(f"⏰ Timeframe: 1-hour")
    print(f"💰 Stake per trade: $100")
    print()

    # Display walk forward results
    if 'walk_forward_analysis' in result:
        wf = result['walk_forward_analysis']

        print("🎯 WALK FORWARD ANALYSIS RESULTS")
        print("=" * 50)

        # Configuration
        if wf.get('configuration'):
            config = wf['configuration']
            print("Configuration:")
            print(f"  • In-sample period: {config['in_sample_months']} months")
            print(f"  • Out-of-sample period: {config['out_of_sample_months']} months")
            print(f"  • Step size: {config['step_months']} months")
            print(f"  • Total periods analyzed: {config['total_periods']}")
            print(f"  • Date range: {config['start_date']} to {config['end_date']}")
            print()

        # Overall statistics
        if wf.get('overall_statistics'):
            overall = wf['overall_statistics']
            print("Overall Performance Across All Periods:")
            print(".1f")
            print(".2f")
            print(f"  • Total P&L across all periods: ${overall['total_pnl']}")
            print(".1f")
            print(f"  • Best period P&L: ${overall['best_period_pnl']}")
            print(f"  • Worst period P&L: ${overall['worst_period_pnl']}")
            print(".2f")
            print()

        # Overfitting summary
        if wf.get('overfitting_summary'):
            print("Overfitting Analysis Summary:")
            print("-" * 30)
            print(wf['overfitting_summary'])
            print()

        # Recommendations
        if wf.get('recommendations'):
            print("Strategy Recommendations:")
            print("-" * 30)
            for rec in wf['recommendations']:
                print(f"• {rec}")
            print()

        # Sample periods
        if wf.get('periods') and len(wf['periods']) > 0:
            periods = wf['periods']
            print("Sample Period Analysis (First 2 & Last 2):")
            print("-" * 50)

            # First 2 periods
            first_periods = periods[:2]
            print("First 2 periods:")
            for period in first_periods:
                print(f"\nPeriod {period['period_number']}:")
                in_sample = period['in_sample_period']
                out_sample = period['out_of_sample_period']
                analysis = period['overfitting_analysis']

                print(f"  In-sample ({in_sample['start_date']} to {in_sample['end_date']}):")
                print(f"    • {in_sample['metrics']['total_trades']} trades, {in_sample['metrics']['win_rate']}% win rate")
                print(f"    • P&L: ${in_sample['metrics']['total_pnl']}")

                print(f"  Out-of-sample ({out_sample['start_date']} to {out_sample['end_date']}):")
                print(f"    • {out_sample['metrics']['total_trades']} trades, {out_sample['metrics']['win_rate']}% win rate")
                print(f"    • P&L: ${out_sample['metrics']['total_pnl']}")

                print(f"  Overfitting Risk: {analysis['overfitting_risk_level']} (Score: {analysis['overfitting_score']})")
                print(".1f")
                print(".1f")

            if len(periods) > 4:
                print(f"\n... {len(periods) - 4} more periods analyzed ...")

            # Last 2 periods
            if len(periods) > 2:
                last_periods = periods[-2:]
                print("\nLast 2 periods:")
                for period in last_periods:
                    print(f"\nPeriod {period['period_number']}:")
                    in_sample = period['in_sample_period']
                    out_sample = period['out_of_sample_period']
                    analysis = period['overfitting_analysis']

                    print(f"  In-sample ({in_sample['start_date']} to {in_sample['end_date']}):")
                    print(f"    • {in_sample['metrics']['total_trades']} trades, {in_sample['metrics']['win_rate']}% win rate")
                    print(f"    • P&L: ${in_sample['metrics']['total_pnl']}")

                    print(f"  Out-of-sample ({out_sample['start_date']} to {out_sample['end_date']}):")
                    print(f"    • {out_sample['metrics']['total_trades']} trades, {out_sample['metrics']['win_rate']}% win rate")
                    print(f"    • P&L: ${out_sample['metrics']['total_pnl']}")

                    print(f"  Overfitting Risk: {analysis['overfitting_risk_level']} (Score: {analysis['overfitting_score']})")
                    print(".1f")
                    print(".1f")

            print("\n🎯 Walk Forward Analysis Benefits:")
            print("• Tests strategy on unseen future data")
            print("• Prevents overfitting to historical patterns")
            print("• Validates robustness across different market conditions")
            print("• Provides realistic performance expectations")
            print("• Identifies optimal parameter stability")

    else:
        print("❌ No walk forward analysis results found")

    print("\n" + "=" * 70)
    print("✅ Walk Forward Analysis Complete - Anti-Overfitting Protection Active!")

if __name__ == "__main__":
    test_walk_forward_analysis()
