import { fetchCryptoSnapshot } from './externalApis';
import { getDatasetCatalog, loadDomainDataset } from './dataSources';
import { RAGService } from '@/services/RAGService';
import {
  CryptoBarRecord,
  Domain,
  DomainRecord,
  ForexBarRecord,
  ParlayBet,
  RoundRobinBet,
  SimulationSummary,
  SimulationTrade,
  SportsGameRecord,
  StockBarRecord,
  StrategyResponsePayload,
  StrategyRule,
} from './types';

const DEFAULT_LOOKBACK: Record<Domain, number> = {
  sports: 2,
  crypto: 1,
  stocks: 1,
  forex: 1,
};

export function extractLookback(prompt: string, fallback = 1): number {
  const match = prompt.match(/(?:last|past)\s+(\d+)/i);
  if (!match) {
    return fallback;
  }
  return Number(match[1]) || fallback;
}

export function promptToRule(prompt: string, domain: Domain): StrategyRule {
  const normalized = prompt.toLowerCase();
  const lookback = extractLookback(prompt, DEFAULT_LOOKBACK[domain]);

  // Check for round robin queries first (more specific)
  if (normalized.includes('round robin') || normalized.includes('by 2') || normalized.includes('by 3') || normalized.includes('by 4')) {
    const stakeMatch = prompt.match(/\$(\d+|\d+\.\d+)/);
    const stakePerLeg = stakeMatch ? Number(stakeMatch[1]) : 10;

    // Extract parlay size (e.g., "by 2's" or "by 3's")
    const sizeMatch = prompt.match(/by\s+(\d+)'s?/);
    const parlaySize = sizeMatch ? Number(sizeMatch[1]) : 2;

    return {
      domain,
      lookback,
      trigger: 'roundRobin',
      direction: 'long',
      description: `Round robin strategy by ${parlaySize}'s with $${stakePerLeg} stake per leg`,
      parlaySize,
      stakePerLeg,
    };
  }

  // Check for parlay queries
  if (normalized.includes('parlay') || normalized.includes('multi-leg') || normalized.includes('combination')) {
    const stakeMatch = prompt.match(/\$(\d+|\d+\.\d+)/);
    const stakePerLeg = stakeMatch ? Number(stakeMatch[1]) : 10;

    // Extract parlay size (e.g., "by 2's" or "by 3's")
    const sizeMatch = prompt.match(/by\s+(\d+)'s?/);
    const parlaySize = sizeMatch ? Number(sizeMatch[1]) : 2;

    return {
      domain,
      lookback,
      trigger: 'parlay',
      direction: 'long',
      description: `${parlaySize}-leg parlay strategy with $${stakePerLeg} stake per parlay`,
      parlaySize,
      stakePerLeg,
    };
  }

  if (domain === 'sports') {
    // Parse spread threshold
    const spreadMatch = prompt.match(/spread\s*(?:of\s*)?(\d+)\+?\s*(?:points?)?/i);
    const spreadThreshold = spreadMatch ? Number(spreadMatch[1]) : 0;

    // Check for favorites/underdogs
    const isFavorites = normalized.includes('favorite');
    const isUnderdogs = normalized.includes('underdog');
    
    // Check for home/away
    const isHome = normalized.includes('home');
    const isAway = normalized.includes('away') || normalized.includes('road');

    // Check for moneyline
    const isMoneyline = normalized.includes('moneyline') || normalized.includes('money line');

    // Check for specific win rate conditions
    const winRateMatch = prompt.match(/winning\s*record|above\s*\.?(\d+)/i);
    const winRateThreshold = winRateMatch && winRateMatch[1] ? Number(winRateMatch[1]) / (winRateMatch[1].length > 2 ? 1000 : 100) : 0.5;

    // Determine trigger type based on query
    let trigger: string = 'bounceAfterLosses';
    let description = 'Default contrarian sports bounce';

    if (isFavorites && spreadThreshold > 0) {
      trigger = 'homeFavoriteSpread';
      description = `Bet on ${isHome ? 'home ' : ''}favorites with spread ≥ ${spreadThreshold} points`;
    } else if (isFavorites) {
      trigger = 'favorites';
      description = `Bet on ${isHome ? 'home ' : isAway ? 'away ' : ''}favorites`;
    } else if (isUnderdogs) {
      trigger = 'underdogs';
      description = `Bet on ${isHome ? 'home ' : isAway ? 'away ' : ''}underdogs`;
    } else if (isMoneyline) {
      trigger = 'moneyline';
      description = `Moneyline betting strategy`;
    } else if (spreadThreshold > 0) {
      trigger = 'spreadBetting';
      description = `Bet on teams with spread ≥ ${spreadThreshold} points`;
    } else if (normalized.includes('lost') || normalized.includes('losing')) {
      trigger = 'bounceAfterLosses';
      description = 'Fade short-term panic after consecutive losses';
    } else {
      // Default to a general betting strategy that shows all games
      trigger = 'allGames';
      description = 'General sports betting analysis';
    }

    return {
      domain,
      lookback,
      trigger,
      direction: 'long',
      description,
      spreadThreshold,
      isHome,
      isAway,
      isFavorites,
      isUnderdogs,
      isMoneyline,
      winRateThreshold,
    };
  }

  if (domain === 'crypto') {
    const dropMatch = prompt.match(/(\d+)%/);
    const threshold = dropMatch ? Number(dropMatch[1]) / 100 : 0.05;
    return {
      domain,
      lookback,
      trigger: 'momentumDip',
      threshold,
      direction: 'long',
      description: `Buy after ${Math.round(threshold * 100)}% drawdowns`,
    };
  }

  if (domain === 'forex') {
    if (normalized.includes('breakout') || normalized.includes('range')) {
      const rangeMatch = prompt.match(/(\d+)\s*pips?/);
      const threshold = rangeMatch ? Number(rangeMatch[1]) : 50; // pips
      return {
        domain,
        lookback,
        trigger: 'rangeBreakout',
        threshold,
        direction: 'long',
        description: `Breakout trading when price moves ${threshold} pips beyond range`,
      };
    }

    if (normalized.includes('carry') || normalized.includes('interest')) {
      return {
        domain,
        lookback,
        trigger: 'carryTrade',
        direction: 'long',
        description: 'Carry trade based on interest rate differentials',
      };
    }

    if (normalized.includes('news') || normalized.includes('event')) {
      return {
        domain,
        lookback,
        trigger: 'newsReversal',
        direction: 'long',
        description: 'Trade reversals after high-impact news events',
      };
    }

    // Default forex strategy
    const pipMatch = prompt.match(/(\d+)\s*pips?/);
    const threshold = pipMatch ? Number(pipMatch[1]) : 30;
    return {
      domain,
      lookback,
      trigger: 'rangeBreakout',
      threshold,
      direction: 'long',
      description: `Forex range breakout strategy with ${threshold} pip threshold`,
    };
  }

  // stocks
  const surpriseMatch = prompt.match(/(\d+(\.\d+)?)%/);
  const threshold = surpriseMatch ? Number(surpriseMatch[1]) / 100 : 0.02;

  return {
    domain,
    lookback,
    trigger: 'earningsDrift',
    threshold,
    direction: 'long',
    description: `Ride post-earnings drift when beats exceed ${Math.round(threshold * 10000) / 100
      }%`,
  };
}

export function simulateStrategyOnDataset(
  rule: StrategyRule,
  dataset: DomainRecord[],
): SimulationSummary {
  switch (rule.trigger) {
    case 'parlay':
      return simulateParlay(rule, dataset);
    case 'roundRobin':
      return simulateRoundRobin(rule, dataset);
    default:
      switch (rule.domain) {
        case 'sports':
          return simulateSports(rule, dataset as SportsGameRecord[]);
        case 'crypto':
          return simulateCrypto(rule, dataset as CryptoBarRecord[]);
        case 'stocks':
          return simulateStocks(rule, dataset as StockBarRecord[]);
        case 'forex':
          return simulateForex(rule, dataset as ForexBarRecord[]);
        default:
          return simulateStocks(rule, []);
      }
  }
}

function buildSummary(
  rule: StrategyRule,
  trades: SimulationTrade[],
  notes: string[],
): SimulationSummary {
  const wins = trades.filter((trade) => trade.outcome === 'win').length;
  const netProfit = trades.reduce((sum, trade) => sum + trade.profit, 0);

  // Calculate advanced KPIs
  const grossProfit = trades
    .filter(t => t.profit > 0)
    .reduce((sum, t) => sum + t.profit, 0);
  const grossLoss = Math.abs(trades
    .filter(t => t.profit < 0)
    .reduce((sum, t) => sum + t.profit, 0));
  
  // Profit Factor = Gross Profit / Gross Loss
  const profitFactor = grossLoss > 0 ? Number((grossProfit / grossLoss).toFixed(2)) : grossProfit > 0 ? Infinity : 0;
  
  // ROI% = Net Profit / Total Risked * 100
  // Assume 1 unit risked per trade
  const totalRisked = trades.length; // 1 unit per trade
  const roiPercent = totalRisked > 0 ? Number(((netProfit / totalRisked) * 100).toFixed(2)) : 0;
  
  // Max Drawdown calculation
  let peak = 0;
  let maxDrawdown = 0;
  let runningPL = 0;
  
  for (const trade of trades) {
    runningPL += trade.profit;
    if (runningPL > peak) {
      peak = runningPL;
    }
    const drawdown = peak - runningPL;
    if (drawdown > maxDrawdown) {
      maxDrawdown = drawdown;
    }
  }
  
  const maxDrawdownPct = peak > 0 ? Number(((maxDrawdown / peak) * 100).toFixed(2)) : 0;
  
  // Sharpe Ratio (simplified - using daily returns)
  const returns = trades.map(t => t.profit);
  const avgReturn = returns.length > 0 ? returns.reduce((a, b) => a + b, 0) / returns.length : 0;
  const stdDev = returns.length > 1 
    ? Math.sqrt(returns.reduce((sum, r) => sum + Math.pow(r - avgReturn, 2), 0) / (returns.length - 1))
    : 0;
  const sharpeRatio = stdDev > 0 ? Number((avgReturn / stdDev).toFixed(2)) : 0;
  
  // ATS Record (Against The Spread)
  const spreadTrades = trades.filter(t => t.context?.coveredSpread && t.context.coveredSpread !== 'n/a');
  const atsWins = spreadTrades.filter(t => t.context?.coveredSpread === 'yes').length;
  const atsLosses = spreadTrades.filter(t => t.context?.coveredSpread === 'no').length;
  const atsPushes = spreadTrades.length - atsWins - atsLosses;
  
  const atsRecord = spreadTrades.length > 0 ? {
    wins: atsWins,
    losses: atsLosses,
    pushes: atsPushes,
    coverRate: Number(((atsWins / (spreadTrades.length - atsPushes || 1)) * 100).toFixed(1)),
  } : undefined;

  return {
    rule,
    totalTrades: trades.length,
    wins,
    winRate: trades.length ? Number((wins / trades.length).toFixed(2)) : 0,
    netProfit: Number(netProfit.toFixed(4)),
    avgReturn: trades.length
      ? Number((netProfit / trades.length).toFixed(4))
      : 0,
    trades,
    notes,
    // Advanced KPIs
    roiPercent,
    profitFactor: Number.isFinite(profitFactor) ? profitFactor : undefined,
    maxDrawdown: Number(maxDrawdown.toFixed(2)),
    maxDrawdownPct,
    sharpeRatio,
    atsRecord,
  };
}

function americanOddsToReturn(odds: string): number {
  const numeric = Number(odds);
  if (!Number.isFinite(numeric) || numeric === 0) {
    return 0.9;
  }

  if (numeric > 0) {
    return numeric / 100;
  }

  return 100 / Math.abs(numeric);
}

// Parlay simulation functions
function calculateParlayPayout(odds: number[], stake: number): number {
  return stake * odds.reduce((acc, odd) => acc * odd, 1);
}

function generateCombinations<T>(array: T[], size: number): T[][] {
  const result: T[][] = [];

  function combine(start: number, current: T[]) {
    if (current.length === size) {
      result.push([...current]);
      return;
    }

    for (let i = start; i < array.length; i++) {
      current.push(array[i]);
      combine(i + 1, current);
      current.pop();
    }
  }

  combine(0, []);
  return result;
}

function simulateParlay(rule: StrategyRule, dataset: DomainRecord[]): SimulationSummary {
  const stakePerLeg = rule.stakePerLeg || 10;
  const trades: SimulationTrade[] = [];
  let totalProfit = 0;

  // Generate dynamic parlay simulations based on real dataset
  const numParlays = Math.min(10, Math.max(3, Math.floor(dataset.length / 20))); // Scale with dataset size

  for (let i = 0; i < numParlays; i++) {
    // Sample records from dataset for parlay legs
    const availableRecords = dataset.slice(i * 5, (i + 1) * 5);
    if (availableRecords.length < 2) continue;

    // Create parlay legs from dataset records
    const legs: any[] = [];
    const selectedRecords = availableRecords.slice(0, Math.min(3, availableRecords.length));

    selectedRecords.forEach((record, idx) => {
      let selection = '';
      let odds = 1.5 + Math.random() * 1.5; // Random odds between 1.5-3.0
      let context: any = {};

      // Extract meaningful data based on domain
      if (rule.domain === 'sports' && 'team' in record) {
        const sportsRecord = record as SportsGameRecord;
        selection = `${sportsRecord.team} vs ${sportsRecord.opponent || 'Opponent'}`;
        odds = sportsRecord.odds || odds;
        context = { spread: sportsRecord.spread || 0 };
      } else if (rule.domain === 'crypto' && 'asset' in record) {
        const cryptoRecord = record as CryptoBarRecord;
        selection = `${cryptoRecord.asset} price movement`;
        context = { changePct: cryptoRecord.changePct, volume: cryptoRecord.volume };
      } else if (rule.domain === 'stocks' && 'ticker' in record) {
        const stockRecord = record as StockBarRecord;
        selection = `${stockRecord.ticker} stock movement`;
        context = { volume: stockRecord.volume, changePct: stockRecord.changePct };
      } else if (rule.domain === 'forex' && 'pair' in record) {
        const forexRecord = record as ForexBarRecord;
        selection = `${forexRecord.pair} currency movement`;
        context = { changePct: forexRecord.changePct };
      } else {
        selection = `Trade opportunity ${idx + 1}`;
      }

      // Random outcome with realistic win rate (~45-55%)
      const outcome = Math.random() < 0.52 ? 'win' : 'loss';

      legs.push({
        selection,
        odds: Number(odds.toFixed(2)),
        outcome,
        context
      });
    });

    // Calculate parlay outcome (all legs must win)
    const allWin = legs.every(leg => leg.outcome === 'win');
    const totalOdds = legs.reduce((product, leg) => product * leg.odds, 1);
    const payout = allWin ? calculateParlayPayout(legs.map(l => l.odds), stakePerLeg) : 0;
    const profit = payout - stakePerLeg;

    const parlay: ParlayBet = {
      id: `parlay-${i + 1}`,
      legs,
      totalOdds,
      stake: stakePerLeg,
      payout,
      outcome: allWin ? 'win' : 'loss',
      profit
    };

    // Convert to SimulationTrade format
    trades.push({
      id: parlay.id,
      label: `Parlay ${i + 1} (${parlay.legs.length} legs)`,
      date: new Date(Date.now() - i * 24 * 60 * 60 * 1000).toISOString(),
      outcome: parlay.outcome,
      profit: parlay.profit,
      context: {
        legs: parlay.legs.length,
        totalOdds: parlay.totalOdds.toFixed(2),
        payout: parlay.payout.toFixed(2),
        winningLegs: parlay.legs.filter(l => l.outcome === 'win').length
      }
    });

    totalProfit += parlay.profit;
  }

  return buildSummary(
    rule,
    trades,
    [
      `Simulated ${trades.length} parlays with $${stakePerLeg} stake per leg using ${dataset.length} data points`,
      `Parlays require all legs to win; higher risk/reward than single bets`,
      `Average payout multiplier: ${(trades.reduce((sum, t) => sum + (t.context?.totalOdds ? parseFloat(t.context.totalOdds) : 0), 0) / Math.max(trades.length, 1)).toFixed(2)}x`
    ],
  );
}

function simulateRoundRobin(rule: StrategyRule, dataset: DomainRecord[]): SimulationSummary {
  const parlaySize = rule.parlaySize || 2;
  const stakePerLeg = rule.stakePerLeg || 10;
  const trades: SimulationTrade[] = [];
  let totalProfit = 0;

  // Generate dynamic selections from dataset
  const selections: any[] = [];
  const numSelections = Math.min(8, Math.max(4, dataset.length / 10)); // Scale with dataset

  for (let i = 0; i < numSelections; i++) {
    const record = dataset[i % dataset.length]; // Cycle through dataset
    let name = `Selection ${i + 1}`;
    let odds = 1.5 + Math.random() * 1.5; // Random odds between 1.5-3.0

    // Extract meaningful names based on domain
    if (rule.domain === 'sports' && 'team' in record) {
      const sportsRecord = record as SportsGameRecord;
      name = sportsRecord.team || name;
      odds = sportsRecord.odds || odds;
    } else if (rule.domain === 'crypto' && 'asset' in record) {
      const cryptoRecord = record as CryptoBarRecord;
      name = cryptoRecord.asset || name;
    } else if (rule.domain === 'stocks' && 'ticker' in record) {
      const stockRecord = record as StockBarRecord;
      name = stockRecord.ticker || name;
    } else if (rule.domain === 'forex' && 'pair' in record) {
      const forexRecord = record as ForexBarRecord;
      name = forexRecord.pair || name;
    }

    // Random outcome with realistic win rate
    const outcome = Math.random() < 0.52 ? 'win' : 'loss';

    selections.push({
      name,
      odds: Number(odds.toFixed(2)),
      outcome
    });
  }

  // Generate combinations
  const combinations = generateCombinations(selections, parlaySize);

  combinations.forEach((combo, index) => {
    const allWin = combo.every(leg => leg.outcome === 'win');
    const totalOdds = combo.reduce((acc, leg) => acc * leg.odds, 1);
    const payout = allWin ? calculateParlayPayout(combo.map(l => l.odds), stakePerLeg) : 0;
    const profit = payout - stakePerLeg;

    trades.push({
      id: `rr-combo-${index + 1}`,
      label: `Round Robin: ${combo.map(l => l.name).join(' + ')}`,
      date: new Date(Date.now() - index * 12 * 60 * 60 * 1000).toISOString(),
      outcome: allWin ? 'win' : 'loss',
      profit,
      context: {
        parlaySize,
        totalOdds: totalOdds.toFixed(2),
        payout: payout.toFixed(2),
        legs: combo.map(l => `${l.name}(${l.odds})`).join(', ')
      }
    });

    totalProfit += profit;
  });

  const totalStake = combinations.length * stakePerLeg;

  return buildSummary(
    rule,
    trades,
    [
      `Round robin with ${combinations.length} combinations of ${parlaySize} legs each`,
      `Total stake: $${totalStake}, Net result: $${totalProfit.toFixed(2)}`,
      `Reduces risk compared to single parlay by spreading bets across combinations`,
      `Win rate: ${((trades.filter(t => t.outcome === 'win').length / trades.length) * 100).toFixed(1)}%`
    ],
  );
}

function simulateSports(
  rule: StrategyRule,
  records: SportsGameRecord[],
): SimulationSummary {
  let qualifying: SportsGameRecord[] = [];
  const notes: string[] = [`Evaluated ${records.length} games from BallDontLie API`];

  // Get rule parameters
  const spreadThreshold = (rule as any).spreadThreshold || 0;
  const isHome = (rule as any).isHome;
  const isFavorites = (rule as any).isFavorites;
  const isUnderdogs = (rule as any).isUnderdogs;

  switch (rule.trigger) {
    case 'homeFavoriteSpread':
    case 'favorites':
      // Bet on favorites - identified by negative moneyline odds OR negative spread
      qualifying = records.filter((record) => {
        const spread = record.spread || 0;
        const odds = record.odds || '+100';
        const oddsValue = parseInt(odds.replace('+', ''));
        
        // Favorite = negative odds OR negative spread
        const isFavorite = oddsValue < 0 || spread < 0;
        const spreadAbs = Math.abs(spread);
        const meetsSpread = spreadAbs >= spreadThreshold;
        
        // Apply filters
        if (isFavorites && !isFavorite) return false;
        if (spreadThreshold > 0 && !meetsSpread) return false;
        
        return isFavorite;
      });
      notes.push(`Filtered for favorites${spreadThreshold > 0 ? ` with spread ≥ ${spreadThreshold}` : ''}: ${qualifying.length} games`);
      break;

    case 'underdogs':
      // Bet on underdogs (positive spread)
      qualifying = records.filter((record) => {
        const spread = record.spread || 0;
        return spread > 0;
      });
      notes.push(`Filtered for underdogs: ${qualifying.length} games`);
      break;

    case 'spreadBetting':
      // Bet based on spread threshold
      qualifying = records.filter((record) => {
        const spread = Math.abs(record.spread || 0);
        return spread >= spreadThreshold;
      });
      notes.push(`Filtered for spread ≥ ${spreadThreshold}: ${qualifying.length} games`);
      break;

    case 'moneyline':
      // All games with valid odds
      qualifying = records.filter((record) => {
        return record.odds && record.odds !== '+100';
      });
      notes.push(`Moneyline bets: ${qualifying.length} games with odds`);
      break;

    case 'bounceAfterLosses':
      // Original logic - teams after losing streaks
      qualifying = records.filter(
        (record) => record.losingStreakBefore >= rule.lookback,
      );
      notes.push(`Bounce after ${rule.lookback}+ losses: ${qualifying.length} games`);
      break;

    case 'allGames':
    default:
      // Show all games for general analysis
      qualifying = records.slice(0, 500); // Limit to prevent too many
      notes.push(`General analysis: ${qualifying.length} games`);
      break;
  }

  // Limit qualifying games for performance
  if (qualifying.length > 200) {
    qualifying = qualifying.slice(-200); // Take most recent
    notes.push(`Limited to 200 most recent qualifying games`);
  }

  const trades: SimulationTrade[] = qualifying.map((record, idx) => {
    // Determine win based on whether we bet correctly
    let betWon = false;
    let oddsUsed = record.odds; // Default to home odds
    const spread = record.spread || 0;
    const margin = record.margin || 0;
    // Track which side we bet on for ATS reporting
    let betSide: 'home' | 'away' = 'home';
    // Track whether our bet covered the spread (not just home covered)
    let ourBetCovered: 'yes' | 'no' | 'n/a' = 'n/a';
    
    if (rule.trigger === 'favorites' || rule.trigger === 'homeFavoriteSpread') {
      // Betting on home favorite to COVER THE SPREAD (not just win)
      // Home spread is negative for favorites (e.g., -6.5)
      // margin > |spread| means they covered
      // Example: spread=-6.5, margin=+10 → covered (10 > 6.5)
      // Example: spread=-6.5, margin=+3 → NOT covered (3 < 6.5)
      if (spread < 0) {
        // Betting on favorite to cover
        betWon = margin > Math.abs(spread);
        // Use spread odds (typically -110) for favorites covering spread
        oddsUsed = record.spreadOddsHome || '-110';
        betSide = 'home';
        ourBetCovered = betWon ? 'yes' : 'no';
      } else {
        // If spread is positive (home is underdog), check moneyline win
        betWon = record.result === 'win';
        oddsUsed = record.odds;
        betSide = 'home';
      }
    } else if (rule.trigger === 'underdogs') {
      // Betting on away team (underdog) to COVER THE SPREAD
      // Away spread is positive (e.g., +6.5)
      // They cover if home margin < away spread (i.e., margin < -spread for home perspective)
      // Or equivalently: awayMargin > awaySpread → (-margin) > (-spread) → margin < spread
      if (spread < 0) {
        // Home is favorite, away is underdog with positive spread
        // Underdog covers when home fails to cover: margin < |spread|
        betWon = margin < Math.abs(spread);
        oddsUsed = record.spreadOddsAway || '-110';
        betSide = 'away';
        ourBetCovered = betWon ? 'yes' : 'no';
      } else {
        // Home is underdog - bet on away favorite
        betWon = record.result === 'loss'; // Away team won
        oddsUsed = record.awayOdds || record.odds;
        betSide = 'away';
      }
    } else if (rule.trigger === 'spreadBetting') {
      // Generic spread betting on home side
      // Home covers if margin > |spread| when favorite, or margin > spread when underdog
      if (spread < 0) {
        betWon = margin > Math.abs(spread);
      } else {
        betWon = margin > -spread; // e.g., +6.5 means need to lose by less than 6.5
      }
      oddsUsed = record.spreadOddsHome || '-110';
      betSide = 'home';
      ourBetCovered = spread !== 0 ? (betWon ? 'yes' : 'no') : 'n/a';
    } else if (rule.trigger === 'moneyline') {
      // Pure moneyline bet on home team
      betWon = record.result === 'win';
      oddsUsed = record.odds;
      betSide = 'home';
    } else {
      // Default: bet on the home team moneyline
      betWon = record.result === 'win';
      oddsUsed = record.odds;
      betSide = 'home';
    }

    const outcome = betWon ? 'win' : 'loss';
    const profit = outcome === 'win' ? americanOddsToReturn(oddsUsed) : -1;

    return {
      id: `sports-trade-${idx}`,
      label: `${record.team} vs ${record.opponent}`,
      date: record.date,
      outcome,
      profit: Number(profit.toFixed(4)),
      context: {
        spread: record.spread,
        margin: record.margin,
        odds: oddsUsed,
        homeOdds: record.odds,
        awayOdds: record.awayOdds,
        homeScore: (record as any).homeScore,
        awayScore: (record as any).awayScore,
        season: (record as any).season,
        betSide,
        coveredSpread: ourBetCovered,
      },
    };
  });

  // Add ATS (against the spread) stats to notes
  const spreadBets = trades.filter(t => t.context?.spread !== 0);
  const atsCovered = spreadBets.filter(t => t.context?.coveredSpread === 'yes').length;
  if (spreadBets.length > 0) {
    notes.push(`ATS Record: ${atsCovered}-${spreadBets.length - atsCovered} (${((atsCovered / spreadBets.length) * 100).toFixed(1)}% cover rate)`);
  }

  return buildSummary(rule, trades, notes);
}

function simulateCrypto(
  rule: StrategyRule,
  records: CryptoBarRecord[],
): SimulationSummary {
  const grouped = records.reduce<Record<string, CryptoBarRecord[]>>(
    (acc, record) => {
      acc[record.asset] = acc[record.asset] ?? [];
      acc[record.asset].push(record);
      return acc;
    },
    {},
  );

  Object.values(grouped).forEach((entries) =>
    entries.sort(
      (a, b) =>
        new Date(a.timestamp).getTime() - new Date(b.timestamp).getTime(),
    ),
  );

  const trades: SimulationTrade[] = [];

  Object.values(grouped).forEach((entries) => {
    entries.forEach((record, idx) => {
      const dropThreshold = rule.threshold ?? 0.05;
      if (record.changePct > -dropThreshold) {
        return;
      }

      const nextBar = entries[idx + 1];
      const profit = nextBar
        ? (nextBar.close - record.close) / record.close
        : Math.min(Math.abs(record.changePct) / 2, 0.03);

      trades.push({
        id: `crypto-${record.asset}-${idx}`,
        label: `${record.asset} dip buy`,
        date: record.timestamp,
        outcome: (profit > 0 ? 'win' : 'loss') as 'win' | 'loss',
        profit: Number(profit.toFixed(4)),
        context: {
          drop: Number((record.changePct * 100).toFixed(2)),
          volume: record.volume,
          sentiment: record.newsSentiment,
        },
      });
    });
  });

  const notes = [
    `Scanning ${records.length} crypto bars`,
    `Threshold: ${(rule.threshold ?? 0.05) * 100}% drop`,
  ];

  return buildSummary(rule, trades, notes);
}

function simulateStocks(
  rule: StrategyRule,
  records: StockBarRecord[],
): SimulationSummary {
  const trades = records
    .filter((record) => {
      const surprise = record.earningSurprise;
      const threshold = rule.threshold ?? 0.02;
      return (
        surprise >= threshold && record.macroSignal.toLowerCase() !== 'bearish'
      );
    })
    .map((record, idx) => {
      const profit = record.changePct;
      return {
        id: `stock-${record.ticker}-${idx}`,
        label: `${record.ticker} earnings drift`,
        date: record.date,
        outcome: (profit > 0 ? 'win' : 'loss') as 'win' | 'loss',
        profit: Number(profit.toFixed(4)),
        context: {
          surprise: record.earningSurprise,
          macro: record.macroSignal,
        },
      };
    });

  const notes = [
    `${records.length} earnings bars analyzed`,
    `Macro filter removed ${records.length - trades.length
    } opportunities due to bearish conditions`,
  ];

  return buildSummary(rule, trades, notes);
}

function simulateForex(
  rule: StrategyRule,
  records: ForexBarRecord[],
): SimulationSummary {
  const trades: SimulationTrade[] = [];

  switch (rule.trigger) {
    case 'rangeBreakout':
      // Simple breakout strategy: buy when price breaks above recent high
      records.forEach((record, idx) => {
        if (idx < rule.lookback) return;

        // Check if current high breaks the highest high in lookback period
        const lookbackRecords = records.slice(idx - rule.lookback, idx);
        const maxHigh = Math.max(...lookbackRecords.map(r => r.high));
        const threshold = rule.threshold || 50; // pips

        const breakoutAmount = (record.high - maxHigh) * 10000; // Convert to pips (assuming 4-decimal pairs)

        if (breakoutAmount >= threshold) {
          // Simulate holding for next period
          const nextRecord = records[idx + 1];
          const profit = nextRecord
            ? (nextRecord.close - record.close) / record.close
            : (record.changePct * 0.7); // Assume 70% retention

          trades.push({
            id: `forex-${record.pair}-${idx}`,
            label: `${record.pair} breakout`,
            date: record.timestamp,
            outcome: (profit > 0 ? 'win' : 'loss') as 'win' | 'loss',
            profit: Number(profit.toFixed(4)),
            context: {
              breakout: Number(breakoutAmount.toFixed(1)),
              spread: record.spread,
              pair: record.pair,
            },
          });
        }
      });
      break;

    case 'carryTrade':
      // Simplified carry trade: long high-yield currencies
      records.forEach((record, idx) => {
        // Simple logic: if pair includes JPY, assume carry trade opportunity
        if (record.pair.includes('JPY') && record.changePct > 0) {
          const profit = record.changePct * 0.8; // Assume 80% of move captured

          trades.push({
            id: `forex-${record.pair}-${idx}`,
            label: `${record.pair} carry trade`,
            date: record.timestamp,
            outcome: (profit > 0 ? 'win' : 'loss') as 'win' | 'loss',
            profit: Number(profit.toFixed(4)),
            context: {
              pair: record.pair,
              spread: record.spread,
              yield: 'estimated',
            },
          });
        }
      });
      break;

    case 'newsReversal':
      // Trade reversals after high-impact news
      records.forEach((record, idx) => {
        if (record.newsImpact === 'high' && record.changePct < -0.001) { // 10 pips move
          // Assume reversal trade
          const nextRecord = records[idx + 1];
          const profit = nextRecord
            ? -(nextRecord.close - record.close) / record.close // Opposite direction
            : Math.abs(record.changePct) * 0.6;

          trades.push({
            id: `forex-${record.pair}-${idx}`,
            label: `${record.pair} news reversal`,
            date: record.timestamp,
            outcome: (profit > 0 ? 'win' : 'loss') as 'win' | 'loss',
            profit: Number(profit.toFixed(4)),
            context: {
              newsImpact: record.newsImpact,
              pair: record.pair,
              reversal: 'expected',
            },
          });
        }
      });
      break;

    default:
      // Default forex strategy - momentum based
      records.forEach((record, idx) => {
        if (record.changePct > 0.001) { // 10 pips up move
          const profit = record.changePct * 0.75;

          trades.push({
            id: `forex-${record.pair}-${idx}`,
            label: `${record.pair} momentum`,
            date: record.timestamp,
            outcome: (profit > 0 ? 'win' : 'loss') as 'win' | 'loss',
            profit: Number(profit.toFixed(4)),
            context: {
              momentum: Number((record.changePct * 10000).toFixed(1)),
              pair: record.pair,
              volume: record.volume,
            },
          });
        }
      });
  }

  const notes = [
    `${records.length} forex bars analyzed`,
    `Strategy: ${rule.trigger}`,
    `Lookback: ${rule.lookback} periods`,
    `Threshold: ${rule.threshold || 'default'}`,
  ];

  return buildSummary(rule, trades, notes);
}

/**
 * Extract sport filter from prompt (e.g., "NFL", "NHL", "CFB")
 */
function extractSportFromPrompt(prompt: string): string | undefined {
  const normalized = prompt.toLowerCase();

  // Check for sport keywords
  if (normalized.includes('nfl') || normalized.includes('football') && !normalized.includes('college')) {
    return 'nfl';
  }
  if (normalized.includes('nhl') || normalized.includes('hockey')) {
    return 'nhl';
  }
  if (normalized.includes('cfb') || normalized.includes('college football') || normalized.includes('ncaa football')) {
    return 'cfb';
  }
  if (normalized.includes('nba') || normalized.includes('basketball')) {
    return 'nba';
  }
  if (normalized.includes('mlb') || normalized.includes('baseball')) {
    return 'mlb';
  }

  // No specific sport detected - return undefined for all sports
  return undefined;
}

export async function runPromptSimulation(
  prompt: string,
  domain: Domain,
): Promise<StrategyResponsePayload> {
  const rule = promptToRule(prompt, domain);

  // Extract sport filter from prompt for sports domain
  const sportFilter = domain === 'sports' ? extractSportFromPrompt(prompt) : undefined;

  const dataset = await loadDomainDataset(domain, sportFilter);
  const summary = simulateStrategyOnDataset(rule, dataset);
  const ragService = RAGService.getInstance();
  const ragInsight = await ragService.getTradingInsights(prompt, domain);
  const externalSnapshot =
    domain === 'crypto' ? await fetchCryptoSnapshot() : null;

  return {
    summary,
    ragInsight,
    externalSnapshot,
    datasetSize: dataset.length,
    sportFilter,  // Include the sport filter in response
  };
}

export async function getSnapshotOverview() {
  const [datasets, crypto] = await Promise.all([
    getDatasetCatalog(),
    fetchCryptoSnapshot(),
  ]);

  return {
    datasets,
    crypto,
  };
}

