/**
 * Backtest Worker
 * Processes backtest jobs from the BullMQ queue
 * 
 * Run with: npx ts-node src/workers/backtest.worker.ts
 * Or in production: node dist/workers/backtest.worker.js
 */

import { createBacktestWorker, BacktestJobData, BacktestJobResult } from '../lib/queue/BacktestQueue';
import { spawn } from 'child_process';
import * as path from 'path';
import { Job } from 'bullmq';

/**
 * Process a single backtest job
 */
async function processBacktest(job: Job<BacktestJobData>): Promise<BacktestJobResult> {
    const { market, strategyName, strategyCode, parameters, season } = job.data;

    console.log(`🔄 Processing: ${market} ${strategyName} for ${season || 'current season'}`);

    // Update progress
    await job.updateProgress(20);

    return new Promise((resolve, reject) => {
        const scriptPath = path.join(process.cwd(), 'universal_backtesting.py');

        const backtestParams = {
            market: market,
            strategy_name: strategyName,
            parameters: {
                ...parameters,
                season: season,
            },
            strategy_code: strategyCode,
            min_trades: 10,
        };

        const args = [
            scriptPath,
            'run_backtest',
            JSON.stringify(backtestParams),
        ];

        const pythonProcess = spawn('python3', args, {
            cwd: process.cwd(),
            env: {
                ...process.env,
                PYTHONPATH: path.join(process.cwd(), 'nba_api_env', 'lib', 'python3.13', 'site-packages'),
            },
        });

        let stdout = '';
        let stderr = '';

        pythonProcess.stdout.on('data', async (data) => {
            stdout += data.toString();
            // Update progress based on output
            if (stdout.includes('Loading')) {
                await job.updateProgress(40);
            }
            if (stdout.includes('Executing')) {
                await job.updateProgress(60);
            }
            if (stdout.includes('Calculating')) {
                await job.updateProgress(80);
            }
        });

        pythonProcess.stderr.on('data', (data) => {
            stderr += data.toString();
        });

        pythonProcess.on('close', async (code) => {
            await job.updateProgress(90);

            if (code !== 0) {
                console.error(`❌ Backtest failed: ${stderr}`);
                reject(new Error(`Backtest failed: ${stderr}`));
                return;
            }

            try {
                // Find JSON in stdout (may have debug output before it)
                const jsonMatch = stdout.match(/\{[\s\S]*\}$/);
                if (!jsonMatch) {
                    reject(new Error('No valid JSON in output'));
                    return;
                }

                const result = JSON.parse(jsonMatch[0]);

                resolve({
                    success: true,
                    results: result,
                });
            } catch (parseError) {
                console.error('❌ Failed to parse result:', stdout);
                reject(new Error('Invalid backtest result format'));
            }
        });

        pythonProcess.on('error', (error) => {
            console.error('❌ Failed to start Python:', error);
            reject(error);
        });

        // Timeout after 5 minutes
        setTimeout(() => {
            pythonProcess.kill();
            reject(new Error('Backtest timed out after 5 minutes'));
        }, 5 * 60 * 1000);
    });
}

// Start the worker
const worker = createBacktestWorker(processBacktest);

// Graceful shutdown
process.on('SIGTERM', async () => {
    console.log('🛑 Shutting down worker...');
    await worker.close();
    process.exit(0);
});

process.on('SIGINT', async () => {
    console.log('🛑 Shutting down worker...');
    await worker.close();
    process.exit(0);
});

console.log('🚀 Backtest worker is running. Press Ctrl+C to stop.');
