/**
 * SportsData.io API Service
 * Provides methods to fetch sports data for backtesting strategies
 * 
 * Includes persistent file-based caching to avoid re-pulling the same data
 */

import * as fs from 'fs';
import * as path from 'path';
import * as crypto from 'crypto';

const SPORTSDATA_BASE_URL = 'https://api.sportsdata.io/v3';

// Cache configuration - data is cached for different durations based on type
const CACHE_DURATIONS = {
    teams: 7 * 24 * 60 * 60 * 1000,      // 7 days for team data (rarely changes)
    players: 24 * 60 * 60 * 1000,         // 24 hours for player data
    schedule: 6 * 60 * 60 * 1000,         // 6 hours for schedule data
    scores: 5 * 60 * 1000,                // 5 minutes for live scores
    historical: 30 * 24 * 60 * 60 * 1000, // 30 days for historical data
    default: 60 * 60 * 1000,              // 1 hour default
};

interface SportsDataConfig {
    apiKey: string;
}

interface GameData {
    GameID: number;
    Season: number;
    Status: string;
    DateTime: string;
    HomeTeam: string;
    AwayTeam: string;
    HomeTeamScore: number | null;
    AwayTeamScore: number | null;
    PointSpread: number | null;
    OverUnder: number | null;
    [key: string]: any;
}

interface PlayerData {
    PlayerID: number;
    FirstName: string;
    LastName: string;
    Team: string;
    Position: string;
    [key: string]: any;
}

interface TeamData {
    TeamID: number;
    Key: string;
    City: string;
    Name: string;
    [key: string]: any;
}

interface CacheEntry {
    data: any;
    cachedAt: number;
    expiresAt: number;
    cacheKey: string;
}

export class SportsDataService {
    private apiKey: string;
    private static instance: SportsDataService;
    private cacheDir: string;
    private memoryCache: Map<string, CacheEntry> = new Map();

    private constructor(config: SportsDataConfig) {
        this.apiKey = config.apiKey;
        // Use data directory for cache storage
        this.cacheDir = path.join(process.cwd(), 'data', 'sportsdata_cache');
        this.ensureCacheDir();
    }

    private ensureCacheDir(): void {
        try {
            if (!fs.existsSync(this.cacheDir)) {
                fs.mkdirSync(this.cacheDir, { recursive: true });
            }
        } catch (error) {
            console.error('Failed to create cache directory:', error);
        }
    }

    public static getInstance(): SportsDataService {
        if (!SportsDataService.instance) {
            const apiKey = process.env.SPORTSDATA_API_KEY;
            if (!apiKey) {
                throw new Error('SPORTSDATA_API_KEY environment variable is not set');
            }
            SportsDataService.instance = new SportsDataService({ apiKey });
        }
        return SportsDataService.instance;
    }

    /**
     * Generate a cache key for an API request
     */
    private getCacheKey(sport: string, endpoint: string): string {
        const hash = crypto.createHash('md5')
            .update(`${sport}:${endpoint}`)
            .digest('hex');
        return hash;
    }

    /**
     * Get cache file path
     */
    private getCacheFilePath(cacheKey: string): string {
        return path.join(this.cacheDir, `${cacheKey}.json`);
    }

    /**
     * Determine cache duration based on endpoint type
     */
    private getCacheDuration(endpoint: string): number {
        const lowerEndpoint = endpoint.toLowerCase();
        
        if (lowerEndpoint.includes('teams')) {
            return CACHE_DURATIONS.teams;
        }
        if (lowerEndpoint.includes('players')) {
            return CACHE_DURATIONS.players;
        }
        if (lowerEndpoint.includes('schedule')) {
            return CACHE_DURATIONS.schedule;
        }
        if (lowerEndpoint.includes('scores') || lowerEndpoint.includes('live')) {
            return CACHE_DURATIONS.scores;
        }
        // Historical data (past seasons/dates)
        if (/\/\d{4}(\/|$)/.test(endpoint) && !endpoint.includes(new Date().getFullYear().toString())) {
            return CACHE_DURATIONS.historical;
        }
        
        return CACHE_DURATIONS.default;
    }

    /**
     * Check memory cache first, then file cache
     */
    private getFromCache<T>(cacheKey: string): T | null {
        const now = Date.now();

        // Check memory cache first
        const memoryEntry = this.memoryCache.get(cacheKey);
        if (memoryEntry && memoryEntry.expiresAt > now) {
            return memoryEntry.data as T;
        }

        // Check file cache
        try {
            const filePath = this.getCacheFilePath(cacheKey);
            if (fs.existsSync(filePath)) {
                const fileContent = fs.readFileSync(filePath, 'utf-8');
                const entry: CacheEntry = JSON.parse(fileContent);
                
                if (entry.expiresAt > now) {
                    // Restore to memory cache
                    this.memoryCache.set(cacheKey, entry);
                    return entry.data as T;
                } else {
                    // Cache expired, delete file
                    fs.unlinkSync(filePath);
                }
            }
        } catch (error) {
            // Cache read error, continue to API
            console.warn('Cache read error:', error);
        }

        return null;
    }

    /**
     * Save to both memory and file cache
     */
    private saveToCache(cacheKey: string, data: any, duration: number): void {
        const now = Date.now();
        const entry: CacheEntry = {
            data,
            cachedAt: now,
            expiresAt: now + duration,
            cacheKey,
        };

        // Save to memory cache
        this.memoryCache.set(cacheKey, entry);

        // Save to file cache (async, non-blocking)
        try {
            const filePath = this.getCacheFilePath(cacheKey);
            fs.writeFileSync(filePath, JSON.stringify(entry), 'utf-8');
        } catch (error) {
            console.warn('Cache write error:', error);
        }
    }

    /**
     * Fetch from API with caching
     */
    private async fetchFromAPI<T>(sport: string, endpoint: string): Promise<T> {
        const cacheKey = this.getCacheKey(sport, endpoint);
        
        // Try to get from cache first
        const cachedData = this.getFromCache<T>(cacheKey);
        if (cachedData !== null) {
            console.log(`[SportsDataCache] HIT: ${sport}/${endpoint}`);
            return cachedData;
        }

        console.log(`[SportsDataCache] MISS: ${sport}/${endpoint} - Fetching from API`);
        
        const url = `${SPORTSDATA_BASE_URL}/${sport}/scores/json/${endpoint}`;

        const response = await fetch(url, {
            method: 'GET',
            headers: {
                'Ocp-Apim-Subscription-Key': this.apiKey,
                'Content-Type': 'application/json',
            },
        });

        if (!response.ok) {
            const errorText = await response.text();
            throw new Error(`SportsData.io API error (${response.status}): ${errorText}`);
        }

        const data = await response.json();
        
        // Cache the result
        const cacheDuration = this.getCacheDuration(endpoint);
        this.saveToCache(cacheKey, data, cacheDuration);
        
        return data;
    }

    /**
     * Clear cache for a specific endpoint or all cache
     */
    public clearCache(sport?: string, endpoint?: string): void {
        if (sport && endpoint) {
            const cacheKey = this.getCacheKey(sport, endpoint);
            this.memoryCache.delete(cacheKey);
            const filePath = this.getCacheFilePath(cacheKey);
            if (fs.existsSync(filePath)) {
                fs.unlinkSync(filePath);
            }
        } else {
            // Clear all cache
            this.memoryCache.clear();
            try {
                const files = fs.readdirSync(this.cacheDir);
                for (const file of files) {
                    if (file.endsWith('.json')) {
                        fs.unlinkSync(path.join(this.cacheDir, file));
                    }
                }
            } catch (error) {
                console.error('Failed to clear cache directory:', error);
            }
        }
    }

    /**
     * Get cache statistics
     */
    public getCacheStats(): { memoryEntries: number; diskEntries: number; diskSizeMB: number } {
        let diskEntries = 0;
        let diskSize = 0;

        try {
            const files = fs.readdirSync(this.cacheDir);
            for (const file of files) {
                if (file.endsWith('.json')) {
                    diskEntries++;
                    const stats = fs.statSync(path.join(this.cacheDir, file));
                    diskSize += stats.size;
                }
            }
        } catch {
            // Ignore errors
        }

        return {
            memoryEntries: this.memoryCache.size,
            diskEntries,
            diskSizeMB: Math.round(diskSize / 1024 / 1024 * 100) / 100,
        };
    }

    // ============ NFL ============
    async getNFLSchedule(season: number): Promise<GameData[]> {
        return this.fetchFromAPI<GameData[]>('nfl', `Schedules/${season}`);
    }

    async getNFLScoresByWeek(season: number, week: number): Promise<GameData[]> {
        return this.fetchFromAPI<GameData[]>('nfl', `ScoresByWeek/${season}/${week}`);
    }

    async getNFLTeams(): Promise<TeamData[]> {
        return this.fetchFromAPI<TeamData[]>('nfl', 'Teams');
    }

    // ============ NBA ============
    async getNBAGames(season: number): Promise<GameData[]> {
        return this.fetchFromAPI<GameData[]>('nba', `Games/${season}`);
    }

    async getNBAGamesByDate(date: string): Promise<GameData[]> {
        // date format: YYYY-MMM-DD (e.g., 2024-DEC-19)
        return this.fetchFromAPI<GameData[]>('nba', `GamesByDate/${date}`);
    }

    async getNBATeams(): Promise<TeamData[]> {
        return this.fetchFromAPI<TeamData[]>('nba', 'Teams');
    }

    async getNBAPlayers(): Promise<PlayerData[]> {
        return this.fetchFromAPI<PlayerData[]>('nba', 'Players');
    }

    // ============ MLB ============
    async getMLBGames(season: number): Promise<GameData[]> {
        return this.fetchFromAPI<GameData[]>('mlb', `Games/${season}`);
    }

    async getMLBGamesByDate(date: string): Promise<GameData[]> {
        return this.fetchFromAPI<GameData[]>('mlb', `GamesByDate/${date}`);
    }

    async getMLBTeams(): Promise<TeamData[]> {
        return this.fetchFromAPI<TeamData[]>('mlb', 'Teams');
    }

    // ============ NHL ============
    async getNHLGames(season: number): Promise<GameData[]> {
        return this.fetchFromAPI<GameData[]>('nhl', `Games/${season}`);
    }

    async getNHLGamesByDate(date: string): Promise<GameData[]> {
        return this.fetchFromAPI<GameData[]>('nhl', `GamesByDate/${date}`);
    }

    async getNHLTeams(): Promise<TeamData[]> {
        return this.fetchFromAPI<TeamData[]>('nhl', 'Teams');
    }

    // ============ College Football ============
    async getCFBGames(season: number): Promise<GameData[]> {
        return this.fetchFromAPI<GameData[]>('cfb', `Games/${season}`);
    }

    async getCFBTeams(): Promise<TeamData[]> {
        return this.fetchFromAPI<TeamData[]>('cfb', 'Teams');
    }

    // ============ College Basketball ============
    async getCBBGames(season: number): Promise<GameData[]> {
        return this.fetchFromAPI<GameData[]>('cbb', `Games/${season}`);
    }

    async getCBBTeams(): Promise<TeamData[]> {
        return this.fetchFromAPI<TeamData[]>('cbb', 'Teams');
    }

    // ============ Soccer ============
    async getSoccerCompetitions(): Promise<any[]> {
        return this.fetchFromAPI<any[]>('soccer', 'Competitions');
    }

    async getSoccerGamesByDate(competition: string, date: string): Promise<GameData[]> {
        return this.fetchFromAPI<GameData[]>('soccer', `GamesByDate/${competition}/${date}`);
    }

    // ============ UFC / MMA ============
    async getMMAEvents(): Promise<any[]> {
        return this.fetchFromAPI<any[]>('mma', 'Schedule');
    }

    async getMMAFighters(): Promise<any[]> {
        return this.fetchFromAPI<any[]>('mma', 'Fighters');
    }

    // ============ Golf ============
    async getGolfTournaments(season: number): Promise<any[]> {
        return this.fetchFromAPI<any[]>('golf', `Tournaments/${season}`);
    }

    async getGolfLeaderboard(tournamentId: number): Promise<any> {
        return this.fetchFromAPI<any>('golf', `Leaderboard/${tournamentId}`);
    }

    // ============ NASCAR ============
    async getNASCARRaces(season: number): Promise<any[]> {
        return this.fetchFromAPI<any[]>('nascar', `races/${season}`);
    }

    async getNASCARRaceResults(raceId: number): Promise<any> {
        return this.fetchFromAPI<any>('nascar', `raceresult/${raceId}`);
    }

    // ============ Tennis ============
    async getTennisPlayers(): Promise<any[]> {
        return this.fetchFromAPI<any[]>('tennis', 'Players');
    }

    async getTennisSchedule(season: number): Promise<any[]> {
        return this.fetchFromAPI<any[]>('tennis', `Schedule/${season}`);
    }

    // ============ Unified Query Method ============
    async queryData(sport: string, endpoint: string, params: Record<string, any> = {}): Promise<any> {
        const sportLower = sport.toLowerCase();

        // Map common sport names to API paths
        const sportMap: Record<string, string> = {
            'nfl': 'nfl',
            'nba': 'nba',
            'mlb': 'mlb',
            'nhl': 'nhl',
            'ncaaf': 'cfb',
            'cfb': 'cfb',
            'college football': 'cfb',
            'ncaab': 'cbb',
            'cbb': 'cbb',
            'college basketball': 'cbb',
            'soccer': 'soccer',
            'mma': 'mma',
            'ufc': 'mma',
            'golf': 'golf',
            'pga': 'golf',
            'nascar': 'nascar',
            'tennis': 'tennis',
        };

        const apiSport = sportMap[sportLower] || sportLower;

        // Build endpoint with params
        let fullEndpoint = endpoint;
        if (Object.keys(params).length > 0) {
            const paramValues = Object.values(params);
            fullEndpoint = `${endpoint}/${paramValues.join('/')}`;
        }

        return this.fetchFromAPI<any>(apiSport, fullEndpoint);
    }

    // ============ Format Data for Chat ============
    formatGamesForChat(games: GameData[], sport: string): string {
        if (!games || games.length === 0) {
            return `No ${sport.toUpperCase()} games found.`;
        }

        const formatted = games.slice(0, 10).map(game => {
            const status = game.Status || 'Scheduled';
            const score = game.HomeTeamScore !== null && game.AwayTeamScore !== null
                ? `${game.HomeTeamScore}-${game.AwayTeamScore}`
                : 'TBD';
            const spread = game.PointSpread ? `Spread: ${game.PointSpread}` : '';
            const ou = game.OverUnder ? `O/U: ${game.OverUnder}` : '';

            return `• ${game.AwayTeam} @ ${game.HomeTeam} | ${status} | Score: ${score} ${spread} ${ou}`.trim();
        }).join('\n');

        return `**${sport.toUpperCase()} Games:**\n${formatted}`;
    }

    formatTeamsForChat(teams: TeamData[], sport: string): string {
        if (!teams || teams.length === 0) {
            return `No ${sport.toUpperCase()} teams found.`;
        }

        const formatted = teams.map(team =>
            `• ${team.City} ${team.Name} (${team.Key})`
        ).join('\n');

        return `**${sport.toUpperCase()} Teams:**\n${formatted}`;
    }
}

export default SportsDataService;
