import { getOptimizedConfig, VectorPerformanceMonitor } from '@/lib/rag/OptimizedVectorConfig';

interface RAGSource {
  source: string;
  preview: string;
  score: number;
  enriched?: boolean;
  summary?: string | null;
  themes?: string[];
}

interface RAGResponse {
  answer: string;
  sources: RAGSource[];
  chunks_searched?: number;
  chunks_returned?: number;
  enriched_count?: number;
  themes?: string[];
  key_points?: string[];
}

export interface RAGInsight {
  answer: string;
  sources: RAGSource[];
  relevanceScore: number;
  // Enriched data fields
  enrichedCount: number;
  themes: string[];
  keyPoints: string[];
  hasEnrichedData: boolean;
}

export class RAGService {
  private static instance: RAGService;
  private baseUrl: string;
  private performanceMonitor: VectorPerformanceMonitor;

  private constructor() {
    // Point to the advanced RAG Flask server (port 5001)
    this.baseUrl = process.env.RAG_API_URL || 'http://localhost:5001';
    this.performanceMonitor = VectorPerformanceMonitor.getInstance();
  }

  static getInstance(): RAGService {
    if (!RAGService.instance) {
      RAGService.instance = new RAGService();
    }
    return RAGService.instance;
  }

  async getTradingInsights(
    question: string,
    domain: 'sports' | 'crypto' | 'stocks' | 'forex',
    useEnriched: boolean = true
  ): Promise<RAGInsight | null> {
    const startTime = Date.now();

    try {
      // Use optimized configuration for document search
      const config = getOptimizedConfig('document_search');

      // Enhance the question with domain-specific context
      const enhancedQuestion = this.enhanceQuestionForDomain(question, domain);

      // Determine if we should use keyword-focused search
      const shouldUseKeywordSearch = this._shouldUseKeywordSearch(question);

      // Add performance hints for the RAG system with enhanced keyword parameters
      const searchParams = {
        question: enhancedQuestion,
        k: 5, // Limit results for speed
        use_hybrid: config.enableKeywordBoost,
        use_metadata_filter: config.enableMetadataFiltering,
        domain_filter: domain, // Add domain filtering for faster search
        use_keyword_focused: shouldUseKeywordSearch, // New parameter for enhanced search
        extract_keywords: true, // Enable keyword extraction
        expand_query: true, // Enable query expansion with synonyms
        use_enriched: useEnriched, // Use enriched chunks if available
      };

      const apiKey = process.env.FLASK_API_KEY;
      if (!apiKey) {
        throw new Error('FLASK_API_KEY environment variable is not set');
      }

      const response = await fetch(`${this.baseUrl}/ask`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-API-Key': apiKey,
        },
        body: JSON.stringify(searchParams),
      });

      if (!response.ok) {
        console.warn(`RAG API call failed: ${response.status}`);
        return null;
      }

      const data: RAGResponse = await response.json();

      // Calculate relevance score based on answer quality
      const relevanceScore = this.calculateRelevanceScore(data);

      // Record performance metrics
      const queryTime = Date.now() - startTime;
      this.performanceMonitor.recordMetrics('document_search', {
        avgQueryTime: queryTime,
        throughput: 1000 / queryTime, // queries per second
      });

      const sources = data.sources || [];
      const enrichedCount = Number(data.enriched_count || 0);
      const themes = Array.isArray(data.themes) ? data.themes : [];
      const keyPoints = Array.isArray(data.key_points) ? data.key_points : [];
      const hasEnrichedData =
        enrichedCount > 0 || sources.some((s) => Boolean((s as any).enriched));

      return {
        answer: data.answer,
        sources,
        relevanceScore,
        enrichedCount,
        themes,
        keyPoints,
        hasEnrichedData,
      };
    } catch (error) {
      console.error('RAG service error:', error);

      // Record error metrics
      this.performanceMonitor.recordMetrics('document_search', {
        avgQueryTime: Date.now() - startTime,
      });

      return null;
    }
  }

  private _shouldUseKeywordSearch(question: string): boolean {
    // Use keyword search for questions with specific trading/finance terminology
    const keywordIndicators = [
      'kelly criterion', 'bollinger bands', 'rsi', 'macd', 'fibonacci',
      'parlay', 'round robin', 'arbitrage', 'hedge', 'leverage',
      'technical analysis', 'fundamental analysis', 'quantitative',
      'momentum', 'mean reversion', 'breakout', 'reversal'
    ];

    const questionLower = question.toLowerCase();
    const keywordMatches = keywordIndicators.filter(term => questionLower.includes(term));

    // Use keyword search if multiple specific terms are found or question is very detailed
    return keywordMatches.length >= 2 || question.split(' ').length > 15;
  }

  private enhanceQuestionForDomain(question: string, domain: 'sports' | 'crypto' | 'stocks' | 'forex'): string {
    const domainPrefixes = {
      sports: 'In the context of sports betting and trading strategies:',
      crypto: 'In the context of cryptocurrency trading and market analysis:',
      stocks: 'In the context of stock market trading and investment strategies:',
      forex: 'In the context of foreign exchange trading and currency markets:',
    };

    return `${domainPrefixes[domain]} ${question}`;
  }

  private calculateRelevanceScore(data: RAGResponse): number {
    // Enhanced relevance scoring with enrichment awareness
    let score = 0.5; // Base score

    // Increase score if we have sources
    if (data.sources && data.sources.length > 0) {
      score += 0.15;
    }

    // Bonus for enriched sources
    const enrichedSources = data.sources?.filter(s => s.enriched) || [];
    if (enrichedSources.length > 0) {
      score += 0.1 * Math.min(enrichedSources.length / 5, 1); // Up to 0.1 bonus
    }

    // Bonus for having themes identified
    if (data.themes && data.themes.length > 0) {
      score += 0.1;
    }

    // Bonus for having key points
    if (data.key_points && data.key_points.length > 0) {
      score += 0.1;
    }

    // Increase score based on answer length (reasonable answers are usually longer)
    const answerLength = data.answer.length;
    if (answerLength > 50 && answerLength < 2000) {
      score += 0.15;
    }

    // Increase score if answer doesn't contain generic responses
    const genericPhrases = ['no relevant documents', 'no documents available', 'please provide', "don't have enough information"];
    const hasGenericResponse = genericPhrases.some(phrase =>
      data.answer.toLowerCase().includes(phrase)
    );

    if (!hasGenericResponse) {
      score += 0.1;
    }

    return Math.min(score, 1.0); // Cap at 1.0
  }

  async isHealthy(): Promise<boolean> {
    try {
      const response = await fetch(`${this.baseUrl}/status`);
      return response.ok;
    } catch {
      return false;
    }
  }
}

