import fs from 'node:fs/promises';
import path from 'node:path';

import { loadSportsGames } from '@/lib/csvLoader';
import { Domain, DomainRecord } from '@/lib/types';

export interface DatasetMeta {
  id: string;
  domain: Domain;
  rowCount: number;
  lastModified: string;
  sizeBytes: number;
}

export class DataService {
  private static instance: DataService;

  private constructor() {}

  static getInstance(): DataService {
    if (!DataService.instance) {
      DataService.instance = new DataService();
    }
    return DataService.instance;
  }

  async getRecordCount(domain: Domain): Promise<number> {
    switch (domain) {
      case 'sports': return (await loadSportsGames()).length;
      default: return 0;
    }
  }

  async getDataset(domain: Domain): Promise<DomainRecord[]> {
    switch (domain) {
      case 'sports': return loadSportsGames();
      default: return [];
    }
  }

  async getAllDatasetsMeta(): Promise<DatasetMeta[]> {
    const csvRoot = path.join(process.cwd(), 'data', 'csv');
    const sportsFile = path.join(csvRoot, 'sports_games.csv');

    const [sports, stat] = await Promise.all([
      loadSportsGames(),
      fs.stat(sportsFile).catch(() => null),
    ]);

    return [
      {
        id: 'sports-master',
        domain: 'sports',
        rowCount: sports.length,
        lastModified: stat?.mtime ? stat.mtime.toISOString() : new Date().toISOString(),
        sizeBytes: stat?.size ?? 0,
      },
    ];
  }
}

