/**
 * BALLDONTLIE API Service
 *
 * Replaces SportsData.io integration.
 *
 * Key goals:
 * - Use BALLDONTLIE_API_KEY from env (never hardcode secrets)
 * - Persistent disk caching + in-memory caching to avoid duplicate API pulls
 * - In-flight request de-duplication to avoid concurrent double-requests
 *
 * Docs: https://www.balldontlie.io/docs/ and https://docs.balldontlie.io/
 */
import * as fs from 'fs';
import * as path from 'path';
import * as crypto from 'crypto';

const BDL_BASE_URL = 'https://api.balldontlie.io';

// Cache configuration - tuned to reduce rate-limit usage while staying fresh
const CACHE_DURATIONS_MS = {
  teams: 7 * 24 * 60 * 60 * 1000,      // 7 days
  players: 24 * 60 * 60 * 1000,        // 24 hours
  rosters: 6 * 60 * 60 * 1000,         // 6 hours
  games: 15 * 60 * 1000,               // 15 minutes (schedules / recent games)
  odds: 5 * 60 * 1000,                 // 5 minutes
  historical: 30 * 24 * 60 * 60 * 1000,// 30 days
  default: 60 * 60 * 1000,             // 1 hour
};

export type BallDontLieLeague =
  | 'nba'
  | 'nfl'
  | 'mlb'
  | 'nhl'
  | 'wnba'
  | 'epl'
  | 'ncaaf'
  | 'ncaab'
  | string;

interface BallDontLieConfig {
  apiKey: string;
}

interface CacheEntry {
  data: any;
  cachedAt: number;
  expiresAt: number;
  cacheKey: string;
  // Useful for debugging / audits
  url: string;
}

export class BallDontLieService {
  private static instance: BallDontLieService | null = null;

  private readonly apiKey: string;
  private readonly cacheDir: string;
  private readonly memoryCache: Map<string, CacheEntry> = new Map();
  private readonly inFlight: Map<string, Promise<any>> = new Map();

  private readonly localDataDir: string;

  private constructor(config: BallDontLieConfig) {
    this.apiKey = config.apiKey;
    const overrideDir = process.env.BALLDONTLIE_CACHE_DIR;
    this.cacheDir = overrideDir
      ? (path.isAbsolute(overrideDir) ? overrideDir : path.join(process.cwd(), overrideDir))
      : path.join(process.cwd(), 'data', 'balldontlie_cache');
    // Local pre-downloaded data directory
    this.localDataDir = path.join(process.cwd(), 'data');
    this.ensureCacheDir();
  }

  public static getInstance(): BallDontLieService {
    if (!BallDontLieService.instance) {
      const apiKey = process.env.BALLDONTLIE_API_KEY;
      if (!apiKey) {
        throw new Error('BALLDONTLIE_API_KEY environment variable is not set');
      }
      BallDontLieService.instance = new BallDontLieService({ apiKey });
    }
    return BallDontLieService.instance;
  }

  private ensureCacheDir(): void {
    try {
      if (!fs.existsSync(this.cacheDir)) {
        fs.mkdirSync(this.cacheDir, { recursive: true });
      }
    } catch (error) {
      console.error('[BallDontLie] Failed to create cache directory:', error);
    }
  }

  private leaguePrefix(league: BallDontLieLeague): string {
    const l = (league || '').toLowerCase();

    // NBA is the "default" API on some endpoints; keep compatibility with /v1/*
    if (l === 'nba' || l === 'basketball') {
      return '/v1';
    }

    // Most other leagues use /{league}/v1/*
    // Examples: /nfl/v1, /mlb/v1, /nhl/v1, /wnba/v1, /epl/v1, /ncaaf/v1, /ncaab/v1
    return `/${l}/v1`;
  }

  private normalizePath(value: string): string {
    return value.startsWith('/') ? value : `/${value}`;
  }

  private buildUrl(league: BallDontLieLeague, resourcePath: string, query?: Record<string, any>): string {
    const prefix = this.leaguePrefix(league);
    const resource = this.normalizePath(resourcePath);

    const qs = new URLSearchParams();
    if (query) {
      const keys = Object.keys(query).sort();
      for (const key of keys) {
        const val = query[key];
        if (val === undefined || val === null) continue;
        if (Array.isArray(val)) {
          for (const v of val) {
            if (v === undefined || v === null) continue;
            qs.append(key, String(v));
          }
        } else {
          qs.append(key, String(val));
        }
      }
    }

    const url = `${BDL_BASE_URL}${prefix}${resource}${qs.toString() ? `?${qs.toString()}` : ''}`;
    return url.replace(/([^:]\/)\/+/g, '$1'); // collapse double slashes (except protocol)
  }

  private getCacheKey(url: string): string {
    return crypto.createHash('md5').update(url).digest('hex');
  }

  private getCacheFilePath(cacheKey: string): string {
    return path.join(this.cacheDir, `${cacheKey}.json`);
  }

  private getCacheDurationMs(resourcePath: string, query?: Record<string, any>): number {
    const p = resourcePath.toLowerCase();

    if (p.includes('team')) return CACHE_DURATIONS_MS.teams;
    if (p.includes('roster')) return CACHE_DURATIONS_MS.rosters;
    if (p.includes('player')) return CACHE_DURATIONS_MS.players;
    if (p.includes('odd')) return CACHE_DURATIONS_MS.odds;
    if (p.includes('game')) {
      // If requesting an explicit past season, cache longer
      const season = query?.season;
      const year = typeof season === 'number' ? season : Number(season);
      if (Number.isFinite(year) && year < new Date().getFullYear()) {
        return CACHE_DURATIONS_MS.historical;
      }
      return CACHE_DURATIONS_MS.games;
    }

    return CACHE_DURATIONS_MS.default;
  }

  private getFromCache<T>(cacheKey: string): T | null {
    const now = Date.now();

    const memory = this.memoryCache.get(cacheKey);
    if (memory && memory.expiresAt > now) {
      return memory.data as T;
    }

    try {
      const filePath = this.getCacheFilePath(cacheKey);
      if (!fs.existsSync(filePath)) return null;
      const raw = fs.readFileSync(filePath, 'utf-8');
      const entry = JSON.parse(raw) as CacheEntry;
      if (entry.expiresAt > now) {
        this.memoryCache.set(cacheKey, entry);
        return entry.data as T;
      }
      // Expired
      try {
        fs.unlinkSync(filePath);
      } catch {
        // ignore
      }
    } catch {
      // ignore cache failures
    }

    return null;
  }

  private saveToCache(cacheKey: string, url: string, data: any, ttlMs: number): void {
    const now = Date.now();
    const entry: CacheEntry = {
      data,
      cachedAt: now,
      expiresAt: now + ttlMs,
      cacheKey,
      url,
    };

    this.memoryCache.set(cacheKey, entry);

    try {
      const filePath = this.getCacheFilePath(cacheKey);
      fs.writeFileSync(filePath, JSON.stringify(entry), 'utf-8');
    } catch {
      // ignore disk write failures
    }
  }

  private async fetchJson<T>(url: string, ttlMs: number): Promise<T> {
    const cacheKey = this.getCacheKey(url);

    const cached = this.getFromCache<T>(cacheKey);
    if (cached !== null) {
      console.log(`[BallDontLieCache] HIT: ${url}`);
      return cached;
    }

    const existing = this.inFlight.get(cacheKey);
    if (existing) {
      return existing as Promise<T>;
    }

    console.log(`[BallDontLieCache] MISS: ${url} - Fetching from API`);
    const reqPromise = (async () => {
      try {
        const resp = await fetch(url, {
          method: 'GET',
          headers: {
            Authorization: this.apiKey,
            Accept: 'application/json',
          },
        });

        if (!resp.ok) {
          const text = await resp.text().catch(() => '');
          throw new Error(`BallDontLie API error (${resp.status}): ${text}`);
        }

        const data = (await resp.json()) as T;
        this.saveToCache(cacheKey, url, data, ttlMs);
        return data;
      } finally {
        this.inFlight.delete(cacheKey);
      }
    })();

    this.inFlight.set(cacheKey, reqPromise);
    return reqPromise;
  }

  // -------------------------
  // Local Data Access (Pre-downloaded JSON files)
  // -------------------------

  private getLocalDataPath(league: BallDontLieLeague, dataType: string): string {
    const leagueLower = league.toLowerCase();
    
    // Map league and data type to file paths
    const pathMap: Record<string, Record<string, string>> = {
      nba: {
        teams: 'nba/teams.json',
        players: 'nba/players.json',
        games: 'nba/games.json',
        odds: 'nba/odds.json',
        standings: 'nba/standings.json',
        stats: 'nba/stats.json',
      },
      nfl: {
        teams: 'nfl/teams.json',
        players: 'nfl/players.json',
        games: 'nfl/games.json',
        odds: 'odds/nfl_odds.json',
        standings: 'nfl/standings.json',
      },
      nhl: {
        teams: 'nhl/teams.json',
        players: 'nhl/players.json',
        games: 'nhl/games.json',
        odds: 'odds/nhl_odds.json',
      },
      mlb: {
        teams: 'mlb/teams.json',
        players: 'mlb/players.json',
        games: 'mlb/games.json',
        odds: 'mlb/odds.json',
        standings: 'mlb/standings.json',
      },
      wnba: {
        teams: 'wnba/teams.json',
        players: 'wnba/players.json',
        games: 'wnba/games.json',
      },
      ncaaf: {
        teams: 'ncaaf/teams.json',
        games: 'ncaaf/games.json',
      },
      ncaab: {
        teams: 'ncaab/teams.json',
        games: 'ncaab/games.json',
      },
      epl: {
        teams: 'epl/teams.json',
        games: 'epl/games.json',
      },
    };

    return pathMap[leagueLower]?.[dataType] || '';
  }

  private loadLocalData<T>(league: BallDontLieLeague, dataType: string): T | null {
    const relativePath = this.getLocalDataPath(league, dataType);
    if (!relativePath) return null;

    const fullPath = path.join(this.localDataDir, relativePath);
    
    try {
      if (fs.existsSync(fullPath)) {
        const content = fs.readFileSync(fullPath, 'utf-8');
        const data = JSON.parse(content);
        console.log(`[BallDontLie] LOCAL DATA: Loaded ${league}/${dataType} from ${relativePath} (${Array.isArray(data) ? data.length : 'object'} items)`);
        return data as T;
      }
    } catch (error) {
      console.warn(`[BallDontLie] Failed to load local data ${relativePath}:`, error);
    }
    
    return null;
  }

  // -------------------------
  // Public, typed-ish helpers
  // -------------------------

  async getTeams(league: BallDontLieLeague, query: Record<string, any> = {}): Promise<any> {
    // Try local data first
    const localData = this.loadLocalData(league, 'teams');
    if (localData) {
      return { data: Array.isArray(localData) ? localData : [localData] };
    }
    
    // Fall back to API
    const url = this.buildUrl(league, '/team', query);
    const ttl = this.getCacheDurationMs('/team', query);
    return this.fetchJson<any>(url, ttl);
  }

  async getPlayers(league: BallDontLieLeague, query: Record<string, any> = {}): Promise<any> {
    // Try local data first
    const localData = this.loadLocalData(league, 'players');
    if (localData) {
      return { data: Array.isArray(localData) ? localData : [localData] };
    }
    
    // Fall back to API
    const url = this.buildUrl(league, '/player', query);
    const ttl = this.getCacheDurationMs('/player', query);
    return this.fetchJson<any>(url, ttl);
  }

  async getGames(league: BallDontLieLeague, query: Record<string, any> = {}): Promise<any> {
    // Try local data first
    const localData = this.loadLocalData<any[]>(league, 'games');
    if (localData && Array.isArray(localData)) {
      let filtered = localData;
      
      // Apply season filter if provided
      if (query.season) {
        filtered = filtered.filter(g => g.season === Number(query.season));
      }
      
      // Apply date filters if provided
      if (query.start_date) {
        const startDate = new Date(query.start_date);
        filtered = filtered.filter(g => new Date(g.date || g.datetime) >= startDate);
      }
      if (query.end_date) {
        const endDate = new Date(query.end_date);
        filtered = filtered.filter(g => new Date(g.date || g.datetime) <= endDate);
      }
      
      console.log(`[BallDontLie] LOCAL: Returning ${filtered.length} games for ${league} (filtered from ${localData.length})`);
      return { data: filtered };
    }
    
    // Fall back to API
    const url = this.buildUrl(league, '/game', query);
    const ttl = this.getCacheDurationMs('/game', query);
    return this.fetchJson<any>(url, ttl);
  }

  async getOdds(league: BallDontLieLeague, query: Record<string, any> = {}): Promise<any> {
    // Try local data first
    const localData = this.loadLocalData<any[]>(league, 'odds');
    if (localData && Array.isArray(localData)) {
      let filtered = localData;
      
      // Apply game_id filter if provided
      if (query.game_id) {
        filtered = filtered.filter(o => o.game_id === Number(query.game_id));
      }
      
      // Apply season filter if provided (for NFL)
      if (query.season) {
        filtered = filtered.filter(o => o.season === Number(query.season));
      }
      
      console.log(`[BallDontLie] LOCAL: Returning ${filtered.length} odds for ${league}`);
      return { data: filtered };
    }
    
    // Fall back to API
    const url = this.buildUrl(league, '/odds', query);
    const ttl = this.getCacheDurationMs('/odds', query);
    return this.fetchJson<any>(url, ttl);
  }

  async getStandings(league: BallDontLieLeague, query: Record<string, any> = {}): Promise<any> {
    // Try local data first
    const localData = this.loadLocalData(league, 'standings');
    if (localData) {
      return { data: Array.isArray(localData) ? localData : [localData] };
    }
    
    // Fall back to API
    const url = this.buildUrl(league, '/standings', query);
    const ttl = this.getCacheDurationMs('/standings', query);
    return this.fetchJson<any>(url, ttl);
  }

  async getPlayerProps(league: BallDontLieLeague, gameId?: number): Promise<any> {
    // Check local player props data
    const propsPath = path.join(this.localDataDir, 'player_props', `${league}_player_props.json`);
    try {
      if (fs.existsSync(propsPath)) {
        const content = fs.readFileSync(propsPath, 'utf-8');
        let data = JSON.parse(content);
        
        if (gameId && Array.isArray(data)) {
          data = data.filter((p: any) => p.game_id === gameId);
        }
        
        console.log(`[BallDontLie] LOCAL: Returning ${Array.isArray(data) ? data.length : 'object'} player props for ${league}`);
        return { data };
      }
    } catch (error) {
      console.warn(`[BallDontLie] Failed to load player props:`, error);
    }
    
    // Note: Player props are live-only from API, so we return empty if no local data
    return { data: [], note: 'Player props are live-only and not currently available' };
  }

  // -------------------------
  // Formatting helpers for chat
  // -------------------------

  private unwrapArray(payload: any): any[] {
    if (Array.isArray(payload)) return payload;
    if (payload && Array.isArray(payload.data)) return payload.data;
    if (payload && Array.isArray(payload.results)) return payload.results;
    return [];
  }

  formatTeamsForChat(payload: any, leagueLabel: string): string {
    const teams = this.unwrapArray(payload);
    if (!teams.length) return `No ${leagueLabel.toUpperCase()} teams found.`;

    const formatted = teams.slice(0, 12).map((t: any) => {
      const name =
        t?.full_name ||
        t?.name ||
        [t?.city, t?.name].filter(Boolean).join(' ') ||
        t?.abbreviation ||
        t?.key ||
        t?.code ||
        t?.slug ||
        String(t?.id ?? '');
      const abbr = t?.abbreviation || t?.key || t?.code;
      return abbr && name && !String(name).includes(String(abbr))
        ? `• ${name} (${abbr})`
        : `• ${name}`;
    }).join('\n');

    return `**${leagueLabel.toUpperCase()} Teams (BallDontLie):**\n${formatted}`;
  }

  formatGamesForChat(payload: any, leagueLabel: string): string {
    const games = this.unwrapArray(payload);
    if (!games.length) return `No ${leagueLabel.toUpperCase()} games found.`;

    const formatted = games.slice(0, 10).map((g: any) => {
      const home = g?.home_team?.name || g?.home_team || g?.home || g?.homeTeam || g?.home_team_name;
      const away = g?.visitor_team?.name || g?.away_team || g?.away || g?.awayTeam || g?.away_team_name;
      const date = g?.date || g?.start_time || g?.datetime || g?.scheduled;
      const status = g?.status || g?.state || g?.game_status || '';
      return `• ${away || 'Away'} @ ${home || 'Home'}${date ? ` | ${String(date).slice(0, 10)}` : ''}${status ? ` | ${status}` : ''}`.trim();
    }).join('\n');

    return `**${leagueLabel.toUpperCase()} Games (BallDontLie):**\n${formatted}`;
  }

  formatStandingsForChat(payload: any, leagueLabel: string): string {
    const rows = this.unwrapArray(payload);
    if (!rows.length) return `No ${leagueLabel.toUpperCase()} standings found.`;

    const parsed = rows
      .map((r: any) => {
        const teamObj = r?.team || r?.team_data || null;
        const teamName =
          teamObj?.full_name ||
          teamObj?.name ||
          r?.team_name ||
          r?.name ||
          r?.team ||
          r?.abbreviation ||
          r?.key ||
          r?.code ||
          String(r?.team_id ?? r?.id ?? '');

        const abbr =
          teamObj?.abbreviation ||
          r?.abbreviation ||
          r?.key ||
          r?.code ||
          null;

        const winsRaw = r?.wins ?? r?.win ?? r?.w ?? teamObj?.wins;
        const lossesRaw = r?.losses ?? r?.loss ?? r?.l ?? teamObj?.losses;
        const wins = Number.isFinite(Number(winsRaw)) ? Number(winsRaw) : 0;
        const losses = Number.isFinite(Number(lossesRaw)) ? Number(lossesRaw) : 0;

        const winPctRaw =
          r?.win_pct ??
          r?.win_percentage ??
          r?.winPercentage ??
          r?.pct ??
          r?.percentage;
        const computedPct = wins + losses > 0 ? wins / (wins + losses) : null;
        const winPct =
          Number.isFinite(Number(winPctRaw)) ? Number(winPctRaw) : computedPct;

        return {
          teamName: String(teamName || '').trim(),
          abbr: abbr ? String(abbr).trim() : null,
          wins,
          losses,
          winPct: winPct ?? 0,
        };
      })
      .filter((r: any) => r.teamName && (r.wins > 0 || r.losses > 0 || r.winPct > 0));

    if (!parsed.length) return `No ${leagueLabel.toUpperCase()} standings found.`;

    parsed.sort((a: any, b: any) => {
      if (b.winPct !== a.winPct) return b.winPct - a.winPct;
      if (b.wins !== a.wins) return b.wins - a.wins;
      return a.teamName.localeCompare(b.teamName);
    });

    const formatted = parsed.slice(0, 20).map((r: any, idx: number) => {
      const label = r.abbr && !r.teamName.includes(r.abbr)
        ? `${r.teamName} (${r.abbr})`
        : r.teamName;
      const wl = `${r.wins}-${r.losses}`;
      const pct = typeof r.winPct === 'number' ? r.winPct.toFixed(3) : '—';
      return `${idx + 1}. ${label} — ${wl} (${pct})`;
    }).join('\n');

    return `**${leagueLabel.toUpperCase()} Standings (BallDontLie):**\n${formatted}`;
  }

  public getCacheStats(): { memoryEntries: number; diskEntries: number; diskSizeMB: number } {
    let diskEntries = 0;
    let diskSize = 0;
    try {
      const files = fs.readdirSync(this.cacheDir);
      for (const file of files) {
        if (!file.endsWith('.json')) continue;
        diskEntries += 1;
        const stats = fs.statSync(path.join(this.cacheDir, file));
        diskSize += stats.size;
      }
    } catch {
      // ignore
    }

    return {
      memoryEntries: this.memoryCache.size,
      diskEntries,
      diskSizeMB: Math.round((diskSize / 1024 / 1024) * 100) / 100,
    };
  }
}

export default BallDontLieService;

