/**
 * Next.js Middleware for Global Security and Authentication
 * Applied to all routes matching the defined patterns
 */

import { NextRequest, NextResponse } from 'next/server';
import { addSecurityHeaders } from '@/lib/securityHeaders';

// Define which routes to apply middleware to
export const config = {
  matcher: [
    /*
     * Match all request paths except for the ones starting with:
     * - _next/static (static files)
     * - _next/image (image optimization files)
     * - favicon.ico (favicon file)
     * - public files with extensions
     */
    '/((?!_next/static|_next/image|favicon.ico|.*\\.).*)',
  ],
};

export async function middleware(request: NextRequest) {
  const { pathname } = request.nextUrl;

  // Skip middleware for static assets and Next.js internals
  if (
    pathname.startsWith('/_next/') ||
    pathname.startsWith('/api/auth/') || // Auth routes handle their own security
    pathname.includes('.') // Static files
  ) {
    return NextResponse.next();
  }

  // Apply security headers to all responses
  const response = NextResponse.next();
  return addSecurityHeaders(response);
}
