export type Domain = 'sports' | 'crypto' | 'stocks' | 'forex';

export interface StrategyRule {
  domain: Domain;
  lookback: number;
  trigger:
  | 'bounceAfterLosses'
  | 'momentumDip'
  | 'volumeExpansion'
  | 'earningsDrift'
  | 'rangeBreakout'
  | 'carryTrade'
  | 'newsReversal'
  | 'parlay'
  | 'roundRobin';
  threshold?: number;
  direction: 'long' | 'short';
  description: string;
  parlaySize?: number; // For round robins: size of each parlay (e.g., 2 for 2-leg parlays)
  stakePerLeg?: number; // Stake per individual bet in the parlay/round robin
}

export interface ParlayBet {
  id: string;
  legs: {
    selection: string;
    odds: number;
    outcome: 'win' | 'loss';
    context: Record<string, string | number>;
  }[];
  totalOdds: number;
  stake: number;
  payout: number;
  outcome: 'win' | 'loss';
  profit: number;
}

export interface RoundRobinBet {
  id: string;
  combinations: ParlayBet[];
  totalStake: number;
  totalPayout: number;
  netProfit: number;
  winningCombos: number;
  totalCombos: number;
}

export interface SimulationTrade {
  id: string;
  label: string;
  date: string;
  outcome: 'win' | 'loss';
  profit: number;
  context: Record<string, string | number>;
}

export interface SimulationSummary {
  rule: StrategyRule;
  totalTrades: number;
  wins: number;
  winRate: number;
  netProfit: number;
  avgReturn: number;
  trades: SimulationTrade[];
  notes: string[];
  // Advanced KPIs
  roiPercent?: number;      // Return on Investment as percentage
  profitFactor?: number;    // Gross profit / Gross loss (>1 is profitable)
  maxDrawdown?: number;     // Maximum peak-to-trough decline
  maxDrawdownPct?: number;  // Max drawdown as percentage of peak
  sharpeRatio?: number;     // Risk-adjusted return metric
  // Sports-specific KPIs
  atsRecord?: {             // Against The Spread record
    wins: number;
    losses: number;
    pushes: number;
    coverRate: number;
  };
}

export interface RagInsight {
  id: string;
  title: string;
  summary: string;
  tags: string[];
  support?: string;
  score: number;
}

export interface CryptoSnapshot {
  asset: string;
  priceUsd: number;
  fetchedAt: string;
  source: string;
}

export interface StrategyResponsePayload {
  summary: SimulationSummary;
  ragInsight?: RagInsight | null;
  externalSnapshot?: CryptoSnapshot[] | null;
  datasetSize: number;
  sportFilter?: string;  // The sport filter applied (nfl, nhl, cfb, etc)
}

export interface SportsGameRecord {
  type: 'sports';
  id: string;
  sport: string;  // nfl, nhl, nba, mlb, etc.
  date: string;
  team: string;
  opponent: string;
  losingStreakBefore: number;
  result: 'win' | 'loss';
  margin: number;
  odds: string;           // Home team moneyline odds
  awayOdds?: string;      // Away team moneyline odds (for underdog bets)
  spread: number;         // Home team spread (negative = home favorite)
  spreadOddsHome?: string; // Spread odds for home side (usually -110)
  spreadOddsAway?: string; // Spread odds for away side (usually -110)
  injuries: string;
  weather: string;
  // Extended fields from BallDontLie API
  homeScore?: number;
  awayScore?: number;
  season?: number;
  isPostseason?: boolean;
  gameId?: number;
}

export interface CryptoBarRecord {
  type: 'crypto';
  id: string;
  timestamp: string;
  asset: string;
  open: number;
  close: number;
  changePct: number;
  volume: number;
  newsSentiment: string;
}

export interface StockBarRecord {
  type: 'stocks';
  id: string;
  date: string;
  ticker: string;
  open: number;
  close: number;
  changePct: number;
  volume: number;
  earningSurprise: number;
  macroSignal: string;
}

export interface ForexBarRecord {
  type: 'forex';
  id: string;
  timestamp: string;
  pair: string; // e.g., 'EURUSD', 'GBPUSD'
  open: number;
  close: number;
  high: number;
  low: number;
  changePct: number;
  volume: number;
  spread: number; // pips
  newsImpact: string; // 'high', 'medium', 'low', 'none'
}

export type DomainRecord = SportsGameRecord | CryptoBarRecord | StockBarRecord | ForexBarRecord;


export enum SubscriptionTier {
  FREE = 'FREE',
  TRIAL = 'TRIAL',
  PRO = 'PRO',
  ELITE = 'ELITE'
}
