/**
 * Stripe Payment Processing Service
 * Handles subscription management, payments, and webhooks
 */

import Stripe from 'stripe';
import { DatabaseService } from './database';
import { logger, ErrorFactory } from './error-handler';

// Lazy Stripe initialization - only create when actually needed
let _stripe: Stripe | null = null;

function getStripe(): Stripe {
  if (!_stripe) {
    const secretKey = process.env.STRIPE_SECRET_KEY;
    if (!secretKey) {
      throw new Error('STRIPE_SECRET_KEY is not configured');
    }
    _stripe = new Stripe(secretKey, {
      apiVersion: '2024-12-18.acacia',
      typescript: true,
    });
  }
  return _stripe;
}

// Legacy stripe export for backward compatibility - now uses getter
const stripe = new Proxy({} as Stripe, {
  get: (_, prop) => {
    const stripeInstance = getStripe();
    return (stripeInstance as any)[prop];
  }
});

// Subscription plans configuration
export const SUBSCRIPTION_PLANS = {
  FREE: {
    name: 'Free',
    priceId: undefined,
    price: 0,
    interval: 'month',
    trialDays: 0,
    description: 'Free tier with limited features',
    features: {
      maxStrategies: 5,
      maxConversations: 20,
      ragAccess: false,
      advancedAnalytics: false,
      prioritySupport: false,
    },
  },
  TRIAL: {
    name: '7-Day Trial',
    priceId: process.env.STRIPE_FREE_PRICE_ID,
    price: 1,
    interval: 'month',
    trialDays: 7,
    description: '$1 for 7 days, then $20/month - cancel anytime',
    features: {
      maxStrategies: 100,
      maxConversations: 500,
      ragAccess: true,
      advancedAnalytics: true,
      prioritySupport: true,
    },
  },
  PRO: {
    name: 'Pro',
    priceId: process.env.STRIPE_PRO_PRICE_ID,
    price: 20,
    trialPrice: 1, // $1 for 7-day trial
    interval: 'month',
    trialDays: 7,
    description: '$1 for 7 days, then $20/month - Professional access with all features',
    features: {
      maxStrategies: 100,
      maxConversations: 500,
      ragAccess: true,
      advancedAnalytics: true,
      prioritySupport: true,
    },
  },
  ELITE: {
    name: 'Elite',
    priceId: process.env.STRIPE_ELITE_PRICE_ID,
    price: 200,
    trialPrice: 20, // $20 for 7-day trial
    interval: 'month',
    trialDays: 7,
    description: '$20 for 7 days, then $200/month - Unlimited access for power users',
    features: {
      maxStrategies: -1,
      maxConversations: -1,
      ragAccess: true,
      advancedAnalytics: true,
      prioritySupport: true,
    },
  },
} as const;

export type SubscriptionTier = keyof typeof SUBSCRIPTION_PLANS;

export class StripeService {
  // Customer Management
  static async createCustomer(userId: string, email: string, name?: string): Promise<string> {
    try {
      const customer = await stripe.customers.create({
        email,
        name,
        metadata: {
          userId,
        },
      });

      // Update user with Stripe customer ID
      await DatabaseService.findUserById(userId).then(user => {
        if (user) {
          // Note: This would require adding stripeCustomerId to the User model
          // For now, we'll store it in a separate table or use a different approach
          logger.info('Created Stripe customer', { userId, customerId: customer.id });
        }
      });

      return customer.id;
    } catch (error) {
      logger.error('Failed to create Stripe customer', { userId, error });
      throw ErrorFactory.externalAPI('Failed to create customer', 'Stripe');
    }
  }

  static async getCustomer(customerId: string) {
    try {
      return await stripe.customers.retrieve(customerId);
    } catch (error) {
      logger.error('Failed to retrieve Stripe customer', { customerId, error });
      throw ErrorFactory.externalAPI('Failed to retrieve customer', 'Stripe');
    }
  }

  // Subscription Management
  static async createSubscription(
    customerId: string,
    priceId: string,
    userId: string,
    trialDays?: number
  ) {
    try {
      const subscriptionData: Stripe.SubscriptionCreateParams = {
        customer: customerId,
        items: [{ price: priceId }],
        metadata: { userId },
        payment_behavior: 'default_incomplete',
        expand: ['latest_invoice.payment_intent'],
      };

      if (trialDays && trialDays > 0) {
        subscriptionData.trial_period_days = trialDays;
      }

      const subscription = await stripe.subscriptions.create(subscriptionData);

      logger.info('Created Stripe subscription', {
        userId,
        subscriptionId: subscription.id,
        priceId,
        trialDays,
      });

      return subscription;
    } catch (error) {
      logger.error('Failed to create Stripe subscription', { userId, priceId, error });
      throw ErrorFactory.externalAPI('Failed to create subscription', 'Stripe');
    }
  }

  static async cancelSubscription(subscriptionId: string, userId: string) {
    try {
      const subscription = await stripe.subscriptions.update(subscriptionId, {
        cancel_at_period_end: true,
      });

      // Update user in database
      await DatabaseService.findUserById(userId).then(user => {
        if (user) {
          // Update canceledAt timestamp
          logger.info('Cancelled Stripe subscription', {
            userId,
            subscriptionId,
            cancelAt: subscription.cancel_at,
          });
        }
      });

      return subscription;
    } catch (error) {
      logger.error('Failed to cancel Stripe subscription', { subscriptionId, userId, error });
      throw ErrorFactory.externalAPI('Failed to cancel subscription', 'Stripe');
    }
  }

  static async updateSubscription(subscriptionId: string, newPriceId: string, userId: string) {
    try {
      const subscription = await stripe.subscriptions.retrieve(subscriptionId);
      const subscriptionItemId = subscription.items.data[0].id;

      const updatedSubscription = await stripe.subscriptions.update(subscriptionId, {
        items: [{
          id: subscriptionItemId,
          price: newPriceId,
        }],
        proration_behavior: 'create_prorations',
      });

      logger.info('Updated Stripe subscription', {
        userId,
        subscriptionId,
        oldPriceId: subscription.items.data[0].price.id,
        newPriceId,
      });

      return updatedSubscription;
    } catch (error) {
      logger.error('Failed to update Stripe subscription', { subscriptionId, newPriceId, userId, error });
      throw ErrorFactory.externalAPI('Failed to update subscription', 'Stripe');
    }
  }

  // Payment Methods
  static async createSetupIntent(customerId: string) {
    try {
      const setupIntent = await stripe.setupIntents.create({
        customer: customerId,
        payment_method_types: ['card'],
      });

      return setupIntent;
    } catch (error) {
      logger.error('Failed to create setup intent', { customerId, error });
      throw ErrorFactory.externalAPI('Failed to create payment setup', 'Stripe');
    }
  }

  static async attachPaymentMethod(customerId: string, paymentMethodId: string) {
    try {
      const paymentMethod = await stripe.paymentMethods.attach(paymentMethodId, {
        customer: customerId,
      });

      // Set as default payment method
      await stripe.customers.update(customerId, {
        invoice_settings: {
          default_payment_method: paymentMethodId,
        },
      });

      return paymentMethod;
    } catch (error) {
      logger.error('Failed to attach payment method', { customerId, paymentMethodId, error });
      throw ErrorFactory.externalAPI('Failed to attach payment method', 'Stripe');
    }
  }

  // Webhook Handling
  static async constructEvent(payload: string, signature: string, webhookSecret: string) {
    try {
      return stripe.webhooks.constructEvent(payload, signature, webhookSecret);
    } catch (error) {
      logger.error('Failed to construct webhook event', { error });
      throw ErrorFactory.externalAPI('Invalid webhook signature', 'Stripe');
    }
  }

  static async handleWebhook(event: Stripe.Event) {
    const { type, data } = event;

    logger.info('Processing Stripe webhook', { type, id: event.id });

    try {
      switch (type) {
        case 'customer.subscription.created':
          await this.handleSubscriptionCreated(data.object as Stripe.Subscription);
          break;

        case 'customer.subscription.updated':
          await this.handleSubscriptionUpdated(data.object as Stripe.Subscription);
          break;

        case 'customer.subscription.deleted':
          await this.handleSubscriptionDeleted(data.object as Stripe.Subscription);
          break;

        case 'invoice.payment_succeeded':
          await this.handlePaymentSucceeded(data.object as Stripe.Invoice);
          break;

        case 'invoice.payment_failed':
          await this.handlePaymentFailed(data.object as Stripe.Invoice);
          break;

        default:
          logger.info('Unhandled webhook event type', { type });
      }
    } catch (error) {
      logger.error('Failed to handle webhook', { type, error });
      throw error;
    }
  }

  private static async handleSubscriptionCreated(subscription: Stripe.Subscription) {
    const userId = subscription.metadata.userId;
    if (!userId) return;

    const tier = this.getTierFromPriceId(subscription.items.data[0].price.id);

    await DatabaseService.findUserById(userId).then(async user => {
      if (user) {
        // Update user subscription info
        logger.info('Subscription created', { userId, tier, subscriptionId: subscription.id });
      }
    });
  }

  private static async handleSubscriptionUpdated(subscription: Stripe.Subscription) {
    const userId = subscription.metadata.userId;
    if (!userId) return;

    const tier = this.getTierFromPriceId(subscription.items.data[0].price.id);
    const status = subscription.status;

    await DatabaseService.findUserById(userId).then(async user => {
      if (user) {
        // Update user subscription status
        logger.info('Subscription updated', { userId, tier, status, subscriptionId: subscription.id });
      }
    });
  }

  private static async handleSubscriptionDeleted(subscription: Stripe.Subscription) {
    const userId = subscription.metadata.userId;
    if (!userId) return;

    await DatabaseService.findUserById(userId).then(async user => {
      if (user) {
        // Downgrade to free tier
        logger.info('Subscription cancelled', { userId, subscriptionId: subscription.id });
      }
    });
  }

  private static async handlePaymentSucceeded(invoice: Stripe.Invoice) {
    const subscription = await stripe.subscriptions.retrieve(invoice.subscription as string);
    const userId = subscription.metadata.userId;

    if (userId) {
      await DatabaseService.recordUserEvent(userId, 'PAYMENT_SUCCESS', {
        amount: invoice.amount_paid,
        currency: invoice.currency,
        invoiceId: invoice.id,
      });

      logger.info('Payment succeeded', { userId, amount: invoice.amount_paid, invoiceId: invoice.id });
    }
  }

  private static async handlePaymentFailed(invoice: Stripe.Invoice) {
    const subscription = await stripe.subscriptions.retrieve(invoice.subscription as string);
    const userId = subscription.metadata.userId;

    if (userId) {
      await DatabaseService.recordUserEvent(userId, 'PAYMENT_FAILED', {
        amount: invoice.amount_due,
        currency: invoice.currency,
        invoiceId: invoice.id,
        attemptCount: invoice.attempt_count,
      });

      logger.error('Payment failed', { userId, invoiceId: invoice.id });
    }
  }

  // Utility Methods
  static getTierFromPriceId(priceId: string): SubscriptionTier | null {
    for (const [tier, config] of Object.entries(SUBSCRIPTION_PLANS)) {
      if (config.priceId === priceId) {
        return tier as SubscriptionTier;
      }
    }
    return null;
  }

  static getPriceIdFromTier(tier: SubscriptionTier): string | null {
    return SUBSCRIPTION_PLANS[tier].priceId;
  }

  static getPlanConfig(tier: SubscriptionTier) {
    return SUBSCRIPTION_PLANS[tier];
  }

  static async getSubscription(subscriptionId: string) {
    try {
      return await stripe.subscriptions.retrieve(subscriptionId);
    } catch (error) {
      logger.error('Failed to retrieve subscription', { subscriptionId, error });
      throw ErrorFactory.externalAPI('Failed to retrieve subscription', 'Stripe');
    }
  }

  static async listCustomerSubscriptions(customerId: string) {
    try {
      return await stripe.subscriptions.list({
        customer: customerId,
        status: 'active',
      });
    } catch (error) {
      logger.error('Failed to list customer subscriptions', { customerId, error });
      throw ErrorFactory.externalAPI('Failed to list subscriptions', 'Stripe');
    }
  }
}

// Export Stripe instance for direct access if needed
export { stripe };
export default StripeService;
