import type {
  Strategy,
  StrategyComment,
  StrategyRating,
  StrategyCollection,
  StrategyTemplate,
  StrategySearchFilters,
  StrategyStats,
} from './types';
import { MemoryService } from '../memory/MemoryService';

export class StrategyLibraryService {
  private static instance: StrategyLibraryService;
  private memoryService: MemoryService;

  // In-memory storage (would be database in production)
  private strategies: Map<string, Strategy> = new Map();
  private collections: Map<string, StrategyCollection> = new Map();
  private templates: StrategyTemplate[] = [];

  private constructor() {
    this.memoryService = MemoryService.getInstance();
    this.initializeTemplates();
  }

  static getInstance(): StrategyLibraryService {
    if (!StrategyLibraryService.instance) {
      StrategyLibraryService.instance = new StrategyLibraryService();
    }
    return StrategyLibraryService.instance;
  }

  // Strategy CRUD Operations
  async createStrategy(strategyData: Omit<Strategy, 'id' | 'createdAt' | 'updatedAt' | 'version' | 'likes' | 'forks' | 'comments' | 'ratings'>): Promise<Strategy> {
    const strategy: Strategy = {
      ...strategyData,
      id: `strategy-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
      createdAt: new Date(),
      updatedAt: new Date(),
      version: 1,
      likes: 0,
      forks: 0,
      comments: [],
      ratings: [],
    };

    this.strategies.set(strategy.id, strategy);

    // Store in memory system for AI context
    await this.memoryService.rememberStrategy(strategy.authorId, {
      strategyId: strategy.id,
      name: strategy.name,
      domain: strategy.domain,
      code: strategy.code,
      performance: {
        winRate: 0, // Will be updated when tested
        totalTrades: 0,
        netProfit: 0,
        createdAt: strategy.createdAt,
      },
      context: {
        userPrompt: strategy.description,
        marketConditions: '',
        successFactors: [],
      },
      usageCount: 0,
      lastUsed: new Date(),
    });

    return strategy;
  }

  async getStrategy(strategyId: string): Promise<Strategy | null> {
    return this.strategies.get(strategyId) || null;
  }

  async updateStrategy(strategyId: string, updates: Partial<Strategy>): Promise<Strategy | null> {
    const strategy = this.strategies.get(strategyId);
    if (!strategy) return null;

    const updatedStrategy = {
      ...strategy,
      ...updates,
      updatedAt: new Date(),
      version: strategy.version + 1,
    };

    this.strategies.set(strategyId, updatedStrategy);
    return updatedStrategy;
  }

  async deleteStrategy(strategyId: string, userId: string): Promise<boolean> {
    const strategy = this.strategies.get(strategyId);
    if (!strategy || strategy.authorId !== userId) return false;

    this.strategies.delete(strategyId);
    return true;
  }

  async forkStrategy(strategyId: string, userId: string, userName: string, modifications?: Partial<Strategy>): Promise<Strategy | null> {
    const originalStrategy = this.strategies.get(strategyId);
    if (!originalStrategy) return null;

    const forkedStrategy = await this.createStrategy({
      ...originalStrategy,
      ...modifications,
      name: modifications?.name || `${originalStrategy.name} (Fork)`,
      authorId: userId,
      authorName: userName,
      isPublic: false, // Private by default
      parentStrategyId: strategyId,
    });

    // Increment fork count on original
    originalStrategy.forks++;
    this.strategies.set(strategyId, originalStrategy);

    return forkedStrategy;
  }

  // Search and Discovery
  async searchStrategies(filters: StrategySearchFilters, limit: number = 20): Promise<Strategy[]> {
    let results = Array.from(this.strategies.values());

    // Apply filters
    if (filters.domain && filters.domain.length > 0) {
      results = results.filter(s => filters.domain!.includes(s.domain));
    }

    if (filters.tags && filters.tags.length > 0) {
      results = results.filter(s =>
        filters.tags!.some(tag => s.tags.includes(tag))
      );
    }

    if (filters.author) {
      results = results.filter(s => s.authorName.toLowerCase().includes(filters.author!.toLowerCase()));
    }

    if (filters.riskLevel && filters.riskLevel.length > 0) {
      results = results.filter(s => filters.riskLevel!.includes(s.riskLevel));
    }

    if (filters.timeHorizon && filters.timeHorizon.length > 0) {
      results = results.filter(s => filters.timeHorizon!.includes(s.timeHorizon));
    }

    if (filters.isPublic !== undefined) {
      results = results.filter(s => s.isPublic === filters.isPublic);
    }

    if (filters.minRating) {
      results = results.filter(s => {
        const avgRating = s.ratings.length > 0
          ? s.ratings.reduce((sum, r) => sum + r.rating, 0) / s.ratings.length
          : 0;
        return avgRating >= filters.minRating!;
      });
    }

    if (filters.searchQuery) {
      const query = filters.searchQuery.toLowerCase();
      results = results.filter(s =>
        s.name.toLowerCase().includes(query) ||
        s.description.toLowerCase().includes(query) ||
        s.tags.some(tag => tag.toLowerCase().includes(query))
      );
    }

    // Apply sorting
    switch (filters.sortBy) {
      case 'newest':
        results.sort((a, b) => b.createdAt.getTime() - a.createdAt.getTime());
        break;
      case 'oldest':
        results.sort((a, b) => a.createdAt.getTime() - b.createdAt.getTime());
        break;
      case 'popular':
        results.sort((a, b) => b.likes - a.likes);
        break;
      case 'rating':
        results.sort((a, b) => {
          const aRating = a.ratings.length > 0 ? a.ratings.reduce((sum, r) => sum + r.rating, 0) / a.ratings.length : 0;
          const bRating = b.ratings.length > 0 ? b.ratings.reduce((sum, r) => sum + r.rating, 0) / b.ratings.length : 0;
          return bRating - aRating;
        });
        break;
      case 'forks':
        results.sort((a, b) => b.forks - a.forks);
        break;
      default:
        results.sort((a, b) => b.createdAt.getTime() - a.createdAt.getTime());
    }

    return results.slice(0, limit);
  }

  // Social Features
  async likeStrategy(strategyId: string, userId: string): Promise<boolean> {
    const strategy = this.strategies.get(strategyId);
    if (!strategy) return false;

    // In a real system, you'd check if user already liked it
    strategy.likes++;
    this.strategies.set(strategyId, strategy);
    return true;
  }

  async addComment(strategyId: string, comment: Omit<StrategyComment, 'id' | 'createdAt' | 'likes' | 'replies'>): Promise<StrategyComment | null> {
    const strategy = this.strategies.get(strategyId);
    if (!strategy) return null;

    const newComment: StrategyComment = {
      ...comment,
      id: `comment-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
      createdAt: new Date(),
      likes: 0,
      replies: [],
    };

    strategy.comments.push(newComment);
    this.strategies.set(strategyId, strategy);
    return newComment;
  }

  async rateStrategy(strategyId: string, userId: string, rating: 1 | 2 | 3 | 4 | 5, comment?: string): Promise<boolean> {
    const strategy = this.strategies.get(strategyId);
    if (!strategy) return false;

    // Remove existing rating from this user
    strategy.ratings = strategy.ratings.filter(r => r.userId !== userId);

    // Add new rating
    strategy.ratings.push({
      userId,
      rating,
      comment,
      createdAt: new Date(),
    });

    this.strategies.set(strategyId, strategy);
    return true;
  }

  // Collections
  async createCollection(collectionData: Omit<StrategyCollection, 'id' | 'createdAt' | 'updatedAt' | 'followers'>): Promise<StrategyCollection> {
    const collection: StrategyCollection = {
      ...collectionData,
      id: `collection-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
      createdAt: new Date(),
      updatedAt: new Date(),
      followers: 0,
    };

    this.collections.set(collection.id, collection);
    return collection;
  }

  async getCollection(collectionId: string): Promise<StrategyCollection | null> {
    return this.collections.get(collectionId) || null;
  }

  async addStrategyToCollection(collectionId: string, strategyId: string): Promise<boolean> {
    const collection = this.collections.get(collectionId);
    if (!collection || collection.strategies.includes(strategyId)) return false;

    collection.strategies.push(strategyId);
    collection.updatedAt = new Date();
    this.collections.set(collectionId, collection);
    return true;
  }

  // Templates
  private initializeTemplates(): void {
    this.templates = [
      {
        id: 'sports-momentum',
        name: 'Sports Momentum Strategy',
        description: 'Bet on teams that have won their last few games',
        domain: 'sports',
        category: 'momentum',
        difficulty: 'beginner',
        code: `function sportsMomentumStrategy(records: SportsGameRecord[]): SimulationTrade[] {
  const trades: SimulationTrade[] = [];

  records.forEach((record, idx) => {
    // Bet on teams with 2+ wins in last 3 games
    const recentGames = records.slice(Math.max(0, idx - 3), idx);
    const teamRecentGames = recentGames.filter(g => g.team === record.team);

    if (teamRecentGames.length >= 2) {
      const winRate = teamRecentGames.filter(g => g.result === 'win').length / teamRecentGames.length;

      if (winRate >= 0.67) { // 2 out of 3 wins
        const outcome = record.result === 'win' ? 'win' : 'loss';
        const profit = outcome === 'win' ? parseFloat(record.odds) * 0.9 : -1;

        trades.push({
          id: \`sports-\${idx}\`,
          label: \`\${record.team} momentum bet\`,
          date: record.date,
          outcome: outcome as 'win' | 'loss',
          profit: Number(profit.toFixed(4)),
          context: {
            winRate: Number((winRate * 100).toFixed(1)),
            recentGames: teamRecentGames.length,
          },
        });
      }
    }
  });

  return trades;
}`,
        placeholders: {},
        preview: 'Captures teams on hot streaks with 67%+ win rate over last 3 games',
      },
      {
        id: 'crypto-dip-buy',
        name: 'Crypto Dip Buying Strategy',
        description: 'Buy cryptocurrencies when they drop significantly',
        domain: 'crypto',
        category: 'mean-reversion',
        difficulty: 'intermediate',
        code: `function cryptoDipBuyStrategy(records: CryptoBarRecord[]): SimulationTrade[] {
  const trades: SimulationTrade[] = [];

  records.forEach((record, idx) => {
    // Buy when price drops 5% or more
    if (record.changePct < -0.05) {
      const nextRecord = records[idx + 1];
      const profit = nextRecord
        ? (nextRecord.close - record.close) / record.close
        : Math.min(Math.abs(record.changePct) / 2, 0.03);

      trades.push({
        id: \`crypto-\${idx}\`,
        label: \`\${record.asset} dip buy\`,
        date: record.timestamp,
        outcome: profit > 0 ? 'win' : 'loss',
        profit: Number(profit.toFixed(4)),
        context: {
          drop: Number((record.changePct * 100).toFixed(2)),
          volume: record.volume,
        },
      });
    }
  });

  return trades;
}`,
        placeholders: {},
        preview: 'Buys cryptocurrencies after 5%+ drops, capturing mean reversion',
      },
    ];
  }

  async getTemplates(domain?: 'sports' | 'crypto' | 'stocks' | 'forex'): Promise<StrategyTemplate[]> {
    if (domain) {
      return this.templates.filter(t => t.domain === domain);
    }
    return this.templates;
  }

  async getTemplate(templateId: string): Promise<StrategyTemplate | null> {
    return this.templates.find(t => t.id === templateId) || null;
  }

  async createFromTemplate(templateId: string, userId: string, userName: string, customizations?: Record<string, any>): Promise<Strategy | null> {
    const template = this.templates.find(t => t.id === templateId);
    if (!template) return null;

    let code = template.code;

    // Apply customizations
    if (customizations) {
      Object.entries(customizations).forEach(([key, value]) => {
        const placeholder = template.placeholders[key];
        if (placeholder) {
          code = code.replace(new RegExp(placeholder, 'g'), String(value));
        }
      });
    }

    return this.createStrategy({
      name: `${template.name} (Custom)`,
      description: template.description,
      domain: template.domain,
      code,
      tags: [template.category, template.difficulty],
      isPublic: false,
      authorId: userId,
      authorName: userName,
      riskLevel: 'medium',
      timeHorizon: 'medium',
      expectedReturn: 0.05, // 5% expected return
    });
  }

  // Statistics and Analytics
  async getStats(): Promise<StrategyStats> {
    const strategies = Array.from(this.strategies.values());
    const publicStrategies = strategies.filter(s => s.isPublic);

    // Calculate tag popularity
    const tagCounts: Record<string, number> = {};
    strategies.forEach(s => {
      s.tags.forEach(tag => {
        tagCounts[tag] = (tagCounts[tag] || 0) + 1;
      });
    });

    const popularTags = Object.entries(tagCounts)
      .sort(([,a], [,b]) => b - a)
      .slice(0, 10)
      .map(([tag, count]) => ({ tag, count }));

    // Domain distribution
    const domainDistribution: Record<string, number> = {};
    strategies.forEach(s => {
      domainDistribution[s.domain] = (domainDistribution[s.domain] || 0) + 1;
    });

    // Mock recent activity (in real system, this would be tracked)
    const recentActivity = strategies
      .sort((a, b) => b.createdAt.getTime() - a.createdAt.getTime())
      .slice(0, 5)
      .map(s => ({
        type: 'created' as const,
        strategyId: s.id,
        strategyName: s.name,
        userId: s.authorId,
        userName: s.authorName,
        timestamp: s.createdAt,
      }));

    return {
      totalStrategies: strategies.length,
      publicStrategies: publicStrategies.length,
      totalUsers: new Set(strategies.map(s => s.authorId)).size,
      popularTags,
      domainDistribution,
      recentActivity,
    };
  }

  // User-specific methods
  async getUserStrategies(userId: string): Promise<Strategy[]> {
    return Array.from(this.strategies.values())
      .filter(s => s.authorId === userId)
      .sort((a, b) => b.updatedAt.getTime() - a.updatedAt.getTime());
  }

  async getPublicStrategies(limit: number = 50): Promise<Strategy[]> {
    return Array.from(this.strategies.values())
      .filter(s => s.isPublic)
      .sort((a, b) => b.likes - a.likes)
      .slice(0, limit);
  }
}

