/**
 * Sports Betting Data Service
 * 
 * Unified data layer for all sports betting data with historical odds.
 * Provides clear feedback when data/odds are not available.
 * 
 * Data Source: BallDontLie API
 * - Historical NBA/NFL/NHL data with odds
 * - Live and current season data (2024-25)
 */

import fs from 'fs';
import path from 'path';

// ============================================================================
// TYPES
// ============================================================================

export interface BettingOdds {
  moneylineHome: number | null;
  moneylineAway: number | null;
  spreadHome: number | null;
  spreadAway: number | null;
  spreadHomeOdds?: number | null;
  spreadAwayOdds?: number | null;
  totalLine: number | null;
  overOdds?: number | null;
  underOdds?: number | null;
  source: 'historical' | 'live' | 'none';
}

export interface GameScore {
  homeScore: number | null;
  awayScore: number | null;
  homeQ1?: number | null;
  homeQ2?: number | null;
  homeQ3?: number | null;
  homeQ4?: number | null;
  awayQ1?: number | null;
  awayQ2?: number | null;
  awayQ3?: number | null;
  awayQ4?: number | null;
  overtime?: boolean;
}

export interface SportGame {
  id: string;
  sport: 'nba' | 'nfl' | 'nhl' | 'mlb' | 'ncaab' | 'ncaaf' | 'soccer';
  date: string;           // YYYY-MM-DD format
  season: number;
  homeTeam: string;
  awayTeam: string;
  scores: GameScore;
  odds: BettingOdds;
  result?: {
    winner: 'home' | 'away' | 'draw';
    spreadCovered: 'home' | 'away' | 'push' | null;
    totalResult: 'over' | 'under' | 'push' | null;
    margin: number;
    totalPoints: number;
  };
}

export interface DataAvailability {
  sport: string;
  hasGames: boolean;
  hasOdds: boolean;
  dateRange: {
    start: string;
    end: string;
  } | null;
  totalGames: number;
  gamesWithOdds: number;
  source: string;
  limitations: string[];
}

export interface DataQueryResult {
  success: boolean;
  games: SportGame[];
  availability: DataAvailability;
  warnings: string[];
  message: string;
}

// ============================================================================
// DATA STORAGE
// ============================================================================

const DATA_DIR = path.join(process.cwd(), 'data', 'betting');

// In-memory cache for loaded data
let dataCache: {
  nba: SportGame[];
  nfl: SportGame[];
  nhl: SportGame[];
  loaded: boolean;
  loadedAt: number | null;
} = {
  nba: [],
  nfl: [],
  nhl: [],
  loaded: false,
  loadedAt: null,
};

// ============================================================================
// DATA LOADING
// ============================================================================

/**
 * Load historical odds data from JSON files
 */
async function loadHistoricalData(): Promise<void> {
  if (dataCache.loaded && dataCache.loadedAt && Date.now() - dataCache.loadedAt < 3600000) {
    return; // Use cache if less than 1 hour old
  }

  console.log('[SportsBettingData] Loading historical betting data...');

  // Ensure data directory exists
  if (!fs.existsSync(DATA_DIR)) {
    fs.mkdirSync(DATA_DIR, { recursive: true });
  }

  // Load each sport's data
  const sports = ['nba', 'nfl', 'nhl'] as const;
  
  for (const sport of sports) {
    const filePath = path.join(DATA_DIR, `${sport}_historical.json`);
    
    if (fs.existsSync(filePath)) {
      try {
        const rawData = JSON.parse(fs.readFileSync(filePath, 'utf-8'));
        dataCache[sport] = rawData;
        console.log(`[SportsBettingData] Loaded ${rawData.length} ${sport.toUpperCase()} games`);
      } catch (error) {
        console.error(`[SportsBettingData] Error loading ${sport} data:`, error);
        dataCache[sport] = [];
      }
    } else {
      console.log(`[SportsBettingData] No ${sport} data file found at ${filePath}`);
      dataCache[sport] = [];
    }
  }

  dataCache.loaded = true;
  dataCache.loadedAt = Date.now();
}

// ============================================================================
// DATA AVAILABILITY
// ============================================================================

/**
 * Get data availability summary for all sports
 */
export async function getDataAvailability(): Promise<Record<string, DataAvailability>> {
  await loadHistoricalData();

  const availability: Record<string, DataAvailability> = {};

  const sportConfigs = [
    { key: 'nba', name: 'NBA Basketball', data: dataCache.nba },
    { key: 'nfl', name: 'NFL Football', data: dataCache.nfl },
    { key: 'nhl', name: 'NHL Hockey', data: dataCache.nhl },
    { key: 'mlb', name: 'MLB Baseball', data: [] as SportGame[] },
    { key: 'ncaab', name: 'NCAA Basketball', data: [] as SportGame[] },
    { key: 'ncaaf', name: 'NCAA Football', data: [] as SportGame[] },
    { key: 'soccer', name: 'Soccer', data: [] as SportGame[] },
  ];

  for (const config of sportConfigs) {
    const games = config.data;
    const gamesWithOdds = games.filter(g => g.odds.source !== 'none').length;
    
    let dateRange: { start: string; end: string } | null = null;
    if (games.length > 0) {
      const dates = games.map(g => g.date).sort();
      dateRange = { start: dates[0], end: dates[dates.length - 1] };
    }

    const limitations: string[] = [];
    if (games.length === 0) {
      limitations.push('No data currently available');
      limitations.push('Data loading from BallDontLie API');
    } else if (gamesWithOdds < games.length) {
      limitations.push(`Only ${gamesWithOdds} of ${games.length} games have odds data`);
    }
    if (dateRange && dateRange.end < '2024-01-01') {
      limitations.push('Historical data - for current 2024-25 season data, use BallDontLie API');
    }

    availability[config.key] = {
      sport: config.name,
      hasGames: games.length > 0,
      hasOdds: gamesWithOdds > 0,
      dateRange,
      totalGames: games.length,
      gamesWithOdds,
      source: games.length > 0 ? 'BallDontLie API' : 'None',
      limitations,
    };
  }

  return availability;
}

// ============================================================================
// DATA QUERYING
// ============================================================================

export interface QueryOptions {
  sport: string;
  startDate?: string;
  endDate?: string;
  season?: number;
  team?: string;
  requireOdds?: boolean;
  limit?: number;
}

/**
 * Query betting data with comprehensive availability feedback
 */
export async function queryBettingData(options: QueryOptions): Promise<DataQueryResult> {
  await loadHistoricalData();

  const warnings: string[] = [];
  const sport = options.sport.toLowerCase();

  // Check if sport is supported
  const supportedSports = ['nba', 'nfl', 'nhl'];
  if (!supportedSports.includes(sport)) {
    return {
      success: false,
      games: [],
      availability: {
        sport: options.sport,
        hasGames: false,
        hasOdds: false,
        dateRange: null,
        totalGames: 0,
        gamesWithOdds: 0,
        source: 'None',
        limitations: [
          `${options.sport.toUpperCase()} data is not currently available`,
          'Available sports: NBA, NFL, NHL',
          'More sports available via BallDontLie API',
        ],
      },
      warnings: [`${options.sport.toUpperCase()} is not currently supported`],
      message: `⚠️ No data available for ${options.sport.toUpperCase()}. Currently supported: NBA, NFL, NHL.`,
    };
  }

  // Get data for the sport
  let games = [...(dataCache[sport as keyof typeof dataCache] as SportGame[] || [])];

  if (games.length === 0) {
    return {
      success: false,
      games: [],
      availability: {
        sport: options.sport.toUpperCase(),
        hasGames: false,
        hasOdds: false,
        dateRange: null,
        totalGames: 0,
        gamesWithOdds: 0,
        source: 'None',
        limitations: ['Data not loaded - please check data files'],
      },
      warnings: ['No data available'],
      message: `⚠️ No ${options.sport.toUpperCase()} data loaded. Please ensure data files are present.`,
    };
  }

  // Filter by date range
  if (options.startDate) {
    const beforeCount = games.length;
    games = games.filter(g => g.date >= options.startDate!);
    if (games.length < beforeCount) {
      warnings.push(`Filtered out ${beforeCount - games.length} games before ${options.startDate}`);
    }
  }
  
  if (options.endDate) {
    const beforeCount = games.length;
    games = games.filter(g => g.date <= options.endDate!);
    if (games.length < beforeCount) {
      warnings.push(`Filtered out ${beforeCount - games.length} games after ${options.endDate}`);
    }
  }

  // Filter by season
  if (options.season) {
    const beforeCount = games.length;
    games = games.filter(g => g.season === options.season);
    if (games.length === 0 && beforeCount > 0) {
      warnings.push(`No games found for season ${options.season}. Try 2024-25 season for current data.`);
    }
  }

  // Filter by team
  if (options.team) {
    const teamLower = options.team.toLowerCase();
    const beforeCount = games.length;
    games = games.filter(g => 
      g.homeTeam.toLowerCase().includes(teamLower) || 
      g.awayTeam.toLowerCase().includes(teamLower)
    );
    if (games.length === 0 && beforeCount > 0) {
      warnings.push(`No games found for team "${options.team}"`);
    }
  }

  // Filter by odds availability
  if (options.requireOdds) {
    const beforeCount = games.length;
    games = games.filter(g => g.odds.source !== 'none');
    if (games.length < beforeCount) {
      warnings.push(`${beforeCount - games.length} games excluded due to missing odds data`);
    }
  }

  // Note: Historical data ends at 2021, but BallDontLie API provides 2024-25 data
  // No warning needed as we support current season via BallDontLie API

  // Apply limit
  if (options.limit && games.length > options.limit) {
    games = games.slice(0, options.limit);
  }

  // Calculate availability stats
  const gamesWithOdds = games.filter(g => g.odds.source !== 'none').length;
  const dates = games.map(g => g.date).sort();

  const availability: DataAvailability = {
    sport: options.sport.toUpperCase(),
    hasGames: games.length > 0,
    hasOdds: gamesWithOdds > 0,
    dateRange: games.length > 0 ? { start: dates[0], end: dates[dates.length - 1] } : null,
    totalGames: games.length,
    gamesWithOdds,
    source: 'BallDontLie API',
    limitations: [],
  };

  // Build user-friendly message
  let message = '';
  if (games.length === 0) {
    message = `No ${options.sport.toUpperCase()} games found matching your criteria.`;
    if (warnings.length > 0) {
      message += ` ${warnings[0]}`;
    }
  } else if (gamesWithOdds === 0) {
    message = `Found ${games.length} ${options.sport.toUpperCase()} games, but ⚠️ NO ODDS DATA available for these games.`;
  } else if (gamesWithOdds < games.length) {
    message = `Found ${games.length} ${options.sport.toUpperCase()} games. ⚠️ Only ${gamesWithOdds} games have odds data (${Math.round(gamesWithOdds/games.length*100)}%).`;
  } else {
    message = `✅ Found ${games.length} ${options.sport.toUpperCase()} games with complete odds data.`;
  }

  return {
    success: games.length > 0,
    games,
    availability,
    warnings,
    message,
  };
}

// ============================================================================
// DATA CATALOG FOR RAG/CHAT
// ============================================================================

/**
 * Get a structured summary of all available data for the RAG system
 */
export async function getDataCatalogForRAG(): Promise<string> {
  const availability = await getDataAvailability();
  
  let catalog = `
# Sports Betting Data Catalog

## Available Data Summary

`;

  for (const [sport, info] of Object.entries(availability)) {
    const status = info.hasGames ? (info.hasOdds ? '✅' : '⚠️') : '❌';
    catalog += `### ${status} ${info.sport}
- **Games Available:** ${info.totalGames.toLocaleString()}
- **Games with Odds:** ${info.gamesWithOdds.toLocaleString()}
- **Date Range:** ${info.dateRange ? `${info.dateRange.start} to ${info.dateRange.end}` : 'N/A'}
- **Data Source:** ${info.source}
`;
    if (info.limitations.length > 0) {
      catalog += `- **Limitations:**\n`;
      for (const lim of info.limitations) {
        catalog += `  - ${lim}\n`;
      }
    }
    catalog += '\n';
  }

  catalog += `
## Betting Data Fields

For each game with odds, the following data is available:

### Moneyline Odds
- Home team moneyline (e.g., -150)
- Away team moneyline (e.g., +130)

### Point Spread
- Home spread (e.g., -6.5)
- Away spread (e.g., +6.5)

### Totals (Over/Under)
- Total line (e.g., 215.5)

### Game Results
- Final scores (including quarter-by-quarter for basketball)
- Winner determination
- Spread coverage result
- Over/Under result

## Important Notes

1. **Data Source:** All data provided exclusively by BallDontLie API
2. **Current Season (2024-25):** Full support for current NBA/NFL/NHL season data
3. **Backtesting:** Users can backtest strategies against available historical data
4. **No Odds Warning:** If a user requests data without odds, they will be notified

## Usage Examples

- "Backtest home favorites in NBA 2024-25 season" → Uses BallDontLie API data
- "Show me Lakers games with spread data" → Filters for odds availability
- "Test over/under strategy for NFL 2024" → Uses BallDontLie API data
`;

  return catalog;
}

// ============================================================================
// BACKTEST HELPER
// ============================================================================

export interface BacktestResult {
  totalBets: number;
  wins: number;
  losses: number;
  pushes: number;
  winRate: number;
  roi: number;
  profit: number;
  bets: Array<{
    date: string;
    matchup: string;
    betType: string;
    odds: number;
    result: 'win' | 'loss' | 'push';
    profit: number;
  }>;
  dataWarnings: string[];
}

/**
 * Run a simple backtest on the betting data
 */
export async function runBacktest(
  sport: string,
  strategy: {
    type: 'moneyline' | 'spread' | 'total';
    condition: (game: SportGame) => boolean;
    pick: (game: SportGame) => 'home' | 'away' | 'over' | 'under';
  },
  options: {
    startDate?: string;
    endDate?: string;
    season?: number;
    unitSize?: number;
  } = {}
): Promise<BacktestResult> {
  const unitSize = options.unitSize || 100;
  
  const queryResult = await queryBettingData({
    sport,
    startDate: options.startDate,
    endDate: options.endDate,
    season: options.season,
    requireOdds: true,
  });

  const result: BacktestResult = {
    totalBets: 0,
    wins: 0,
    losses: 0,
    pushes: 0,
    winRate: 0,
    roi: 0,
    profit: 0,
    bets: [],
    dataWarnings: queryResult.warnings,
  };

  if (!queryResult.success || queryResult.games.length === 0) {
    result.dataWarnings.push(queryResult.message);
    return result;
  }

  for (const game of queryResult.games) {
    // Skip if game doesn't match strategy condition
    if (!strategy.condition(game)) continue;
    
    // Skip if no result data
    if (!game.result) continue;

    const pick = strategy.pick(game);
    let odds = 0;
    let betResult: 'win' | 'loss' | 'push' = 'loss';

    if (strategy.type === 'moneyline') {
      odds = pick === 'home' ? (game.odds.moneylineHome || -110) : (game.odds.moneylineAway || -110);
      betResult = game.result.winner === pick ? 'win' : 'loss';
    } else if (strategy.type === 'spread') {
      odds = -110; // Standard spread odds
      if (game.result.spreadCovered === pick) {
        betResult = 'win';
      } else if (game.result.spreadCovered === 'push') {
        betResult = 'push';
      }
    } else if (strategy.type === 'total') {
      odds = -110;
      if (game.result.totalResult === pick) {
        betResult = 'win';
      } else if (game.result.totalResult === 'push') {
        betResult = 'push';
      }
    }

    // Calculate profit
    let profit = 0;
    if (betResult === 'win') {
      profit = odds > 0 ? unitSize * (odds / 100) : unitSize * (100 / Math.abs(odds));
      result.wins++;
    } else if (betResult === 'loss') {
      profit = -unitSize;
      result.losses++;
    } else {
      result.pushes++;
    }

    result.totalBets++;
    result.profit += profit;

    result.bets.push({
      date: game.date,
      matchup: `${game.awayTeam} @ ${game.homeTeam}`,
      betType: `${strategy.type} ${pick}`,
      odds,
      result: betResult,
      profit,
    });
  }

  // Calculate final stats
  if (result.totalBets > 0) {
    result.winRate = result.wins / (result.wins + result.losses) * 100;
    result.roi = (result.profit / (result.totalBets * unitSize)) * 100;
  }

  return result;
}

// ============================================================================
// PARLAY BACKTEST HELPER (REAL ODDS)
// ============================================================================

type BacktestPick = 'home' | 'away' | 'over' | 'under';
type BacktestOutcome = 'win' | 'loss' | 'push';

export interface ParlayBacktestLeg {
  gameId: string;
  date: string;
  matchup: string;
  betType: 'moneyline' | 'spread' | 'total';
  pick: BacktestPick;
  oddsAmerican: number;
  oddsDecimal: number;
  result: BacktestOutcome;
}

export interface ParlayBacktestTrade {
  id: string;
  date: string;
  action: 'PARLAY';
  outcome: 'win' | 'loss';
  stake: number;
  payout: number;
  profit: number;
  return_pct: number;
  legs: ParlayBacktestLeg[];
  total_odds_decimal: number;
  winning_legs: number;
  total_legs: number;
  pushes: number;
}

export interface ParlayBacktestEngineResult {
  success: boolean;
  sport: string;
  market: string;
  strategy_name: string;
  parlay_size: number;
  stake_per_parlay: number;
  total_trades: number;
  win_rate: number;
  total_profit: number;
  roi: number;
  results: {
    win_rate: number;
    total_profit: number;
    profit_factor: number;
    sharpe_ratio: number;
    max_drawdown: number;
  };
  performance_summary: string;
  data_warnings: string[];
  all_trades: ParlayBacktestTrade[];
  trades: ParlayBacktestTrade[];
  detailed_statistics: {
    trade_analysis: {
      total_trades: number;
      winning_trades: number;
      losing_trades: number;
      win_rate_percent: number;
    };
    profit_loss_analysis: {
      total_profit: number;
      biggest_win: number;
      biggest_loss: number;
      smallest_win: number;
      smallest_loss: number;
      average_win: number;
      average_loss: number;
    };
    risk_metrics: {
      sharpe_ratio: number;
      max_drawdown: number;
      profit_volatility: number;
    };
    ratios_and_factors: {
      profit_factor: number;
      win_loss_ratio: number;
      expectancy_per_trade: number;
      kelly_fraction_percent: number;
      recovery_factor: number;
    };
    streak_analysis: {
      max_win_streak: number;
      max_loss_streak: number;
      avg_win_streak: number;
      avg_loss_streak: number;
      current_streak_type: 'win' | 'loss' | null;
      current_streak_length: number;
    };
  };
  chart_visualization: {
    labels: string[];
    datasets: Array<{
      label: string;
      data: number[];
      borderColor?: string;
      backgroundColor?: string;
    }>;
    summary: {
      total_trades: number;
      total_profit: number;
      win_rate: number;
      roi: number;
    };
  };
  verification_data: {
    data_source: string;
    calculation_method: string;
    verification_status: string;
    timestamp: string;
  };
}

function americanToDecimalOdds(oddsAmerican: number): number {
  if (!Number.isFinite(oddsAmerican) || oddsAmerican === 0) return 1.0;
  if (oddsAmerican > 0) return 1 + oddsAmerican / 100;
  return 1 + 100 / Math.abs(oddsAmerican);
}

function computeMaxDrawdown(values: number[]): number {
  let peak = 0;
  let maxDrawdown = 0;
  let running = 0;
  for (const v of values) {
    running += v;
    if (running > peak) peak = running;
    const dd = running - peak; // <= 0
    if (dd < maxDrawdown) maxDrawdown = dd;
  }
  return Number(maxDrawdown.toFixed(2));
}

function computeStreaks(outcomes: Array<'win' | 'loss'>) {
  let maxWin = 0;
  let maxLoss = 0;
  let currentType: 'win' | 'loss' | null = null;
  let currentLen = 0;
  let winRuns: number[] = [];
  let lossRuns: number[] = [];

  for (const o of outcomes) {
    if (currentType === o) {
      currentLen += 1;
    } else {
      if (currentType === 'win') winRuns.push(currentLen);
      if (currentType === 'loss') lossRuns.push(currentLen);
      currentType = o;
      currentLen = 1;
    }
    if (o === 'win') maxWin = Math.max(maxWin, currentLen);
    if (o === 'loss') maxLoss = Math.max(maxLoss, currentLen);
  }

  if (currentType === 'win') winRuns.push(currentLen);
  if (currentType === 'loss') lossRuns.push(currentLen);

  const avg = (xs: number[]) =>
    xs.length ? xs.reduce((a, b) => a + b, 0) / xs.length : 0;

  return {
    max_win_streak: maxWin,
    max_loss_streak: maxLoss,
    avg_win_streak: Number(avg(winRuns).toFixed(2)),
    avg_loss_streak: Number(avg(lossRuns).toFixed(2)),
    current_streak_type: currentType,
    current_streak_length: currentLen,
  };
}

function computeSharpe(returns: number[]): number {
  if (!returns.length) return 0;
  const mean = returns.reduce((a, b) => a + b, 0) / returns.length;
  const variance =
    returns.reduce((acc, r) => acc + (r - mean) ** 2, 0) / returns.length;
  const std = Math.sqrt(variance);
  if (!std) return 0;
  // Scale by sqrt(n) so longer samples don't look artificially "worse"
  return Number(((mean / std) * Math.sqrt(returns.length)).toFixed(3));
}

/**
 * Backtest a simple N-leg parlay strategy on historical odds.
 *
 * Current implementation is intentionally conservative + explainable:
 * - Builds at most 1 parlay per day (the top N qualifying legs by implied probability)
 * - Uses real closing odds and real results
 */
export async function runParlayBacktest(
  sport: string,
  strategy: {
    type: 'moneyline' | 'spread' | 'total';
    condition: (game: SportGame) => boolean;
    pick: (game: SportGame) => BacktestPick;
  },
  options: {
    startDate?: string;
    endDate?: string;
    season?: number;
    parlaySize?: number;
    stakePerParlay?: number;
    limitDays?: number;
  } = {},
): Promise<ParlayBacktestEngineResult> {
  const parlaySize = Math.max(2, Math.min(10, options.parlaySize || 2));
  const stake = Math.max(1, options.stakePerParlay || 10);

  const queryResult = await queryBettingData({
    sport,
    startDate: options.startDate,
    endDate: options.endDate,
    season: options.season,
    requireOdds: true,
    limit: 50000,
  });

  const warnings = [...queryResult.warnings];
  if (!queryResult.success || queryResult.games.length === 0) {
    warnings.push(queryResult.message);

    return {
      success: false,
      sport,
      market: sport,
      strategy_name: 'parlay',
      parlay_size: parlaySize,
      stake_per_parlay: stake,
      total_trades: 0,
      win_rate: 0,
      total_profit: 0,
      roi: 0,
      results: {
        win_rate: 0,
        total_profit: 0,
        profit_factor: 0,
        sharpe_ratio: 0,
        max_drawdown: 0,
      },
      performance_summary: `⚠️ No qualifying ${sport.toUpperCase()} games with odds found for this time period. ${queryResult.message}`,
      data_warnings: warnings,
      all_trades: [],
      trades: [],
      detailed_statistics: {
        trade_analysis: { total_trades: 0, winning_trades: 0, losing_trades: 0, win_rate_percent: 0 },
        profit_loss_analysis: {
          total_profit: 0, biggest_win: 0, biggest_loss: 0, smallest_win: 0, smallest_loss: 0, average_win: 0, average_loss: 0,
        },
        risk_metrics: { sharpe_ratio: 0, max_drawdown: 0, profit_volatility: 0 },
        ratios_and_factors: { profit_factor: 0, win_loss_ratio: 0, expectancy_per_trade: 0, kelly_fraction_percent: 0, recovery_factor: 0 },
        streak_analysis: { max_win_streak: 0, max_loss_streak: 0, avg_win_streak: 0, avg_loss_streak: 0, current_streak_type: null, current_streak_length: 0 },
      },
      chart_visualization: { labels: [], datasets: [{ label: 'Cumulative Profit', data: [] }], summary: { total_trades: 0, total_profit: 0, win_rate: 0, roi: 0 } },
      verification_data: {
        data_source: 'BallDontLie API',
        calculation_method: 'Parlay backtest (daily top-N legs) using American→Decimal odds conversion',
        verification_status: 'no_data',
        timestamp: new Date().toISOString(),
      },
    };
  }

  // Build qualifying single-leg candidates
  type Candidate = {
    date: string;
    impliedProb: number;
    leg: ParlayBacktestLeg;
  };

  const candidates: Candidate[] = [];
  for (const game of queryResult.games) {
    if (!game.result) continue;
    if (!strategy.condition(game)) continue;

    const pick = strategy.pick(game);
    let oddsAmerican: number | null = null;
    let betResult: BacktestOutcome = 'loss';

    if (strategy.type === 'moneyline') {
      oddsAmerican = pick === 'home' ? game.odds.moneylineHome : game.odds.moneylineAway;
      if (oddsAmerican == null) continue;
      betResult = game.result.winner === pick ? 'win' : 'loss';
    } else if (strategy.type === 'spread') {
      oddsAmerican = pick === 'home'
        ? (game.odds.spreadHomeOdds ?? -110)
        : (game.odds.spreadAwayOdds ?? -110);
      if (game.result.spreadCovered === pick) betResult = 'win';
      else if (game.result.spreadCovered === 'push') betResult = 'push';
      else betResult = 'loss';
    } else {
      // total
      oddsAmerican = pick === 'over'
        ? (game.odds.overOdds ?? -110)
        : (game.odds.underOdds ?? -110);
      if (game.result.totalResult === pick) betResult = 'win';
      else if (game.result.totalResult === 'push') betResult = 'push';
      else betResult = 'loss';
    }

    const oddsDecimal = americanToDecimalOdds(oddsAmerican);
    const impliedProb = oddsDecimal ? 1 / oddsDecimal : 0;

    const leg: ParlayBacktestLeg = {
      gameId: game.id,
      date: game.date,
      matchup: `${game.awayTeam} @ ${game.homeTeam}`,
      betType: strategy.type,
      pick,
      oddsAmerican,
      oddsDecimal: Number(oddsDecimal.toFixed(4)),
      result: betResult,
    };

    candidates.push({ date: game.date, impliedProb, leg });
  }

  // Group candidates by day
  const byDate = new Map<string, Candidate[]>();
  for (const c of candidates) {
    const list = byDate.get(c.date) ?? [];
    list.push(c);
    byDate.set(c.date, list);
  }

  const dates = Array.from(byDate.keys()).sort();
  const limitDays = options.limitDays ? Math.max(1, options.limitDays) : undefined;
  const datesToRun = limitDays ? dates.slice(0, limitDays) : dates;

  const trades: ParlayBacktestTrade[] = [];
  for (const date of datesToRun) {
    const dayCandidates = byDate.get(date) ?? [];
    if (dayCandidates.length < parlaySize) continue;

    // Deterministic selection: choose top N legs by implied probability (highest win chance)
    dayCandidates.sort((a, b) => b.impliedProb - a.impliedProb);
    const legs = dayCandidates.slice(0, parlaySize).map((c) => c.leg);

    const hasLoss = legs.some((l) => l.result === 'loss');
    const winLegs = legs.filter((l) => l.result === 'win').length;
    const pushLegs = legs.filter((l) => l.result === 'push').length;

    const totalOddsDecimal = legs.reduce((acc, l) => {
      if (l.result === 'push') return acc; // push leg treated as 1.0 multiplier
      return acc * l.oddsDecimal;
    }, 1);

    const payout = hasLoss ? 0 : (stake * totalOddsDecimal);
    const profit = hasLoss ? -stake : Number((payout - stake).toFixed(2));
    const outcome: 'win' | 'loss' = profit >= 0 ? 'win' : 'loss';
    const returnPct = stake ? Number(((profit / stake) * 100).toFixed(2)) : 0;

    trades.push({
      id: `parlay-${sport}-${date}`,
      date,
      action: 'PARLAY',
      outcome,
      stake,
      payout: Number(payout.toFixed(2)),
      profit,
      return_pct: returnPct,
      legs,
      total_odds_decimal: Number(totalOddsDecimal.toFixed(4)),
      winning_legs: winLegs,
      total_legs: legs.length,
      pushes: pushLegs,
    });
  }

  // Sort chronologically (already by date, but keep safe)
  trades.sort((a, b) => a.date.localeCompare(b.date));

  const totalTrades = trades.length;
  const totalProfit = Number(trades.reduce((sum, t) => sum + t.profit, 0).toFixed(2));
  const totalStake = Number((totalTrades * stake).toFixed(2));

  const wins = trades.filter((t) => t.profit > 0).length;
  const losses = trades.filter((t) => t.profit < 0).length;
  const pushes = trades.filter((t) => t.profit === 0).length;
  const winRate = wins + losses ? Number(((wins / (wins + losses)) * 100).toFixed(2)) : 0;
  const roi = totalStake ? Number(((totalProfit / totalStake) * 100).toFixed(2)) : 0;

  const winProfits = trades.filter(t => t.profit > 0).map(t => t.profit);
  const lossProfits = trades.filter(t => t.profit < 0).map(t => t.profit);
  const sumWins = winProfits.reduce((a, b) => a + b, 0);
  const sumLossesAbs = Math.abs(lossProfits.reduce((a, b) => a + b, 0));
  const profitFactor = sumLossesAbs ? Number((sumWins / sumLossesAbs).toFixed(3)) : (sumWins ? 999 : 0);

  const returns = trades.map((t) => (t.stake ? t.profit / t.stake : 0));
  const sharpe = computeSharpe(returns);
  const maxDrawdown = computeMaxDrawdown(trades.map(t => t.profit));

  const profitVolatility = (() => {
    if (!trades.length) return 0;
    const profits = trades.map(t => t.profit);
    const mean = profits.reduce((a, b) => a + b, 0) / profits.length;
    const variance = profits.reduce((acc, p) => acc + (p - mean) ** 2, 0) / profits.length;
    return Number(Math.sqrt(variance).toFixed(2));
  })();

  const winLossRatio = (() => {
    const avgWin = winProfits.length ? sumWins / winProfits.length : 0;
    const avgLoss = lossProfits.length ? Math.abs(lossProfits.reduce((a, b) => a + b, 0)) / lossProfits.length : 0;
    return avgLoss ? Number((avgWin / avgLoss).toFixed(3)) : (avgWin ? 999 : 0);
  })();

  const expectancyPerTrade = totalTrades ? Number((totalProfit / totalTrades).toFixed(2)) : 0;
  const recoveryFactor = maxDrawdown ? Number((totalProfit / Math.abs(maxDrawdown)).toFixed(3)) : (totalProfit ? 999 : 0);

  const kellyFraction = (() => {
    if ((wins + losses) === 0) return 0;
    const p = wins + losses ? wins / (wins + losses) : 0;
    const q = 1 - p;
    const b = Math.max(winLossRatio, 0.0001);
    const f = (b * p - q) / b;
    return Number((Math.max(0, Math.min(f, 1)) * 100).toFixed(2));
  })();

  const streaks = computeStreaks(trades.map(t => t.profit >= 0 ? 'win' : 'loss'));

  // Chart visualization (client render)
  const labels = trades.map(t => t.date);
  const cumulative: number[] = [];
  let running = 0;
  for (const t of trades) {
    running += t.profit;
    cumulative.push(Number(running.toFixed(2)));
  }

  const performanceSummary = [
    `✅ Parlay backtest complete (${sport.toUpperCase()})`,
    `Leg rule: ${strategy.type} legs`,
    `Parlay size: ${parlaySize} legs | Stake: $${stake.toFixed(2)} per parlay`,
    `Trades: ${totalTrades} parlays (1 per day max)`,
    `Win rate: ${winRate}% | ROI: ${roi}% | Total P&L: $${totalProfit.toFixed(2)}`,
    `Note: Parlays have high variance; small edges can swing results dramatically.`,
  ].join('\n');

  return {
    success: true,
    sport,
    market: sport,
    strategy_name: 'parlay',
    parlay_size: parlaySize,
    stake_per_parlay: stake,
    total_trades: totalTrades,
    win_rate: winRate,
    total_profit: totalProfit,
    roi,
    results: {
      win_rate: winRate,
      total_profit: totalProfit,
      profit_factor: profitFactor,
      sharpe_ratio: sharpe,
      max_drawdown: maxDrawdown,
    },
    performance_summary: performanceSummary,
    data_warnings: warnings,
    all_trades: trades,
    trades: trades.slice(-50),
    detailed_statistics: {
      trade_analysis: {
        total_trades: totalTrades,
        winning_trades: wins,
        losing_trades: losses,
        win_rate_percent: winRate,
      },
      profit_loss_analysis: {
        total_profit: totalProfit,
        biggest_win: winProfits.length ? Number(Math.max(...winProfits).toFixed(2)) : 0,
        biggest_loss: lossProfits.length ? Number(Math.min(...lossProfits).toFixed(2)) : 0,
        smallest_win: winProfits.length ? Number(Math.min(...winProfits).toFixed(2)) : 0,
        smallest_loss: lossProfits.length ? Number(Math.max(...lossProfits).toFixed(2)) : 0,
        average_win: winProfits.length ? Number((sumWins / winProfits.length).toFixed(2)) : 0,
        average_loss: lossProfits.length ? Number((Math.abs(lossProfits.reduce((a, b) => a + b, 0)) / lossProfits.length).toFixed(2)) : 0,
      },
      risk_metrics: {
        sharpe_ratio: sharpe,
        max_drawdown: maxDrawdown,
        profit_volatility: profitVolatility,
      },
      ratios_and_factors: {
        profit_factor: profitFactor,
        win_loss_ratio: winLossRatio,
        expectancy_per_trade: expectancyPerTrade,
        kelly_fraction_percent: kellyFraction,
        recovery_factor: recoveryFactor,
      },
      streak_analysis: streaks,
    },
    chart_visualization: {
      labels,
      datasets: [
        {
          label: 'Cumulative Profit',
          data: cumulative,
          borderColor: '#60a5fa',
          backgroundColor: 'rgba(96,165,250,0.15)',
        },
      ],
      summary: {
        total_trades: totalTrades,
        total_profit: totalProfit,
        win_rate: winRate,
        roi,
      },
    },
    verification_data: {
      data_source: 'BallDontLie API',
      calculation_method: 'Parlay backtest (daily top-N qualifying legs) using American→Decimal odds conversion',
      verification_status: 'computed',
      timestamp: new Date().toISOString(),
    },
  };
}

function combinations<T>(items: T[], size: number): T[][] {
  const result: T[][] = [];
  if (size <= 0 || size > items.length) return result;

  const walk = (start: number, current: T[]) => {
    if (current.length === size) {
      result.push([...current]);
      return;
    }
    for (let i = start; i < items.length; i++) {
      current.push(items[i]);
      walk(i + 1, current);
      current.pop();
    }
  };

  walk(0, []);
  return result;
}

/**
 * Backtest a simple round-robin (by N's) on historical odds.
 *
 * Implementation notes:
 * - For each day, we select the top `selectionsPerDay` qualifying legs by implied probability
 * - Then we generate combinations of size `parlaySize` (\"by N's\") and treat each combo as a mini-parlay
 * - We cap combos per day to prevent combinatorial explosion in chat
 */
export async function runRoundRobinBacktest(
  sport: string,
  strategy: {
    type: 'moneyline' | 'spread' | 'total';
    condition: (game: SportGame) => boolean;
    pick: (game: SportGame) => BacktestPick;
  },
  options: {
    startDate?: string;
    endDate?: string;
    season?: number;
    parlaySize?: number;
    stakePerCombo?: number;
    selectionsPerDay?: number;
    maxCombosPerDay?: number;
    limitDays?: number;
  } = {},
): Promise<ParlayBacktestEngineResult> {
  const parlaySize = Math.max(2, Math.min(10, options.parlaySize || 2));
  const stake = Math.max(1, options.stakePerCombo || 10);
  const selectionsPerDay = Math.max(parlaySize, Math.min(12, options.selectionsPerDay || 6));
  const maxCombosPerDay = Math.max(1, Math.min(200, options.maxCombosPerDay || 50));

  const queryResult = await queryBettingData({
    sport,
    startDate: options.startDate,
    endDate: options.endDate,
    season: options.season,
    requireOdds: true,
    limit: 50000,
  });

  const warnings = [...queryResult.warnings];
  if (!queryResult.success || queryResult.games.length === 0) {
    warnings.push(queryResult.message);
    return {
      success: false,
      sport,
      market: sport,
      strategy_name: 'round_robin',
      parlay_size: parlaySize,
      stake_per_parlay: stake,
      total_trades: 0,
      win_rate: 0,
      total_profit: 0,
      roi: 0,
      results: {
        win_rate: 0,
        total_profit: 0,
        profit_factor: 0,
        sharpe_ratio: 0,
        max_drawdown: 0,
      },
      performance_summary: `⚠️ No qualifying ${sport.toUpperCase()} games with odds found for this time period. ${queryResult.message}`,
      data_warnings: warnings,
      all_trades: [],
      trades: [],
      detailed_statistics: {
        trade_analysis: { total_trades: 0, winning_trades: 0, losing_trades: 0, win_rate_percent: 0 },
        profit_loss_analysis: {
          total_profit: 0, biggest_win: 0, biggest_loss: 0, smallest_win: 0, smallest_loss: 0, average_win: 0, average_loss: 0,
        },
        risk_metrics: { sharpe_ratio: 0, max_drawdown: 0, profit_volatility: 0 },
        ratios_and_factors: { profit_factor: 0, win_loss_ratio: 0, expectancy_per_trade: 0, kelly_fraction_percent: 0, recovery_factor: 0 },
        streak_analysis: { max_win_streak: 0, max_loss_streak: 0, avg_win_streak: 0, avg_loss_streak: 0, current_streak_type: null, current_streak_length: 0 },
      },
      chart_visualization: {
        labels: [],
        datasets: [{ label: 'Cumulative Profit', data: [] }],
        summary: { total_trades: 0, total_profit: 0, win_rate: 0, roi: 0 },
      },
      verification_data: {
        data_source: 'BallDontLie API',
        calculation_method: 'Round robin backtest (daily top-M pool, combos by N) using American→Decimal odds conversion',
        verification_status: 'no_data',
        timestamp: new Date().toISOString(),
      },
    };
  }

  type Candidate = { date: string; impliedProb: number; leg: ParlayBacktestLeg };
  const candidates: Candidate[] = [];
  for (const game of queryResult.games) {
    if (!game.result) continue;
    if (!strategy.condition(game)) continue;

    const pick = strategy.pick(game);
    let oddsAmerican: number | null = null;
    let betResult: BacktestOutcome = 'loss';

    if (strategy.type === 'moneyline') {
      oddsAmerican = pick === 'home' ? game.odds.moneylineHome : game.odds.moneylineAway;
      if (oddsAmerican == null) continue;
      betResult = game.result.winner === pick ? 'win' : 'loss';
    } else if (strategy.type === 'spread') {
      oddsAmerican = pick === 'home'
        ? (game.odds.spreadHomeOdds ?? -110)
        : (game.odds.spreadAwayOdds ?? -110);
      if (game.result.spreadCovered === pick) betResult = 'win';
      else if (game.result.spreadCovered === 'push') betResult = 'push';
      else betResult = 'loss';
    } else {
      oddsAmerican = pick === 'over'
        ? (game.odds.overOdds ?? -110)
        : (game.odds.underOdds ?? -110);
      if (game.result.totalResult === pick) betResult = 'win';
      else if (game.result.totalResult === 'push') betResult = 'push';
      else betResult = 'loss';
    }

    const oddsDecimal = americanToDecimalOdds(oddsAmerican);
    const impliedProb = oddsDecimal ? 1 / oddsDecimal : 0;

    candidates.push({
      date: game.date,
      impliedProb,
      leg: {
        gameId: game.id,
        date: game.date,
        matchup: `${game.awayTeam} @ ${game.homeTeam}`,
        betType: strategy.type,
        pick,
        oddsAmerican,
        oddsDecimal: Number(oddsDecimal.toFixed(4)),
        result: betResult,
      }
    });
  }

  const byDate = new Map<string, Candidate[]>();
  for (const c of candidates) {
    const list = byDate.get(c.date) ?? [];
    list.push(c);
    byDate.set(c.date, list);
  }

  const dates = Array.from(byDate.keys()).sort();
  const limitDays = options.limitDays ? Math.max(1, options.limitDays) : undefined;
  const datesToRun = limitDays ? dates.slice(0, limitDays) : dates;

  const trades: ParlayBacktestTrade[] = [];
  for (const date of datesToRun) {
    const dayCandidates = byDate.get(date) ?? [];
    if (dayCandidates.length < parlaySize) continue;

    dayCandidates.sort((a, b) => b.impliedProb - a.impliedProb);
    const pool = dayCandidates.slice(0, selectionsPerDay).map(c => c.leg);
    if (pool.length < parlaySize) continue;

    const combos = combinations(pool, parlaySize).slice(0, maxCombosPerDay);
    combos.forEach((legs, idx) => {
      const hasLoss = legs.some((l) => l.result === 'loss');
      const winLegs = legs.filter((l) => l.result === 'win').length;
      const pushLegs = legs.filter((l) => l.result === 'push').length;

      const totalOddsDecimal = legs.reduce((acc, l) => {
        if (l.result === 'push') return acc;
        return acc * l.oddsDecimal;
      }, 1);

      const payout = hasLoss ? 0 : (stake * totalOddsDecimal);
      const profit = hasLoss ? -stake : Number((payout - stake).toFixed(2));
      const outcome: 'win' | 'loss' = profit >= 0 ? 'win' : 'loss';
      const returnPct = stake ? Number(((profit / stake) * 100).toFixed(2)) : 0;

      trades.push({
        id: `rr-${sport}-${date}-${idx + 1}`,
        date,
        action: 'PARLAY',
        outcome,
        stake,
        payout: Number(payout.toFixed(2)),
        profit,
        return_pct: returnPct,
        legs,
        total_odds_decimal: Number(totalOddsDecimal.toFixed(4)),
        winning_legs: winLegs,
        total_legs: legs.length,
        pushes: pushLegs,
      });
    });
  }

  trades.sort((a, b) => a.date.localeCompare(b.date));

  const totalTrades = trades.length;
  const totalProfit = Number(trades.reduce((sum, t) => sum + t.profit, 0).toFixed(2));
  const totalStake = Number((totalTrades * stake).toFixed(2));
  const wins = trades.filter((t) => t.profit > 0).length;
  const losses = trades.filter((t) => t.profit < 0).length;
  const pushes = trades.filter((t) => t.profit === 0).length;
  const winRate = wins + losses ? Number(((wins / (wins + losses)) * 100).toFixed(2)) : 0;
  const roi = totalStake ? Number(((totalProfit / totalStake) * 100).toFixed(2)) : 0;

  const winProfits = trades.filter(t => t.profit > 0).map(t => t.profit);
  const lossProfits = trades.filter(t => t.profit < 0).map(t => t.profit);
  const sumWins = winProfits.reduce((a, b) => a + b, 0);
  const sumLossesAbs = Math.abs(lossProfits.reduce((a, b) => a + b, 0));
  const profitFactor = sumLossesAbs ? Number((sumWins / sumLossesAbs).toFixed(3)) : (sumWins ? 999 : 0);

  const returns = trades.map((t) => (t.stake ? t.profit / t.stake : 0));
  const sharpe = computeSharpe(returns);
  const maxDrawdown = computeMaxDrawdown(trades.map(t => t.profit));

  const profitVolatility = (() => {
    if (!trades.length) return 0;
    const profits = trades.map(t => t.profit);
    const mean = profits.reduce((a, b) => a + b, 0) / profits.length;
    const variance = profits.reduce((acc, p) => acc + (p - mean) ** 2, 0) / profits.length;
    return Number(Math.sqrt(variance).toFixed(2));
  })();

  const winLossRatio = (() => {
    const avgWin = winProfits.length ? sumWins / winProfits.length : 0;
    const avgLoss = lossProfits.length ? Math.abs(lossProfits.reduce((a, b) => a + b, 0)) / lossProfits.length : 0;
    return avgLoss ? Number((avgWin / avgLoss).toFixed(3)) : (avgWin ? 999 : 0);
  })();

  const expectancyPerTrade = totalTrades ? Number((totalProfit / totalTrades).toFixed(2)) : 0;
  const recoveryFactor = maxDrawdown ? Number((totalProfit / Math.abs(maxDrawdown)).toFixed(3)) : (totalProfit ? 999 : 0);

  const kellyFraction = (() => {
    if ((wins + losses) === 0) return 0;
    const p = wins + losses ? wins / (wins + losses) : 0;
    const q = 1 - p;
    const b = Math.max(winLossRatio, 0.0001);
    const f = (b * p - q) / b;
    return Number((Math.max(0, Math.min(f, 1)) * 100).toFixed(2));
  })();

  const streaks = computeStreaks(trades.map(t => t.profit >= 0 ? 'win' : 'loss'));

  const labels = trades.map(t => `${t.date}#${t.id.split('-').at(-1)}`);
  const cumulative: number[] = [];
  let running = 0;
  for (const t of trades) {
    running += t.profit;
    cumulative.push(Number(running.toFixed(2)));
  }

  const performanceSummary = [
    `✅ Round robin backtest complete (${sport.toUpperCase()})`,
    `Round robin by ${parlaySize}'s | Stake: $${stake.toFixed(2)} per combo`,
    `Daily pool: top ${selectionsPerDay} qualifying legs; max ${maxCombosPerDay} combos/day`,
    `Trades: ${totalTrades} combos`,
    `Win rate: ${winRate}% | ROI: ${roi}% | Total P&L: $${totalProfit.toFixed(2)}`,
    `Note: Round robins reduce variance vs a single parlay, but still carry significant risk.`,
  ].join('\n');

  // Re-use the same envelope format used by other backtests.
  return {
    success: true,
    sport,
    market: sport,
    strategy_name: 'round_robin',
    parlay_size: parlaySize,
    stake_per_parlay: stake,
    total_trades: totalTrades,
    win_rate: winRate,
    total_profit: totalProfit,
    roi,
    results: {
      win_rate: winRate,
      total_profit: totalProfit,
      profit_factor: profitFactor,
      sharpe_ratio: sharpe,
      max_drawdown: maxDrawdown,
    },
    performance_summary: performanceSummary,
    data_warnings: warnings,
    all_trades: trades,
    trades: trades.slice(-50),
    detailed_statistics: {
      trade_analysis: {
        total_trades: totalTrades,
        winning_trades: wins,
        losing_trades: losses,
        win_rate_percent: winRate,
      },
      profit_loss_analysis: {
        total_profit: totalProfit,
        biggest_win: winProfits.length ? Number(Math.max(...winProfits).toFixed(2)) : 0,
        biggest_loss: lossProfits.length ? Number(Math.min(...lossProfits).toFixed(2)) : 0,
        smallest_win: winProfits.length ? Number(Math.min(...winProfits).toFixed(2)) : 0,
        smallest_loss: lossProfits.length ? Number(Math.max(...lossProfits).toFixed(2)) : 0,
        average_win: winProfits.length ? Number((sumWins / winProfits.length).toFixed(2)) : 0,
        average_loss: lossProfits.length ? Number((Math.abs(lossProfits.reduce((a, b) => a + b, 0)) / lossProfits.length).toFixed(2)) : 0,
      },
      risk_metrics: {
        sharpe_ratio: sharpe,
        max_drawdown: maxDrawdown,
        profit_volatility: profitVolatility,
      },
      ratios_and_factors: {
        profit_factor: profitFactor,
        win_loss_ratio: winLossRatio,
        expectancy_per_trade: expectancyPerTrade,
        kelly_fraction_percent: kellyFraction,
        recovery_factor: recoveryFactor,
      },
      streak_analysis: streaks,
    },
    chart_visualization: {
      labels,
      datasets: [
        {
          label: 'Cumulative Profit',
          data: cumulative,
          borderColor: '#a78bfa',
          backgroundColor: 'rgba(167,139,250,0.15)',
        },
      ],
      summary: {
        total_trades: totalTrades,
        total_profit: totalProfit,
        win_rate: winRate,
        roi,
      },
    },
    verification_data: {
      data_source: 'BallDontLie API',
      calculation_method: 'Round robin backtest (daily top-M pool, combos by N) using American→Decimal odds conversion',
      verification_status: 'computed',
      timestamp: new Date().toISOString(),
    },
  };
}

// ============================================================================
// INITIALIZATION
// ============================================================================

/**
 * Initialize the betting data service
 */
export async function initializeBettingData(): Promise<{ success: boolean; message: string }> {
  try {
    await loadHistoricalData();
    
    const availability = await getDataAvailability();
    const totalGames = Object.values(availability).reduce((sum, a) => sum + a.totalGames, 0);
    const totalWithOdds = Object.values(availability).reduce((sum, a) => sum + a.gamesWithOdds, 0);

    return {
      success: true,
      message: `Loaded ${totalGames.toLocaleString()} games (${totalWithOdds.toLocaleString()} with odds) across ${Object.keys(availability).length} sports`,
    };
  } catch (error) {
    return {
      success: false,
      message: `Failed to initialize betting data: ${error}`,
    };
  }
}
