/**
 * Security Configuration and Validation
 * Centralized security settings and API key validation
 */

import { APIKeyValidator } from './auth';

// Security configuration constants
export const SECURITY_CONFIG = {
  // Rate limiting
  RATE_LIMITS: {
    AUTH: { requests: 5, windowMs: 15 * 60 * 1000 }, // 5 auth attempts per 15 minutes
    SIGNUP: { requests: 3, windowMs: 60 * 60 * 1000 }, // 3 signups per hour
    CHAT: { requests: 50, windowMs: 60 * 60 * 1000 }, // 50 chat messages per hour
    STRATEGIES: { requests: 20, windowMs: 60 * 60 * 1000 }, // 20 strategy simulations per hour
    GENERAL: { requests: 100, windowMs: 15 * 60 * 1000 }, // 100 general requests per 15 minutes
  },

  // CORS settings
  CORS: {
    allowedOrigins: process.env.NODE_ENV === 'production'
      ? (process.env.ALLOWED_ORIGINS || 'https://eventheodds.ai,https://eventheoddsai.com,https://www.eventheodds.ai,https://www.eventheoddsai.com').split(',')
      : ['http://localhost:3000', 'http://localhost:3001'],
    allowedMethods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS'],
    allowedHeaders: ['Content-Type', 'Authorization', 'X-Requested-With'],
    maxAge: 86400, // 24 hours
  },

  // JWT settings - CRITICAL SECURITY FIX: No fallback allowed
  JWT: {
    secret: process.env.JWT_SECRET || (() => {
      throw new Error(
        'CRITICAL SECURITY ERROR: JWT_SECRET environment variable is not set. ' +
        'This is a critical security vulnerability. Set JWT_SECRET in your environment variables immediately.'
      );
    })(),
    expiresIn: '7d',
    issuer: 'backtest-ai',
    audience: 'backtest-users',
  },

  // Password security
  PASSWORD: {
    minLength: 8,
    maxLength: 128,
    requireUppercase: true,
    requireLowercase: true,
    requireNumbers: true,
    requireSpecialChars: false,
  },

  // Input validation
  INPUT_VALIDATION: {
    maxStringLength: 10000,
    maxArrayLength: 1000,
    maxObjectDepth: 10,
    allowedFileTypes: ['image/jpeg', 'image/png', 'image/webp'],
    maxFileSize: 5 * 1024 * 1024, // 5MB
  },

  // Security headers
  HEADERS: {
    contentSecurityPolicy: "default-src 'self'; script-src 'self' 'unsafe-inline' 'unsafe-eval'; style-src 'self' 'unsafe-inline';",
    strictTransportSecurity: 'max-age=31536000; includeSubDomains',
    xFrameOptions: 'DENY',
    xContentTypeOptions: 'nosniff',
    referrerPolicy: 'strict-origin-when-cross-origin',
    permissionsPolicy: 'geolocation=(), microphone=(), camera=()',
  },
};

// Environment validation
export function validateEnvironment(): { valid: boolean; errors: string[] } {
  const errors: string[] = [];

  // Required environment variables
  const required = [
    'DATABASE_URL',
    'JWT_SECRET',
    'GROK_API_KEY',
  ];

  for (const envVar of required) {
    if (!process.env[envVar]) {
      errors.push(`Missing required environment variable: ${envVar}`);
    }
  }

  // API Key validation
  if (!APIKeyValidator.validateGrokKey()) {
    errors.push('Invalid or missing GROK_API_KEY');
  }

  // Database URL format validation
  const dbUrl = process.env.DATABASE_URL;
  if (dbUrl && !dbUrl.startsWith('mysql://')) {
    errors.push('DATABASE_URL must be a valid MySQL connection string');
  }

  // JWT secret strength
  const jwtSecret = process.env.JWT_SECRET;
  if (jwtSecret && jwtSecret.length < 32) {
    errors.push('JWT_SECRET should be at least 32 characters long');
  }

  return {
    valid: errors.length === 0,
    errors,
  };
}

// Security utilities
export class SecurityUtils {
  static generateSecureToken(length: number = 32): string {
    const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
    let result = '';
    for (let i = 0; i < length; i++) {
      result += chars.charAt(Math.floor(Math.random() * chars.length));
    }
    return result;
  }

  static hashString(input: string): string {
    // SECURITY FIX: Use proper cryptographic hash instead of weak bit-shift hash
    // This is suitable for non-sensitive data (not passwords - use bcrypt/scrypt for passwords)
    const crypto = require('crypto');
    return crypto.createHash('sha256').update(input).digest('hex');
  }

  static isValidEmail(email: string): boolean {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email) && email.length <= 255;
  }

  static isValidPassword(password: string): { valid: boolean; errors: string[] } {
    const errors: string[] = [];

    if (password.length < SECURITY_CONFIG.PASSWORD.minLength) {
      errors.push(`Password must be at least ${SECURITY_CONFIG.PASSWORD.minLength} characters long`);
    }

    if (password.length > SECURITY_CONFIG.PASSWORD.maxLength) {
      errors.push(`Password must be no more than ${SECURITY_CONFIG.PASSWORD.maxLength} characters long`);
    }

    if (SECURITY_CONFIG.PASSWORD.requireUppercase && !/[A-Z]/.test(password)) {
      errors.push('Password must contain at least one uppercase letter');
    }

    if (SECURITY_CONFIG.PASSWORD.requireLowercase && !/[a-z]/.test(password)) {
      errors.push('Password must contain at least one lowercase letter');
    }

    if (SECURITY_CONFIG.PASSWORD.requireNumbers && !/\d/.test(password)) {
      errors.push('Password must contain at least one number');
    }

    if (SECURITY_CONFIG.PASSWORD.requireSpecialChars && !/[!@#$%^&*()_+\-=\[\]{};':"\\|,.<>\/?]/.test(password)) {
      errors.push('Password must contain at least one special character');
    }

    return {
      valid: errors.length === 0,
      errors,
    };
  }

  static sanitizeFilename(filename: string): string {
    // Remove path traversal attempts and dangerous characters
    return filename
      .replace(/\.\./g, '') // Remove ..
      .replace(/[<>:"|?*]/g, '') // Remove dangerous chars
      .replace(/\s+/g, '_') // Replace spaces with underscores
      .substring(0, 255); // Limit length
  }

  static isValidUrl(url: string): boolean {
    try {
      new URL(url);
      return url.startsWith('http://') || url.startsWith('https://');
    } catch {
      return false;
    }
  }
}

// Export validation result
export const environmentValidation = validateEnvironment();

if (!environmentValidation.valid) {
  console.error('🚨 Security configuration errors:');
  environmentValidation.errors.forEach(error => console.error(`  ❌ ${error}`));
  console.error('Please fix these issues before deploying to production.');
}
