// Optimized vector configuration based on 2025 advancements
export interface VectorConfig {
  // Algorithm optimizations
  indexType: 'HNSW' | 'IVF' | 'SNG'; // Sparse Neighborhood Graph for 2025
  dimension: number;
  metric: 'cosine' | 'euclidean' | 'dot';

  // Performance optimizations
  efConstruction: number; // HNSW build parameter - higher = better quality, slower build
  efSearch: number; // HNSW search parameter - higher = better recall, slower search
  nprobe: number; // IVF search parameter

  // Memory optimizations
  maxConnections: number; // HNSW connections per layer
  useQuantization: boolean; // Reduce memory footprint
  quantizationBits: number; // 8, 4, or 2 bits

  // Real-time optimizations
  enableIncrementalUpdates: boolean;
  batchSize: number;
  updateFrequency: number; // milliseconds

  // Hybrid search
  enableMetadataFiltering: boolean;
  enableKeywordBoost: boolean;

  // Hardware optimizations
  useGPU: boolean;
  numThreads: number;
  prefetchSize: number; // Memory prefetching
}

export const OPTIMIZED_CONFIGS: Record<string, VectorConfig> = {
  // Ultra-fast configuration for user queries
  realtime_chat: {
    indexType: 'HNSW',
    dimension: 384, // Common embedding size
    metric: 'cosine',
    efConstruction: 200, // Balanced build speed vs quality
    efSearch: 64, // Fast search with good recall
    nprobe: 10,
    maxConnections: 16,
    useQuantization: true,
    quantizationBits: 8, // 4x memory reduction
    enableIncrementalUpdates: true,
    batchSize: 100,
    updateFrequency: 5000, // Update every 5 seconds
    enableMetadataFiltering: true,
    enableKeywordBoost: true,
    useGPU: false, // Keep on CPU for real-time
    numThreads: 4,
    prefetchSize: 1024,
  },

  // High-quality configuration for document search
  document_search: {
    indexType: 'SNG', // 2025 Sparse Neighborhood Graph
    dimension: 768,
    metric: 'cosine',
    efConstruction: 400, // Higher quality for documents
    efSearch: 128, // Better recall for complex queries
    nprobe: 20,
    maxConnections: 32,
    useQuantization: false, // Full precision for documents
    quantizationBits: 8,
    enableIncrementalUpdates: false, // Batch updates for documents
    batchSize: 1000,
    updateFrequency: 300000, // Update every 5 minutes
    enableMetadataFiltering: true,
    enableKeywordBoost: true,
    useGPU: false,
    numThreads: 8,
    prefetchSize: 2048,
  },

  // Memory-efficient configuration for user profiles
  user_memory: {
    indexType: 'IVF', // Inverted File for memory efficiency
    dimension: 384,
    metric: 'cosine',
    efConstruction: 100,
    efSearch: 32,
    nprobe: 5, // Fewer probes for speed
    maxConnections: 8,
    useQuantization: true,
    quantizationBits: 4, // 8x memory reduction
    enableIncrementalUpdates: true,
    batchSize: 50,
    updateFrequency: 1000, // Near real-time updates
    enableMetadataFiltering: true,
    enableKeywordBoost: false,
    useGPU: false,
    numThreads: 2,
    prefetchSize: 512,
  },

  // Balanced configuration for strategy search
  strategy_search: {
    indexType: 'HNSW',
    dimension: 512,
    metric: 'dot', // Better for semantic similarity
    efConstruction: 300,
    efSearch: 96,
    nprobe: 15,
    maxConnections: 24,
    useQuantization: true,
    quantizationBits: 8,
    enableIncrementalUpdates: true,
    batchSize: 200,
    updateFrequency: 10000, // Update every 10 seconds
    enableMetadataFiltering: true,
    enableKeywordBoost: true,
    useGPU: false,
    numThreads: 6,
    prefetchSize: 1536,
  },
};

export function getOptimizedConfig(useCase: keyof typeof OPTIMIZED_CONFIGS): VectorConfig {
  return OPTIMIZED_CONFIGS[useCase] || OPTIMIZED_CONFIGS.realtime_chat;
}

// Performance monitoring
export interface PerformanceMetrics {
  indexBuildTime: number;
  avgQueryTime: number;
  memoryUsage: number;
  recallRate: number;
  throughput: number; // queries per second
}

export class VectorPerformanceMonitor {
  private static instance: VectorPerformanceMonitor;
  private metrics: Map<string, PerformanceMetrics> = new Map();

  private constructor() {}

  static getInstance(): VectorPerformanceMonitor {
    if (!VectorPerformanceMonitor.instance) {
      VectorPerformanceMonitor.instance = new VectorPerformanceMonitor();
    }
    return VectorPerformanceMonitor.instance;
  }

  recordMetrics(useCase: string, metrics: Partial<PerformanceMetrics>): void {
    const existing = this.metrics.get(useCase) || {
      indexBuildTime: 0,
      avgQueryTime: 0,
      memoryUsage: 0,
      recallRate: 0,
      throughput: 0,
    };

    this.metrics.set(useCase, { ...existing, ...metrics });
  }

  getMetrics(useCase: string): PerformanceMetrics | undefined {
    return this.metrics.get(useCase);
  }

  getAllMetrics(): Record<string, PerformanceMetrics> {
    return Object.fromEntries(this.metrics);
  }

  // Auto-tune configuration based on performance
  autoTuneConfig(baseConfig: VectorConfig, currentMetrics: PerformanceMetrics): VectorConfig {
    const tunedConfig = { ...baseConfig };

    // If query time is too slow, reduce efSearch
    if (currentMetrics.avgQueryTime > 100) { // > 100ms
      tunedConfig.efSearch = Math.max(32, tunedConfig.efSearch - 16);
    }

    // If memory usage is high, enable quantization
    if (currentMetrics.memoryUsage > 1024 * 1024 * 1024) { // > 1GB
      tunedConfig.useQuantization = true;
      tunedConfig.quantizationBits = Math.min(tunedConfig.quantizationBits || 8, 4);
    }

    // If throughput is low, increase threads
    if (currentMetrics.throughput < 100) { // < 100 qps
      tunedConfig.numThreads = Math.min(16, tunedConfig.numThreads + 2);
    }

    return tunedConfig;
  }
}
