/**
 * BullMQ-based Job Queue Service
 * Handles async backtest processing with Redis
 * 
 * Benefits:
 * - Persistent job storage (survives server restarts)
 * - Automatic retries on failure
 * - Job progress tracking
 * - Horizontal scaling with multiple workers
 */

import { Queue, Worker, Job, QueueEvents } from 'bullmq';

// Redis connection configuration
const REDIS_CONNECTION = {
    host: process.env.REDIS_HOST || 'localhost',
    port: parseInt(process.env.REDIS_PORT || '6379'),
    password: process.env.REDIS_PASSWORD || undefined,
};

// Job types
export interface BacktestJobData {
    id: string;
    userId: string;
    market: string;
    strategyName: string;
    strategyCode?: string;
    parameters: Record<string, any>;
    season?: string;
    createdAt: string;
}

export interface BacktestJobResult {
    success: boolean;
    results?: any;
    error?: string;
    executionTime?: number;
}

// Singleton queue instance
let backtestQueue: Queue<BacktestJobData, BacktestJobResult> | null = null;
let queueEvents: QueueEvents | null = null;

/**
 * Get or create the backtest queue
 */
export function getBacktestQueue(): Queue<BacktestJobData, BacktestJobResult> {
    if (!backtestQueue) {
        backtestQueue = new Queue<BacktestJobData, BacktestJobResult>('backtests', {
            connection: REDIS_CONNECTION,
            defaultJobOptions: {
                attempts: 3,                    // Retry up to 3 times
                backoff: {
                    type: 'exponential',
                    delay: 1000,               // Start with 1 second delay
                },
                removeOnComplete: {
                    age: 24 * 60 * 60,         // Keep completed jobs for 24 hours
                    count: 1000,               // Keep last 1000 completed jobs
                },
                removeOnFail: {
                    age: 7 * 24 * 60 * 60,     // Keep failed jobs for 7 days
                },
            },
        });
        console.log('🚀 BullMQ backtest queue initialized');
    }
    return backtestQueue;
}

/**
 * Get queue events for job status tracking
 */
export function getQueueEvents(): QueueEvents {
    if (!queueEvents) {
        queueEvents = new QueueEvents('backtests', {
            connection: REDIS_CONNECTION,
        });
    }
    return queueEvents;
}

/**
 * Add a backtest job to the queue
 */
export async function addBacktestJob(
    data: Omit<BacktestJobData, 'id' | 'createdAt'>
): Promise<Job<BacktestJobData, BacktestJobResult>> {
    const queue = getBacktestQueue();

    const jobData: BacktestJobData = {
        ...data,
        id: `bt-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
        createdAt: new Date().toISOString(),
    };

    const job = await queue.add('backtest', jobData, {
        jobId: jobData.id,
    });

    console.log(`📝 Backtest job queued: ${job.id}`);
    return job;
}

/**
 * Get job status by ID
 */
export async function getJobStatus(jobId: string): Promise<{
    status: 'waiting' | 'active' | 'completed' | 'failed' | 'delayed' | 'unknown';
    progress?: number;
    result?: BacktestJobResult;
    error?: string;
}> {
    const queue = getBacktestQueue();
    const job = await queue.getJob(jobId);

    if (!job) {
        return { status: 'unknown' };
    }

    const state = await job.getState();

    return {
        status: state as any,
        progress: job.progress as number | undefined,
        result: job.returnvalue,
        error: job.failedReason,
    };
}

/**
 * Get recent jobs for a user
 */
export async function getUserJobs(
    userId: string,
    limit: number = 10
): Promise<Job<BacktestJobData, BacktestJobResult>[]> {
    const queue = getBacktestQueue();

    // Get completed, failed, and active jobs
    const [completed, failed, active, waiting] = await Promise.all([
        queue.getCompleted(0, limit),
        queue.getFailed(0, limit),
        queue.getActive(0, limit),
        queue.getWaiting(0, limit),
    ]);

    // Combine and filter by userId
    const allJobs = [...completed, ...failed, ...active, ...waiting];
    const userJobs = allJobs.filter(job => job.data.userId === userId);

    // Sort by creation time (newest first)
    userJobs.sort((a, b) =>
        new Date(b.data.createdAt).getTime() - new Date(a.data.createdAt).getTime()
    );

    return userJobs.slice(0, limit);
}

/**
 * Create a worker to process backtest jobs
 * Call this in a separate worker process for production
 */
export function createBacktestWorker(
    processFunction: (job: Job<BacktestJobData>) => Promise<BacktestJobResult>
): Worker<BacktestJobData, BacktestJobResult> {
    const worker = new Worker<BacktestJobData, BacktestJobResult>(
        'backtests',
        async (job) => {
            const startTime = Date.now();
            console.log(`🔄 Processing backtest job: ${job.id}`);

            try {
                // Update progress
                await job.updateProgress(10);

                // Process the backtest
                const result = await processFunction(job);

                await job.updateProgress(100);

                const executionTime = Date.now() - startTime;
                console.log(`✅ Backtest job completed: ${job.id} (${executionTime}ms)`);

                return {
                    ...result,
                    executionTime,
                };
            } catch (error) {
                console.error(`❌ Backtest job failed: ${job.id}`, error);
                throw error;
            }
        },
        {
            connection: REDIS_CONNECTION,
            concurrency: 3,  // Process up to 3 jobs concurrently
        }
    );

    // Worker event handlers
    worker.on('completed', (job, result) => {
        console.log(`✅ Job ${job.id} completed successfully`);
    });

    worker.on('failed', (job, error) => {
        console.error(`❌ Job ${job?.id} failed:`, error.message);
    });

    worker.on('error', (error) => {
        console.error('🔴 Worker error:', error);
    });

    console.log('🚀 BullMQ backtest worker started');
    return worker;
}

/**
 * Wait for a job to complete (with timeout)
 */
export async function waitForJob(
    jobId: string,
    timeoutMs: number = 60000
): Promise<BacktestJobResult | null> {
    const queue = getBacktestQueue();
    const events = getQueueEvents();

    return new Promise(async (resolve, reject) => {
        const timeout = setTimeout(() => {
            resolve(null);  // Return null on timeout (job still processing)
        }, timeoutMs);

        const job = await queue.getJob(jobId);
        if (!job) {
            clearTimeout(timeout);
            resolve(null);
            return;
        }

        // Check if already completed
        const state = await job.getState();
        if (state === 'completed') {
            clearTimeout(timeout);
            resolve(job.returnvalue);
            return;
        }
        if (state === 'failed') {
            clearTimeout(timeout);
            reject(new Error(job.failedReason || 'Job failed'));
            return;
        }

        // Wait for completion
        try {
            const result = await job.waitUntilFinished(events, timeoutMs);
            clearTimeout(timeout);
            resolve(result);
        } catch (error) {
            clearTimeout(timeout);
            reject(error);
        }
    });
}

/**
 * Clean up old jobs
 */
export async function cleanupOldJobs(olderThanDays: number = 7): Promise<void> {
    const queue = getBacktestQueue();

    const grace = olderThanDays * 24 * 60 * 60 * 1000;

    await queue.clean(grace, 1000, 'completed');
    await queue.clean(grace, 1000, 'failed');

    console.log(`🧹 Cleaned up jobs older than ${olderThanDays} days`);
}

/**
 * Get queue statistics
 */
export async function getQueueStats(): Promise<{
    waiting: number;
    active: number;
    completed: number;
    failed: number;
    delayed: number;
}> {
    const queue = getBacktestQueue();

    const [waiting, active, completed, failed, delayed] = await Promise.all([
        queue.getWaitingCount(),
        queue.getActiveCount(),
        queue.getCompletedCount(),
        queue.getFailedCount(),
        queue.getDelayedCount(),
    ]);

    return { waiting, active, completed, failed, delayed };
}
