/**
 * Performance Monitoring Middleware
 * Tracks response times, caches responses, and optimizes API performance
 */

import { NextRequest, NextResponse } from 'next/server';
import { performanceMonitor, logger } from './error-handler';
import { cache, CACHE_KEYS, CACHE_TAGS, CacheHelpers } from './cache';
import { DatabaseService, prisma } from './database';

function safeParseJson<T>(value: any, fallback: T): T {
  if (!value || typeof value !== 'string') return fallback;
  try {
    return JSON.parse(value) as T;
  } catch {
    return fallback;
  }
}

export class PerformanceMiddleware {
  static async trackPerformance(
    request: NextRequest,
    handler: () => Promise<NextResponse>
  ): Promise<NextResponse> {
    const startTime = Date.now();
    const url = new URL(request.url);
    const method = request.method;
    const pathname = url.pathname;

    try {
      // Execute the handler
      const response = await handler();

      // Track performance metrics
      const duration = Date.now() - startTime;
      performanceMonitor.recordMetric(`${method} ${pathname}`, duration);

      // Log slow requests
      if (duration > 1000) { // Over 1 second
        logger.warn('Slow API request', {
          method,
          pathname,
          duration,
          statusCode: response.status,
        });
      }

      // Add performance headers
      response.headers.set('X-Response-Time', `${duration}ms`);
      response.headers.set('X-Server-Timing', `total;dur=${duration}`);

      return response;

    } catch (error) {
      // Track error performance
      const duration = Date.now() - startTime;
      performanceMonitor.recordMetric(`${method} ${pathname}`, duration, true);

      logger.error('API request failed', {
        method,
        pathname,
        duration,
        error: error.message,
      });

      throw error;
    }
  }

  static async cacheAPIResponse<T>(
    cacheKey: string,
    request: NextRequest,
    handler: () => Promise<NextResponse>,
    options: {
      ttl?: number;
      tags?: string[];
      skipCache?: boolean;
    } = {}
  ): Promise<NextResponse> {
    const { ttl = 5 * 60 * 1000, tags = [], skipCache = false } = options;

    // Skip caching for non-GET requests or when explicitly disabled
    if (request.method !== 'GET' || skipCache) {
      return this.trackPerformance(request, handler);
    }

    try {
      // Try to get cached response
      if (!skipCache) {
        const cachedResponse = await cache.get<string>(cacheKey);
        if (cachedResponse) {
          logger.debug('API cache hit', { cacheKey });

          // Return cached response
          const response = NextResponse.json(JSON.parse(cachedResponse));
          response.headers.set('X-Cache-Status', 'HIT');
          return response;
        }
      }

      // Execute handler and cache result
      const response = await handler();

      // Only cache successful GET responses
      if (response.status === 200 && request.method === 'GET') {
        try {
          // Clone the response to read its body
          const clonedResponse = response.clone();
          const responseData = await clonedResponse.json();

          // Cache the response data
          await cache.set(cacheKey, JSON.stringify(responseData), {
            ttl,
            tags,
          });

          logger.debug('API response cached', { cacheKey, ttl });
          response.headers.set('X-Cache-Status', 'MISS');
        } catch (cacheError) {
          logger.warn('Failed to cache API response', { cacheKey, error: cacheError.message });
          response.headers.set('X-Cache-Status', 'ERROR');
        }
      }

      return response;

    } catch (error) {
      logger.error('Cache middleware error', { cacheKey, error: error.message });
      throw error;
    }
  }
}

// Specific caching strategies for different API endpoints
export class APICacheStrategies {
  static async cacheUserProfile(request: NextRequest, userId: string): Promise<NextResponse | null> {
    const cacheKey = CACHE_KEYS.userProfile(userId);

    return CacheHelpers.cached(
      cacheKey,
      async () => {
        const profile = await DatabaseService.findUserById(userId);
        if (!profile) return null;

        // Include related data
        const stats = await DatabaseService.getUserStats(userId);

        return {
          profile,
          stats,
        };
      },
      {
        ttl: 10 * 60 * 1000, // 10 minutes
        tags: [CACHE_TAGS.user(userId)],
      }
    );
  }

  static async cacheUserStrategies(request: NextRequest, userId: string): Promise<NextResponse | null> {
    const url = new URL(request.url);
    const page = parseInt(url.searchParams.get('page') || '1');
    const limit = parseInt(url.searchParams.get('limit') || '20');

    const cacheKey = `user_strategies:${userId}:${page}:${limit}`;

    return CacheHelpers.cached(
      cacheKey,
      async () => {
        const strategies = await DatabaseService.findStrategiesByUser(userId, limit);
        return strategies;
      },
      {
        ttl: 5 * 60 * 1000, // 5 minutes
        tags: [CACHE_TAGS.user(userId)],
      }
    );
  }

  static async cachePublicStrategies(request: NextRequest): Promise<NextResponse | null> {
    const url = new URL(request.url);
    const domain = url.searchParams.get('domain');
    const page = parseInt(url.searchParams.get('page') || '1');

    const cacheKey = CACHE_KEYS.publicStrategies(domain || undefined, page);

    return CacheHelpers.cached(
      cacheKey,
      async () => {
        const strategies = await DatabaseService.findPublicStrategies(20);
        return strategies;
      },
      {
        ttl: 10 * 60 * 1000, // 10 minutes
        tags: [CACHE_TAGS.strategies],
      }
    );
  }

  static async cacheDatasetOverview(request: NextRequest): Promise<NextResponse | null> {
    const cacheKey = CACHE_KEYS.datasetOverview();

    return CacheHelpers.cached(
      cacheKey,
      async () => {
        // This would need to import the strategy engine
        // For now, return a placeholder
        return {
          datasets: [],
          lastUpdated: new Date().toISOString(),
        };
      },
      {
        ttl: 30 * 60 * 1000, // 30 minutes - dataset overview doesn't change often
        tags: ['datasets'],
      }
    );
  }

  static async cacheSystemStats(request: NextRequest): Promise<NextResponse | null> {
    const cacheKey = CACHE_KEYS.systemStats();

    return CacheHelpers.cached(
      cacheKey,
      async () => {
        const stats = await DatabaseService.getPlatformStats();
        return stats;
      },
      {
        ttl: 60 * 60 * 1000, // 1 hour - stats update hourly
        tags: [CACHE_TAGS.system],
      }
    );
  }
}

// Cache invalidation helpers
export class CacheInvalidation {
  static async invalidateUserCache(userId: string): Promise<void> {
    await CacheHelpers.invalidateUserData(userId);

    // Also invalidate related caches
    await cache.invalidateTag(`user_strategies:${userId}`);
    await cache.invalidateTag(CACHE_TAGS.analytics);
  }

  static async invalidateStrategyCache(strategyId: string, userId?: string): Promise<void> {
    await CacheHelpers.invalidateStrategyData(strategyId);

    if (userId) {
      await this.invalidateUserCache(userId);
    }
  }

  static async invalidateAnalyticsCache(): Promise<void> {
    await CacheHelpers.invalidateAnalytics();
  }

  static async invalidateSystemCache(): Promise<void> {
    await cache.invalidateTag(CACHE_TAGS.system);
    await cache.clear('stats:*');
  }
}

// Database query optimization helpers
export class QueryOptimization {
  static async getUserWithOptimizedQuery(userId: string) {
    // Use cached version first
    const cachedProfile = await APICacheStrategies.cacheUserProfile({} as NextRequest, userId);
    if (cachedProfile) {
      return cachedProfile;
    }

    // Fallback to database with optimized query
    return DatabaseService.findUserById(userId);
  }

  static async getStrategiesWithPagination(userId: string, options: {
    page?: number;
    limit?: number;
    sortBy?: 'createdAt' | 'updatedAt' | 'likes' | 'views';
    sortOrder?: 'asc' | 'desc';
  } = {}) {
    const { page = 1, limit = 20, sortBy = 'updatedAt', sortOrder = 'desc' } = options;

    // Try cache first
    const cacheKey = `user_strategies:${userId}:${page}:${limit}:${sortBy}:${sortOrder}`;
    const cached = await cache.get(cacheKey);
    if (cached) {
      return cached;
    }

    // Database query with optimized sorting
    const strategies = await DatabaseService.findStrategiesByUser(userId, limit);

    // Cache the result
    await cache.set(cacheKey, strategies, {
      ttl: 5 * 60 * 1000, // 5 minutes
      tags: [CACHE_TAGS.user(userId)],
    });

    return strategies;
  }

  static async batchUserStats(userIds: string[]): Promise<Map<string, any>> {
    const results = new Map();

    // Batch database queries to reduce round trips
    const userProfiles = await Promise.all(
      userIds.map(id => DatabaseService.findUserById(id))
    );

    const userStats = await Promise.all(
      userIds.map(id => DatabaseService.getUserStats(id))
    );

    // Combine results
    userIds.forEach((userId, index) => {
      results.set(userId, {
        profile: userProfiles[index],
        stats: userStats[index],
      });
    });

    return results;
  }
}

// Performance monitoring dashboard data
export class PerformanceDashboard {
  static async getPerformanceMetrics(timeRange: '1h' | '24h' | '7d' = '24h'): Promise<{
    apiResponseTimes: any[];
    cacheStats: any;
    databaseStats: any;
    errorRates: any;
    topSlowEndpoints: any[];
  }> {
    const cacheStats = await cache.getStats();

    const windowMs =
      timeRange === '1h' ? 60 * 60 * 1000 :
      timeRange === '7d' ? 7 * 24 * 60 * 60 * 1000 :
      24 * 60 * 60 * 1000;

    const since = new Date(Date.now() - windowMs);

    // Pull persisted performance metrics (recorded by PerformanceMonitor)
    const metricRows = await prisma.systemAnalytics.findMany({
      where: {
        eventType: 'PERFORMANCE_METRIC',
        createdAt: { gte: since },
      },
      select: {
        eventData: true,
        createdAt: true,
      },
      orderBy: { createdAt: 'desc' },
      take: 5000,
    });

    const aggregates = new Map<string, { count: number; totalTime: number; errors: number }>();

    for (const row of metricRows) {
      const metric = safeParseJson<any>(row.eventData, null);
      if (!metric || typeof metric !== 'object') continue;
      const operation = String(metric.operation || '');
      const count = Number(metric.count || 0);
      const totalTime = Number(metric.totalTime || 0);
      const errors = Number(metric.errors || 0);
      if (!operation || !Number.isFinite(count) || !Number.isFinite(totalTime) || !Number.isFinite(errors)) continue;

      const existing = aggregates.get(operation) || { count: 0, totalTime: 0, errors: 0 };
      existing.count += Math.max(0, count);
      existing.totalTime += Math.max(0, totalTime);
      existing.errors += Math.max(0, errors);
      aggregates.set(operation, existing);
    }

    const apiResponseTimes = Array.from(aggregates.entries()).map(([operation, data]) => {
      const avgTime = data.count > 0 ? data.totalTime / data.count : 0;
      const errorRate = data.count > 0 ? data.errors / data.count : 0;
      return {
        operation,
        count: data.count,
        avgTime,
        totalTime: data.totalTime,
        errors: data.errors,
        errorRate,
      };
    });

    const topSlowEndpoints = [...apiResponseTimes]
      .sort((a, b) => (b.avgTime || 0) - (a.avgTime || 0))
      .slice(0, 10);

    const totalRequests = apiResponseTimes.reduce((sum, m) => sum + (m.count || 0), 0);
    const totalErrors = apiResponseTimes.reduce((sum, m) => sum + (m.errors || 0), 0);

    // Cross-check persisted error events
    const errorEvents = await prisma.systemAnalytics.count({
      where: {
        eventType: { in: ['ERROR_OCCURRED', 'UNHANDLED_ERROR'] },
        createdAt: { gte: since },
      },
    });

    return {
      apiResponseTimes,
      cacheStats,
      databaseStats: {
        // Prisma doesn't expose pool stats in SQLite; provide useful counts instead.
        queryCount: totalRequests,
        trackedOperations: aggregates.size,
        lastMetricAt: metricRows[0]?.createdAt?.toISOString?.() || null,
      },
      errorRates: {
        totalErrors,
        errorRate: totalRequests > 0 ? totalErrors / totalRequests : 0,
        errorEvents,
      },
      topSlowEndpoints,
    };
  }

  static async getSystemHealth(): Promise<{
    status: 'healthy' | 'warning' | 'critical';
    checks: Record<string, { status: 'pass' | 'fail'; latency?: number; message?: string }>;
  }> {
    const checks: Record<string, { status: 'pass' | 'fail'; latency?: number; message?: string }> = {};

    // Database health
    const dbStart = Date.now();
    try {
      const dbHealthy = await DatabaseService.healthCheck();
      checks.database = {
        status: dbHealthy ? 'pass' : 'fail',
        latency: Date.now() - dbStart,
      };
    } catch (error) {
      checks.database = {
        status: 'fail',
        message: error.message,
      };
    }

    // Cache health
    const cacheStart = Date.now();
    try {
      const cacheStats = await cache.getStats();
      checks.cache = {
        status: 'pass',
        latency: Date.now() - cacheStart,
      };
    } catch (error) {
      checks.cache = {
        status: 'fail',
        message: error.message,
      };
    }

    // Determine overall status
    const failedChecks = Object.values(checks).filter(check => check.status === 'fail').length;
    const status = failedChecks === 0 ? 'healthy' : failedChecks === 1 ? 'warning' : 'critical';

    return { status, checks };
  }
}
