import type { UserProfile, ConversationMemory, StrategyMemory, MemoryContext } from './types';

export interface VectorEmbedding {
  id: string;
  vector: number[];
  metadata: Record<string, any>;
  timestamp: Date;
}

export interface SimilarityResult {
  id: string;
  score: number;
  metadata: Record<string, any>;
  timestamp?: Date;
}

export class VectorMemoryService {
  private static instance: VectorMemoryService;
  private vectors: Map<string, VectorEmbedding[]> = new Map();
  private indexCache: Map<string, any> = new Map(); // FAISS-like index cache

  private constructor() {}

  static getInstance(): VectorMemoryService {
    if (!VectorMemoryService.instance) {
      VectorMemoryService.instance = new VectorMemoryService();
    }
    return VectorMemoryService.instance;
  }

  // User Profile Vector Storage
  async storeUserProfile(profile: UserProfile): Promise<void> {
    const profileText = this.profileToText(profile);
    const embedding = await this.generateEmbedding(profileText);

    const vector: VectorEmbedding = {
      id: `profile-${profile.id}`,
      vector: embedding,
      metadata: {
        type: 'user_profile',
        userId: profile.id,
        riskTolerance: profile.riskTolerance,
        experienceLevel: profile.experienceLevel,
        preferredMarkets: profile.preferredMarkets,
      },
      timestamp: new Date(),
    };

    this.storeVector('user_profiles', vector);
    this.updateIndex('user_profiles');
  }

  async findSimilarUsers(queryProfile: Partial<UserProfile>, limit: number = 5): Promise<SimilarityResult[]> {
    const queryText = this.profileToText(queryProfile as UserProfile);
    const queryEmbedding = await this.generateEmbedding(queryText);

    return this.searchSimilar('user_profiles', queryEmbedding, limit);
  }

  // Conversation Memory Vectors
  async storeConversationMemory(memory: ConversationMemory): Promise<void> {
    const embedding = await this.generateEmbedding(memory.content);

    const vector: VectorEmbedding = {
      id: `conversation-${memory.id}`,
      vector: embedding,
      metadata: {
        type: 'conversation',
        userId: memory.userId,
        sessionId: memory.sessionId,
        role: memory.role,
        hasStrategy: !!memory.strategyGenerated,
        domain: memory.strategyGenerated?.name.toLowerCase().includes('crypto') ? 'crypto' :
                memory.strategyGenerated?.name.toLowerCase().includes('stock') ? 'stocks' :
                memory.strategyGenerated?.name.toLowerCase().includes('forex') ? 'forex' : 'sports',
      },
      timestamp: memory.timestamp,
    };

    this.storeVector('conversations', vector);
    this.updateIndex('conversations');
  }

  async findRelevantConversations(userId: string, query: string, limit: number = 5): Promise<SimilarityResult[]> {
    const queryEmbedding = await this.generateEmbedding(query);

    // Filter by user and search
    const userConversations = this.vectors.get('conversations')?.filter(v =>
      v.metadata.userId === userId
    ) || [];

    const results = await this.searchInSubset(userConversations, queryEmbedding, limit);
    // Add timestamps to results
    return results.map(result => ({
      ...result,
      timestamp: userConversations.find(v => v.id === result.id)?.timestamp,
    }));
  }

  // Strategy Memory Vectors
  async storeStrategyMemory(strategy: StrategyMemory): Promise<void> {
    const strategyText = `${strategy.name}: ${strategy.code} - Win Rate: ${strategy.performance.winRate}`;
    const embedding = await this.generateEmbedding(strategyText);

    const vector: VectorEmbedding = {
      id: `strategy-${strategy.strategyId}`,
      vector: embedding,
      metadata: {
        type: 'strategy',
        userId: strategy.context.userPrompt ? 'user_generated' : 'system',
        domain: strategy.domain,
        winRate: strategy.performance.winRate,
        totalTrades: strategy.performance.totalTrades,
        isSuccessful: strategy.performance.winRate > 0.6,
      },
      timestamp: new Date(),
    };

    this.storeVector('strategies', vector);
    this.updateIndex('strategies');
  }

  async findSimilarStrategies(queryStrategy: Partial<StrategyMemory>, limit: number = 5): Promise<SimilarityResult[]> {
    const queryText = `Strategy for ${queryStrategy.domain}: ${queryStrategy.context?.userPrompt || 'trading strategy'}`;
    const queryEmbedding = await this.generateEmbedding(queryText);

    return this.searchSimilar('strategies', queryEmbedding, limit);
  }

  // Fast Memory Context Building
  async buildFastMemoryContext(userId: string, currentQuery: string): Promise<MemoryContext> {
    const [userProfile, relevantConversations, similarStrategies] = await Promise.all([
      this.getUserProfileFast(userId),
      this.findRelevantConversations(userId, currentQuery, 3),
      this.findSuccessfulStrategies(userId, 3),
    ]);

    // Extract preferred domain from conversation patterns
    const domainCounts: Record<string, number> = {};
    relevantConversations.forEach(conv => {
      const domain = conv.metadata.domain;
      domainCounts[domain] = (domainCounts[domain] || 0) + 1;
    });

    const preferredDomain = Object.entries(domainCounts)
      .sort(([,a], [,b]) => b - a)[0]?.[0] || 'crypto';

    return {
      userProfile: userProfile || {
        riskTolerance: 'medium',
        experienceLevel: 'intermediate',
        preferredMarkets: [preferredDomain as 'sports' | 'crypto' | 'stocks' | 'forex'],
      },
      recentConversations: relevantConversations.map(c => ({
        id: c.id,
        userId: c.metadata.userId,
        sessionId: c.metadata.sessionId,
        timestamp: c.timestamp || new Date(),
        role: c.metadata.role,
        content: 'Context from previous conversation',
      })),
      relevantStrategies: similarStrategies.map(s => ({
        strategyId: s.id,
        name: `Similar ${s.metadata.domain} strategy`,
        domain: s.metadata.domain,
        code: '',
        performance: {
          winRate: s.metadata.winRate,
          totalTrades: s.metadata.totalTrades,
          netProfit: 0,
          createdAt: s.timestamp || new Date(),
        },
        context: { userPrompt: '', marketConditions: '', successFactors: [] },
        usageCount: 1,
        lastUsed: new Date(),
      })),
      conversationSummaries: [],
      marketInsights: [],
      personalization: {
        preferredDomain,
        riskAdjusted: true,
        experienceLevel: 'intermediate',
      },
    };
  }

  // Optimized Storage Methods
  private storeVector(collection: string, vector: VectorEmbedding): void {
    if (!this.vectors.has(collection)) {
      this.vectors.set(collection, []);
    }
    this.vectors.get(collection)!.push(vector);
  }

  private updateIndex(collection: string): void {
    // In a real implementation, this would rebuild FAISS/optimized index
    // For now, we'll use simple in-memory indexing
    const vectors = this.vectors.get(collection) || [];
    this.indexCache.set(collection, {
      vectors,
      lastUpdated: new Date(),
    });
  }

  private async searchSimilar(collection: string, queryVector: number[], limit: number): Promise<SimilarityResult[]> {
    const index = this.indexCache.get(collection);
    if (!index) return [];

    // Fast cosine similarity search
    const results = index.vectors
      .map((vector: VectorEmbedding) => ({
        id: vector.id,
        score: this.cosineSimilarity(queryVector, vector.vector),
        metadata: vector.metadata,
      }))
      .sort((a: SimilarityResult, b: SimilarityResult) => b.score - a.score)
      .slice(0, limit);

    return results;
  }

  private async searchInSubset(vectors: VectorEmbedding[], queryVector: number[], limit: number): Promise<SimilarityResult[]> {
    return vectors
      .map((vector: VectorEmbedding) => ({
        id: vector.id,
        score: this.cosineSimilarity(queryVector, vector.vector),
        metadata: vector.metadata,
      }))
      .sort((a: SimilarityResult, b: SimilarityResult) => b.score - a.score)
      .slice(0, limit);
  }

  private cosineSimilarity(a: number[], b: number[]): number {
    if (a.length !== b.length) return 0;

    let dotProduct = 0;
    let normA = 0;
    let normB = 0;

    for (let i = 0; i < a.length; i++) {
      dotProduct += a[i] * b[i];
      normA += a[i] * a[i];
      normB += b[i] * b[i];
    }

    if (normA === 0 || normB === 0) return 0;

    return dotProduct / (Math.sqrt(normA) * Math.sqrt(normB));
  }

  // Mock embedding generation (in real system, use actual embedding model)
  private async generateEmbedding(text: string): Promise<number[]> {
    // Simple hash-based mock embedding for demonstration
    // In production, this would call an actual embedding API
    const hash = this.simpleHash(text);
    const embedding: number[] = [];

    // Generate 384-dimensional embedding (common for many models)
    for (let i = 0; i < 384; i++) {
      embedding.push((hash * (i + 1)) % 1000 / 500 - 1); // Normalize to [-1, 1]
    }

    return embedding;
  }

  private simpleHash(text: string): number {
    let hash = 0;
    for (let i = 0; i < text.length; i++) {
      const char = text.charCodeAt(i);
      hash = ((hash << 5) - hash) + char;
      hash = hash & hash; // Convert to 32-bit integer
    }
    return Math.abs(hash);
  }

  private profileToText(profile: Partial<UserProfile>): string {
    return `User profile: risk tolerance ${profile.riskTolerance}, experience ${profile.experienceLevel}, markets ${profile.preferredMarkets?.join(' ')}, goals ${profile.learningGoals?.join(' ')}`;
  }

  private async getUserProfileFast(userId: string): Promise<Partial<UserProfile> | null> {
    // Fast lookup from vector metadata
    const profileVectors = this.vectors.get('user_profiles') || [];
    const userProfile = profileVectors.find(v => v.metadata.userId === userId);

    if (!userProfile) return null;

    return {
      riskTolerance: userProfile.metadata.riskTolerance,
      experienceLevel: userProfile.metadata.experienceLevel,
      preferredMarkets: userProfile.metadata.preferredMarkets,
    };
  }

  private async findSuccessfulStrategies(userId: string, limit: number): Promise<SimilarityResult[]> {
    const strategyVectors = this.vectors.get('strategies') || [];
    return strategyVectors
      .filter(v => v.metadata.isSuccessful)
      .sort((a, b) => b.metadata.winRate - a.metadata.winRate)
      .slice(0, limit)
      .map(v => ({
        id: v.id,
        score: v.metadata.winRate,
        metadata: v.metadata,
      }));
  }
}

