import type {
  UserProfile,
  ConversationMemory,
  ConversationSummary,
  StrategyMemory,
  UserFeedback,
  MemoryContext,
} from './types';
import { VectorMemoryService } from './VectorMemoryService';
import { GlobalCacheManager, CacheKey } from './MemoryCache';
import DatabaseService, { prisma } from '@/lib/database';

function safeParseJsonArray(value: any): string[] | undefined {
  if (!value) return undefined;
  if (Array.isArray(value)) return value.map(String);
  if (typeof value !== 'string') return undefined;
  try {
    const parsed = JSON.parse(value);
    return Array.isArray(parsed) ? parsed.map(String) : undefined;
  } catch {
    return undefined;
  }
}

function normalizeRiskTolerance(value: any): 'low' | 'medium' | 'high' {
  const v = String(value || '').toLowerCase();
  if (v.includes('low')) return 'low';
  if (v.includes('high')) return 'high';
  return 'medium';
}

function normalizeExperienceLevel(value: any): 'beginner' | 'intermediate' | 'advanced' {
  const v = String(value || '').toLowerCase();
  if (v.includes('beginner')) return 'beginner';
  if (v.includes('advanced') || v.includes('expert')) return 'advanced';
  return 'intermediate';
}

export class MemoryService {
  private static instance: MemoryService;
  private vectorMemory: VectorMemoryService;
  private cacheManager: GlobalCacheManager;

  private constructor() {
    this.vectorMemory = VectorMemoryService.getInstance();
    this.cacheManager = GlobalCacheManager.getInstance();
  }

  static getInstance(): MemoryService {
    if (!MemoryService.instance) {
      MemoryService.instance = new MemoryService();
    }
    return MemoryService.instance;
  }

  // User Profile Management
  async getUserProfile(userId: string): Promise<UserProfile | null> {
    // Check cache first for ultra-fast access
    const cacheKey = CacheKey.userProfile(userId);
    const cache = this.cacheManager.getCache<UserProfile>('user_profiles');

    const cachedProfile = cache.get(cacheKey);
    if (cachedProfile) {
      return cachedProfile;
    }

    // Pull user profile from the real database
    const dbUser = await DatabaseService.findUserById(userId);
    if (!dbUser) return null;

    const preferredMarkets = (safeParseJsonArray(dbUser.preferredMarkets) || ['sports']) as any;
    const favoriteStrategies = safeParseJsonArray(dbUser.favoriteStrategies) || [];

    const profile: UserProfile = {
      id: userId,
      createdAt: dbUser.createdAt,
      lastActive: dbUser.lastLoginAt || dbUser.updatedAt,
      preferredMarkets,
      riskTolerance: normalizeRiskTolerance(dbUser.riskTolerance),
      experienceLevel: normalizeExperienceLevel(dbUser.experienceLevel),
      favoriteStrategies,
      successfulStrategies: [],
      failedStrategies: [],
      sessionFrequency: 1,
      averageSessionDuration: 15,
      preferredDomains: { sports: 1 },
      timePreferences: {
        activeHours: [],
        timezone: 'UTC',
      },
      learningGoals: [],
      knowledgeGaps: [],
      recommendedStrategies: [],
      feedbackHistory: [],
    };

    // Cache for 10 minutes
    cache.set(cacheKey, profile, 600000);
    return profile;
  }

  async updateUserProfile(userId: string, updates: Partial<UserProfile>): Promise<void> {
    // Persist only supported fields into the DB (keep the rest in vector memory)
    try {
      const preferredMarkets = updates.preferredMarkets
        ? JSON.stringify(updates.preferredMarkets)
        : undefined;
      const favoriteStrategies = updates.favoriteStrategies
        ? JSON.stringify(updates.favoriteStrategies)
        : undefined;

      await DatabaseService.updateUser(userId, {
        riskTolerance: updates.riskTolerance ? updates.riskTolerance.toUpperCase() : undefined,
        experienceLevel: updates.experienceLevel ? updates.experienceLevel.toUpperCase() : undefined,
        preferredMarkets: preferredMarkets ? JSON.parse(preferredMarkets) : undefined, // updateUser will stringify
        favoriteStrategies: favoriteStrategies ? JSON.parse(favoriteStrategies) : undefined,
      } as any);
    } catch (e) {
      console.warn('Failed to persist user profile updates', e);
    }

    // Update vector memory for fast retrieval
    const fullProfile = await this.getUserProfile(userId);
    if (fullProfile) {
      const updatedProfile = { ...fullProfile, ...updates };
      await this.vectorMemory.storeUserProfile(updatedProfile);
    }
  }

  async createUserProfile(userId: string, initialData?: Partial<UserProfile>): Promise<UserProfile> {
    const profile: UserProfile = {
      id: userId,
      createdAt: new Date(),
      lastActive: new Date(),
      preferredMarkets: [],
      riskTolerance: 'medium',
      experienceLevel: 'beginner',
      favoriteStrategies: [],
      successfulStrategies: [],
      failedStrategies: [],
      sessionFrequency: 1,
      averageSessionDuration: 15,
      preferredDomains: {},
      timePreferences: {
        activeHours: [],
        timezone: 'UTC',
      },
      learningGoals: [],
      knowledgeGaps: [],
      recommendedStrategies: [],
      feedbackHistory: [],
      ...initialData,
    };

    // In a real implementation, save to database
    console.log(`Creating profile for user ${userId}`);
    return profile;
  }

  // Conversation Memory
  async addConversationMemory(memory: ConversationMemory): Promise<void> {
    // Store conversation turn
    console.log(`Adding conversation memory for user ${memory.userId}:`, memory.content.substring(0, 50));

    // Store in vector memory for fast retrieval
    await this.vectorMemory.storeConversationMemory(memory);

    // Update user profile with domain preferences
    if (memory.strategyGenerated) {
      const profile = await this.getUserProfile(memory.userId);
      if (profile) {
        const domain = memory.strategyGenerated.name.toLowerCase().includes('crypto') ? 'crypto' :
                      memory.strategyGenerated.name.toLowerCase().includes('stock') ? 'stocks' :
                      memory.strategyGenerated.name.toLowerCase().includes('forex') ? 'forex' : 'sports';

        profile.preferredDomains[domain] = (profile.preferredDomains[domain] || 0) + 1;
        profile.lastActive = new Date();
        await this.updateUserProfile(memory.userId, profile);
      }
    }
  }

  async getRecentConversations(userId: string, limit: number = 10): Promise<ConversationMemory[]> {
    const messages = await prisma.message.findMany({
      where: {
        conversation: {
          userId,
        },
      },
      orderBy: { createdAt: 'desc' },
      take: limit,
      select: {
        id: true,
        role: true,
        content: true,
        createdAt: true,
        conversationId: true,
      },
    });

    return messages.map((m) => ({
      id: m.id,
      userId,
      sessionId: m.conversationId,
      timestamp: m.createdAt,
      role: m.role === 'ASSISTANT' ? 'assistant' : 'user',
      content: m.content,
    }));
  }

  async getConversationBuffer(userId: string, sessionId: string): Promise<ConversationMemory[]> {
    // Get short-term conversation buffer for context
    return this.getRecentConversations(userId, 5);
  }

  // Strategy Memory
  async rememberStrategy(userId: string, strategy: StrategyMemory): Promise<void> {
    const profile = await this.getUserProfile(userId);
    if (profile) {
      if (strategy.performance.winRate > 0.6) {
        profile.successfulStrategies.push(strategy);
      } else {
        profile.failedStrategies.push(strategy);
      }
      await this.updateUserProfile(userId, profile);

      // Store strategy in vector memory for fast similarity search
      await this.vectorMemory.storeStrategyMemory(strategy);
    }
  }

  async getRelevantStrategies(userId: string, domain: string): Promise<StrategyMemory[]> {
    const profile = await this.getUserProfile(userId);
    if (!profile) return [];

    return profile.successfulStrategies.filter(s =>
      s.domain === domain &&
      s.performance.winRate > 0.5
    ).slice(0, 3);
  }

  // Learning & Adaptation
  async extractUserPreferences(text: string): Promise<Partial<UserProfile>> {
    // Simple keyword extraction (in real implementation, use LLM)
    const preferences: Partial<UserProfile> = {};

    const lowerText = text.toLowerCase();

    // Risk tolerance
    if (lowerText.includes('conservative') || lowerText.includes('safe')) {
      preferences.riskTolerance = 'low';
    } else if (lowerText.includes('aggressive') || lowerText.includes('high risk')) {
      preferences.riskTolerance = 'high';
    }

    // Experience level
    if (lowerText.includes('beginner') || lowerText.includes('new to')) {
      preferences.experienceLevel = 'beginner';
    } else if (lowerText.includes('experienced') || lowerText.includes('professional')) {
      preferences.experienceLevel = 'advanced';
    }

    // Market preferences
    const markets: Array<'sports' | 'crypto' | 'stocks' | 'forex'> = [];
    if (lowerText.includes('crypto') || lowerText.includes('bitcoin') || lowerText.includes('ethereum')) {
      markets.push('crypto');
    }
    if (lowerText.includes('stock') || lowerText.includes('equity') || lowerText.includes('market')) {
      markets.push('stocks');
    }
    if (lowerText.includes('forex') || lowerText.includes('currency') || lowerText.includes('eurusd')) {
      markets.push('forex');
    }
    if (lowerText.includes('sports') || lowerText.includes('football') || lowerText.includes('basketball')) {
      markets.push('sports');
    }

    if (markets.length > 0) {
      preferences.preferredMarkets = markets;
    }

    return preferences;
  }

  // Track report downloads and analysis history
  async trackReportDownload(userId: string, reportData: {
    reportType: string;
    market: string;
    totalTrades: number;
    strategyName: string;
    timestamp: string;
    keyMetrics: any;
  }): Promise<void> {
    // Store in vector memory for AI context
    await this.vectorMemory.storeReportMemory(userId, {
      type: 'report_download',
      content: `User downloaded ${reportData.reportType} report for ${reportData.market} strategy "${reportData.strategyName}" with ${reportData.totalTrades} trades`,
      metadata: {
        reportType: reportData.reportType,
        market: reportData.market,
        strategyName: reportData.strategyName,
        totalTrades: reportData.totalTrades,
        timestamp: reportData.timestamp,
        keyMetrics: reportData.keyMetrics
      },
      timestamp: new Date(reportData.timestamp)
    });

    // Update user profile with report activity
    const profile = await this.getUserProfile(userId);
    if (profile) {
      profile.reportsDownloaded = (profile.reportsDownloaded || 0) + 1;
      profile.lastReportDownload = new Date(reportData.timestamp);
      profile.favoriteMarkets = profile.favoriteMarkets || [];
      if (!profile.favoriteMarkets.includes(reportData.market)) {
        profile.favoriteMarkets.push(reportData.market);
      }
      await this.updateUserProfile(userId, profile);
    }
  }

  // Fast Context Building for AI (using vector search)
  async buildMemoryContext(userId: string, currentQuery: string, currentDomain?: string): Promise<MemoryContext> {
    // Try fast vector-based context first, fall back to traditional method
    try {
      return await this.vectorMemory.buildFastMemoryContext(userId, currentQuery);
    } catch (error) {
      console.warn('Vector memory failed, falling back to traditional method:', error);
      return this.buildTraditionalMemoryContext(userId, currentQuery, currentDomain);
    }
  }

  // Traditional memory context (fallback)
  private async buildTraditionalMemoryContext(userId: string, currentQuery: string, currentDomain?: string): Promise<MemoryContext> {
    const profile = await this.getUserProfile(userId) || await this.createUserProfile(userId);
    const recentConversations = await this.getRecentConversations(userId, 5);
    const relevantStrategies = currentDomain ? await this.getRelevantStrategies(userId, currentDomain) : [];

    // Determine preferred domain
    const domainUsage = Object.entries(profile.preferredDomains || {});
    const preferredDomain = domainUsage.length > 0
      ? domainUsage.sort(([,a], [,b]) => b - a)[0][0]
      : currentDomain || 'sports';

    return {
      userProfile: profile,
      recentConversations,
      relevantStrategies,
      conversationSummaries: [], // Would be populated from vector store
      marketInsights: [], // Would be populated from RAG
      personalization: {
        preferredDomain,
        riskAdjusted: profile.riskTolerance !== 'high',
        experienceLevel: profile.experienceLevel,
      },
    };
  }

  // Feedback & Learning
  async recordFeedback(userId: string, feedback: UserFeedback): Promise<void> {
    const profile = await this.getUserProfile(userId);
    if (profile) {
      profile.feedbackHistory.push(feedback);
      await this.updateUserProfile(userId, profile);
    }
  }

  async getPersonalizedRecommendations(userId: string): Promise<string[]> {
    const profile = await this.getUserProfile(userId);
    if (!profile) return [];

    const recommendations: string[] = [];

    // Based on successful strategies
    if (profile.successfulStrategies.length > 0) {
      recommendations.push(`Try variations of your successful ${profile.successfulStrategies[0].domain} strategies`);
    }

    // Based on risk tolerance
    if (profile.riskTolerance === 'low') {
      recommendations.push('Consider conservative strategies with stop-losses');
    }

    // Based on experience level
    if (profile.experienceLevel === 'beginner') {
      recommendations.push('Start with simple momentum or mean-reversion strategies');
    }

    return recommendations;
  }
}

