// High-performance memory cache with TTL and LRU eviction
export interface CacheEntry<T> {
  data: T;
  timestamp: number;
  ttl: number; // Time to live in milliseconds
  accessCount: number;
  lastAccessed: number;
}

export class MemoryCache<T> {
  private cache: Map<string, CacheEntry<T>> = new Map();
  private maxSize: number;
  private cleanupInterval: NodeJS.Timeout;

  constructor(maxSize: number = 1000, cleanupIntervalMs: number = 60000) { // 1 minute cleanup
    this.maxSize = maxSize;
    this.cleanupInterval = setInterval(() => this.cleanup(), cleanupIntervalMs);
  }

  set(key: string, value: T, ttl: number = 300000): void { // 5 minute default TTL
    // Evict if at capacity (LRU)
    if (this.cache.size >= this.maxSize) {
      this.evictLRU();
    }

    this.cache.set(key, {
      data: value,
      timestamp: Date.now(),
      ttl,
      accessCount: 0,
      lastAccessed: Date.now(),
    });
  }

  get(key: string): T | null {
    const entry = this.cache.get(key);

    if (!entry) return null;

    // Check TTL
    if (Date.now() - entry.timestamp > entry.ttl) {
      this.cache.delete(key);
      return null;
    }

    // Update access stats
    entry.accessCount++;
    entry.lastAccessed = Date.now();

    return entry.data;
  }

  has(key: string): boolean {
    return this.get(key) !== null;
  }

  delete(key: string): boolean {
    return this.cache.delete(key);
  }

  clear(): void {
    this.cache.clear();
  }

  size(): number {
    return this.cache.size;
  }

  // Get cache statistics
  getStats(): {
    size: number;
    maxSize: number;
    hitRate: number;
    totalAccesses: number;
    avgTTL: number;
  } {
    const entries = Array.from(this.cache.values());
    const totalAccesses = entries.reduce((sum, entry) => sum + entry.accessCount, 0);
    const totalEntries = entries.length;

    return {
      size: totalEntries,
      maxSize: this.maxSize,
      hitRate: totalEntries > 0 ? totalAccesses / totalEntries : 0,
      totalAccesses,
      avgTTL: totalEntries > 0
        ? entries.reduce((sum, entry) => sum + entry.ttl, 0) / totalEntries
        : 0,
    };
  }

  private evictLRU(): void {
    if (this.cache.size === 0) return;

    // Find least recently used entry
    let lruKey = '';
    let lruTime = Date.now();

    for (const [key, entry] of Array.from(this.cache.entries())) {
      if (entry.lastAccessed < lruTime) {
        lruTime = entry.lastAccessed;
        lruKey = key;
      }
    }

    if (lruKey) {
      this.cache.delete(lruKey);
    }
  }

  private cleanup(): void {
    const now = Date.now();
    const expiredKeys: string[] = [];

    for (const [key, entry] of Array.from(this.cache.entries())) {
      if (now - entry.timestamp > entry.ttl) {
        expiredKeys.push(key);
      }
    }

    expiredKeys.forEach(key => this.cache.delete(key));
  }

  destroy(): void {
    if (this.cleanupInterval) {
      clearInterval(this.cleanupInterval);
    }
    this.clear();
  }
}

// Global cache instances for different data types
export class GlobalCacheManager {
  private static instance: GlobalCacheManager;
  private caches: Map<string, MemoryCache<any>> = new Map();

  private constructor() {
    // Initialize caches for different data types
    this.caches.set('user_profiles', new MemoryCache(500, 60000)); // 500 profiles, 1min cleanup
    this.caches.set('conversations', new MemoryCache(2000, 30000)); // 2000 conversations, 30s cleanup
    this.caches.set('strategies', new MemoryCache(1000, 120000)); // 1000 strategies, 2min cleanup
    this.caches.set('embeddings', new MemoryCache(5000, 300000)); // 5000 embeddings, 5min cleanup
    this.caches.set('search_results', new MemoryCache(1000, 60000)); // 1000 search results, 1min cleanup
  }

  static getInstance(): GlobalCacheManager {
    if (!GlobalCacheManager.instance) {
      GlobalCacheManager.instance = new GlobalCacheManager();
    }
    return GlobalCacheManager.instance;
  }

  getCache<T>(name: string): MemoryCache<T> {
    return this.caches.get(name) as MemoryCache<T>;
  }

  getStats(): Record<string, any> {
    const stats: Record<string, any> = {};
    for (const [name, cache] of Array.from(this.caches.entries())) {
      stats[name] = cache.getStats();
    }
    return stats;
  }

  clearAll(): void {
    for (const cache of Array.from(this.caches.values())) {
      cache.clear();
    }
  }

  // Prefetch optimization - preload frequently accessed data
  async prefetch<T>(cacheName: string, keys: string[], fetcher: (key: string) => Promise<T>): Promise<void> {
    const cache = this.getCache<T>(cacheName);

    const prefetchPromises = keys.map(async (key) => {
      if (!cache.has(key)) {
        try {
          const data = await fetcher(key);
          cache.set(key, data, 300000); // 5 minute TTL
        } catch (error) {
          console.warn(`Failed to prefetch ${cacheName}:${key}`, error);
        }
      }
    });

    await Promise.allSettled(prefetchPromises);
  }
}

// Cache key generation utilities
export class CacheKey {
  static userProfile(userId: string): string {
    return `profile:${userId}`;
  }

  static conversationHistory(userId: string, sessionId: string): string {
    return `conversations:${userId}:${sessionId}`;
  }

  static strategySearch(query: string, domain: string): string {
    return `strategy_search:${this.hash(query)}:${domain}`;
  }

  static userEmbeddings(userId: string): string {
    return `embeddings:user:${userId}`;
  }

  static conversationEmbeddings(conversationId: string): string {
    return `embeddings:conversation:${conversationId}`;
  }

  static searchResults(query: string, filters: Record<string, any>): string {
    const filterHash = this.hash(JSON.stringify(filters));
    return `search:${this.hash(query)}:${filterHash}`;
  }

  private static hash(text: string): string {
    let hash = 0;
    for (let i = 0; i < text.length; i++) {
      const char = text.charCodeAt(i);
      hash = ((hash << 5) - hash) + char;
      hash = hash & hash; // Convert to 32-bit integer
    }
    return Math.abs(hash).toString(36);
  }
}

