import { useState, useEffect } from 'react';

interface User {
  id: string;
  email: string;
  firstName?: string;
  lastName?: string;
  subscriptionTier: string;
  emailVerified: boolean;
  riskTolerance?: string;
  experienceLevel?: string;
  preferredMarkets?: string[];
  favoriteStrategies?: string[];
  lastLoginAt?: string;
  trialEndsAt?: string;
  createdAt: string;
  updatedAt: string;
}

interface UserStats {
  aiGenerations: number;
  strategiesCreated: number;
  backtestsRun: number;
  chatMessages: number;
}

interface UserProfile {
  user: User;
  stats: UserStats;
}

export function useUser() {
  const [user, setUser] = useState<UserProfile | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    const fetchUser = async () => {
      try {
        const token = localStorage.getItem('authToken');
        const headers: Record<string, string> = {
          'Content-Type': 'application/json',
        };
        if (token) {
          headers['Authorization'] = `Bearer ${token}`;
        }

        // Always attempt profile fetch; supports cookie-based auth too.
        const response = await fetch('/api/auth/profile', {
          headers,
          credentials: 'include',
        });

        if (response.ok) {
          const data = await response.json();
          setUser(data);
        } else {
          // If profile fetch fails, user might not be authenticated
          if (response.status === 401 || response.status === 403) {
            localStorage.removeItem('authToken');
            setUser(null);
            setError(null);
          } else {
            setError('Failed to load user profile');
          }
        }
      } catch (err) {
        console.error('Error fetching user profile:', err);
        setError('Failed to load user profile');
      } finally {
        setLoading(false);
      }
    };

    fetchUser();
  }, []);

  return { user, loading, error };
}
