/**
 * Comprehensive Error Handling and Monitoring System
 * Global error boundaries, structured logging, and performance monitoring
 */

import { NextRequest, NextResponse } from 'next/server';
import { DatabaseService } from './database';

// Error types
export enum ErrorType {
  VALIDATION_ERROR = 'VALIDATION_ERROR',
  AUTHENTICATION_ERROR = 'AUTHENTICATION_ERROR',
  AUTHORIZATION_ERROR = 'AUTHORIZATION_ERROR',
  DATABASE_ERROR = 'DATABASE_ERROR',
  EXTERNAL_API_ERROR = 'EXTERNAL_API_ERROR',
  RATE_LIMIT_ERROR = 'RATE_LIMIT_ERROR',
  INPUT_VALIDATION_ERROR = 'INPUT_VALIDATION_ERROR',
  BUSINESS_LOGIC_ERROR = 'BUSINESS_LOGIC_ERROR',
  SYSTEM_ERROR = 'SYSTEM_ERROR',
  NETWORK_ERROR = 'NETWORK_ERROR',
}

export enum ErrorSeverity {
  LOW = 'LOW',
  MEDIUM = 'MEDIUM',
  HIGH = 'HIGH',
  CRITICAL = 'CRITICAL',
}

export interface ErrorContext {
  userId?: string;
  sessionId?: string;
  requestId: string;
  url?: string;
  method?: string;
  userAgent?: string;
  ip?: string;
  timestamp: Date;
  stackTrace?: string;
  metadata?: Record<string, any>;
  details?: any;
  service?: string;
}

export class AppError extends Error {
  public readonly type: ErrorType;
  public readonly severity: ErrorSeverity;
  public readonly statusCode: number;
  public readonly context: ErrorContext;
  public readonly isOperational: boolean;

  constructor(
    message: string,
    type: ErrorType,
    severity: ErrorSeverity = ErrorSeverity.MEDIUM,
    statusCode: number = 500,
    context: Partial<ErrorContext> = {},
    isOperational: boolean = true
  ) {
    super(message);
    this.name = 'AppError';
    this.type = type;
    this.severity = severity;
    this.statusCode = statusCode;
    this.isOperational = isOperational;

    this.context = {
      requestId: context.requestId || generateRequestId(),
      timestamp: new Date(),
      ...context,
    };

    // Capture stack trace
    Error.captureStackTrace(this, this.constructor);
    this.context.stackTrace = this.stack;
  }
}

// Error creation helpers
export class ErrorFactory {
  static validation(message: string, context: Partial<ErrorContext> = {}) {
    return new AppError(message, ErrorType.VALIDATION_ERROR, ErrorSeverity.LOW, 400, context);
  }

  static authentication(message: string = 'Authentication required', context: Partial<ErrorContext> = {}) {
    return new AppError(message, ErrorType.AUTHENTICATION_ERROR, ErrorSeverity.MEDIUM, 401, context);
  }

  static authorization(message: string = 'Insufficient permissions', context: Partial<ErrorContext> = {}) {
    return new AppError(message, ErrorType.AUTHORIZATION_ERROR, ErrorSeverity.MEDIUM, 403, context);
  }

  static notFound(resource: string = 'Resource', context: Partial<ErrorContext> = {}) {
    return new AppError(`${resource} not found`, ErrorType.BUSINESS_LOGIC_ERROR, ErrorSeverity.LOW, 404, context);
  }

  static rateLimit(message: string = 'Rate limit exceeded', context: Partial<ErrorContext> = {}) {
    return new AppError(message, ErrorType.RATE_LIMIT_ERROR, ErrorSeverity.MEDIUM, 429, context);
  }

  static database(message: string, context: Partial<ErrorContext> = {}) {
    return new AppError(message, ErrorType.DATABASE_ERROR, ErrorSeverity.HIGH, 500, context);
  }

  static externalAPI(message: string, service: string, context: Partial<ErrorContext> = {}) {
    return new AppError(
      `External API error: ${service} - ${message}`,
      ErrorType.EXTERNAL_API_ERROR,
      ErrorSeverity.HIGH,
      502,
      { ...context, service }
    );
  }

  static system(message: string, context: Partial<ErrorContext> = {}) {
    return new AppError(message, ErrorType.SYSTEM_ERROR, ErrorSeverity.CRITICAL, 500, context, false);
  }
}

// Logging system
export class Logger {
  private static instance: Logger;
  private logLevel: string;

  private constructor() {
    this.logLevel = process.env.LOG_LEVEL || 'info';
  }

  static getInstance(): Logger {
    if (!Logger.instance) {
      Logger.instance = new Logger();
    }
    return Logger.instance;
  }

  private shouldLog(level: string): boolean {
    const levels = ['error', 'warn', 'info', 'debug'];
    const currentLevelIndex = levels.indexOf(this.logLevel);
    const messageLevelIndex = levels.indexOf(level);
    return messageLevelIndex <= currentLevelIndex;
  }

  private formatLog(level: string, message: string, meta: any = {}): string {
    const timestamp = new Date().toISOString();
    const logEntry = {
      timestamp,
      level,
      message,
      ...meta,
    };
    return JSON.stringify(logEntry);
  }

  error(message: string, meta: any = {}) {
    if (this.shouldLog('error')) {
      console.error(this.formatLog('error', message, meta));
    }
  }

  warn(message: string, meta: any = {}) {
    if (this.shouldLog('warn')) {
      console.warn(this.formatLog('warn', message, meta));
    }
  }

  info(message: string, meta: any = {}) {
    if (this.shouldLog('info')) {
      console.log(this.formatLog('info', message, meta));
    }
  }

  debug(message: string, meta: any = {}) {
    if (this.shouldLog('debug')) {
      console.debug(this.formatLog('debug', message, meta));
    }
  }
}

// Global error handler
export class ErrorHandler {
  private static logger = Logger.getInstance();

  static async handleError(error: Error | AppError, context: Partial<ErrorContext> = {}): Promise<void> {
    const isAppError = error instanceof AppError;

    const errorContext: ErrorContext = {
      requestId: generateRequestId(),
      timestamp: new Date(),
      stackTrace: error.stack,
      ...context,
    };

    // Log error
    if (isAppError) {
      const appError = error as AppError;
      this.logger.error(appError.message, {
        type: appError.type,
        severity: appError.severity,
        statusCode: appError.statusCode,
        context: errorContext,
        isOperational: appError.isOperational,
      });

      // Record in database for high-severity errors
      if (appError.severity === ErrorSeverity.HIGH || appError.severity === ErrorSeverity.CRITICAL) {
        try {
          await DatabaseService.recordSystemEvent('ERROR_OCCURRED', {
            type: appError.type,
            severity: appError.severity,
            message: appError.message,
            context: errorContext,
          });
        } catch (dbError) {
          this.logger.error('Failed to record error in database', { error: dbError });
        }
      }
    } else {
      this.logger.error(error.message, {
        type: 'UNHANDLED_ERROR',
        severity: ErrorSeverity.HIGH,
        context: errorContext,
        stack: error.stack,
      });

      // Record unhandled errors
      try {
        await DatabaseService.recordSystemEvent('UNHANDLED_ERROR', {
          message: error.message,
          stack: error.stack,
          context: errorContext,
        });
      } catch (dbError) {
        this.logger.error('Failed to record unhandled error in database', { error: dbError });
      }
    }
  }

  static handleAPIError(error: Error | AppError, request: NextRequest): NextResponse {
    const isAppError = error instanceof AppError;

    if (isAppError) {
      const appError = error as AppError;
      return NextResponse.json(
        {
          error: appError.message,
          type: appError.type,
          ...(process.env.NODE_ENV === 'development' && { stack: appError.stack }),
        },
        { status: appError.statusCode }
      );
    }

    // Handle unknown errors
    this.handleError(error, {
      url: request.url,
      method: request.method,
      userAgent: request.headers.get('user-agent'),
      ip: (request as any).ip,
    });

    return NextResponse.json(
      {
        error: 'Internal server error',
        ...(process.env.NODE_ENV === 'development' && { stack: error.stack }),
      },
      { status: 500 }
    );
  }

  static createErrorResponse(
    message: string,
    statusCode: number = 500,
    type: string = 'ERROR'
  ): NextResponse {
    return NextResponse.json(
      { error: message, type },
      { status: statusCode }
    );
  }
}

// Performance monitoring
export class PerformanceMonitor {
  private static instance: PerformanceMonitor;
  private metrics: Map<string, { count: number; totalTime: number; errors: number }> = new Map();

  private constructor() {
    // Start periodic metrics reporting
    if (typeof globalThis !== 'undefined' && globalThis.process) {
      setInterval(() => this.reportMetrics(), 60000); // Report every minute
    }
  }

  static getInstance(): PerformanceMonitor {
    if (!PerformanceMonitor.instance) {
      PerformanceMonitor.instance = new PerformanceMonitor();
    }
    return PerformanceMonitor.instance;
  }

  startTimer(operation: string): () => void {
    const startTime = Date.now();

    return () => {
      const duration = Date.now() - startTime;
      this.recordMetric(operation, duration);
    };
  }

  recordMetric(operation: string, duration: number, error: boolean = false): void {
    const existing = this.metrics.get(operation) || { count: 0, totalTime: 0, errors: 0 };

    existing.count++;
    existing.totalTime += duration;
    if (error) existing.errors++;

    this.metrics.set(operation, existing);
  }

  private async reportMetrics(): Promise<void> {
    if (this.metrics.size === 0) return;

    const logger = Logger.getInstance();
    const metrics = Array.from(this.metrics.entries()).map(([operation, data]) => ({
      operation,
      count: data.count,
      avgTime: data.totalTime / data.count,
      totalTime: data.totalTime,
      errors: data.errors,
      errorRate: data.errors / data.count,
    }));

    logger.info('Performance metrics report', { metrics });

    // Record in database
    try {
      for (const metric of metrics) {
        await DatabaseService.recordSystemEvent('PERFORMANCE_METRIC', metric);
      }
    } catch (error) {
      logger.error('Failed to record performance metrics', { error });
    }

    // Reset metrics
    this.metrics.clear();
  }
}

// Request tracking middleware
export class RequestTracker {
  static trackRequest(request: NextRequest, response: NextResponse, duration: number): void {
    const logger = Logger.getInstance();
    const monitor = PerformanceMonitor.getInstance();

    const route = request.nextUrl.pathname;
    const method = request.method;
    const statusCode = response.status;
    const userAgent = request.headers.get('user-agent');
    const ip = (request as any).ip;

    // Log request
    logger.info('API Request', {
      route,
      method,
      statusCode,
      duration,
      userAgent: userAgent?.substring(0, 100),
      ip,
    });

    // Record performance
    monitor.recordMetric(`${method} ${route}`, duration, statusCode >= 400);

    // Record analytics if user is authenticated
    const user = (request as any).user;
    if (user) {
      DatabaseService.recordUserEvent(user.id, 'API_REQUEST', {
        route,
        method,
        statusCode,
        duration,
        userAgent: userAgent?.substring(0, 100),
      }).catch(error => {
        logger.error('Failed to record user analytics', { error, userId: user.id });
      });
    }
  }
}

// Health check endpoint data
export class HealthChecker {
  static async getHealthStatus(): Promise<{
    status: 'healthy' | 'degraded' | 'unhealthy';
    checks: Record<string, { status: 'pass' | 'fail'; message?: string }>;
    timestamp: string;
  }> {
    const checks: Record<string, { status: 'pass' | 'fail'; message?: string }> = {};

    // Database health check
    try {
      const dbHealthy = await DatabaseService.healthCheck();
      checks.database = dbHealthy
        ? { status: 'pass' }
        : { status: 'fail', message: 'Database connection failed' };
    } catch (error) {
      checks.database = { status: 'fail', message: `Database error: ${error.message}` };
    }

    // External API checks
    try {
      // Check Grok API key format (don't make actual API call)
      const grokKey = process.env.GROK_API_KEY;
      checks.grok_api = grokKey?.startsWith('xai-')
        ? { status: 'pass' }
        : { status: 'fail', message: 'Grok API key not configured' };
    } catch (error) {
      checks.grok_api = { status: 'fail', message: 'Grok API check failed' };
    }

    // Determine overall status
    const failedChecks = Object.values(checks).filter(check => check.status === 'fail');
    const status = failedChecks.length === 0 ? 'healthy' :
      failedChecks.length === Object.keys(checks).length ? 'unhealthy' : 'degraded';

    return {
      status,
      checks,
      timestamp: new Date().toISOString(),
    };
  }
}

// Utility functions
function generateRequestId(): string {
  return `req_${Date.now()}_${Math.random().toString(36).substring(2, 15)}`;
}

// Export singleton instances
export const logger = Logger.getInstance();
export const performanceMonitor = PerformanceMonitor.getInstance();
export const handleApiError = (error: Error | AppError, request: NextRequest) => ErrorHandler.handleAPIError(error, request);
