/**
 * Email Service Layer
 * Handles sending emails via SMTP with templates and tracking
 */

import nodemailer from 'nodemailer';
import { DatabaseService } from './database';
import { logger, ErrorFactory } from './error-handler';

// Email configuration - supports both authenticated SMTP and local postfix
const getSmtpConfig = () => {
  const host = process.env.SMTP_HOST || 'localhost';
  const port = parseInt(process.env.SMTP_PORT || '25');
  const useAuth = process.env.SMTP_USER && process.env.SMTP_PASS;
  
  // Base config for local postfix (no auth needed)
  const baseConfig: any = {
    host,
    port,
    secure: port === 465, // true for 465, false for other ports
    tls: {
      rejectUnauthorized: false, // Allow self-signed or mismatched certificates
    },
  };
  
  // Add auth only if credentials are provided
  if (useAuth) {
    baseConfig.auth = {
      user: process.env.SMTP_USER,
      pass: process.env.SMTP_PASS,
    };
  }
  
  return baseConfig;
};

const SMTP_CONFIG = getSmtpConfig();

// Email templates
export const EMAIL_TEMPLATES = {
  WELCOME: 'welcome',
  EMAIL_VERIFICATION: 'email_verification',
  PASSWORD_RESET: 'password_reset',
  SUBSCRIPTION_CONFIRMED: 'subscription_confirmed',
  PAYMENT_FAILED: 'payment_failed',
  TRIAL_ENDING: 'trial_ending',
  ACCOUNT_DEACTIVATED: 'account_deactivated',
} as const;

export type EmailTemplate = typeof EMAIL_TEMPLATES[keyof typeof EMAIL_TEMPLATES];

export interface EmailData {
  to: string;
  subject: string;
  template: EmailTemplate;
  data: Record<string, any>;
}

export class EmailService {
  private static transporter: nodemailer.Transporter | null = null;
  private static initialized = false;

  static initialize() {
    if (this.initialized) return;

    try {
      const smtpHost = process.env.SMTP_HOST || 'localhost';
      const smtpPort = parseInt(process.env.SMTP_PORT || '25');
      
      // Check if we're in production with localhost SMTP (postfix) or have explicit credentials
      const isLocalPostfix = smtpHost === 'localhost';
      const hasCredentials = process.env.SMTP_USER && process.env.SMTP_PASS;
      
      // For local postfix, no credentials needed
      if (isLocalPostfix) {
        console.log(`📧 Initializing local postfix transport: ${smtpHost}:${smtpPort}`);
        this.transporter = nodemailer.createTransport({
          host: smtpHost,
          port: smtpPort,
          secure: false,
          tls: {
            rejectUnauthorized: false,
          },
        });
      } else if (hasCredentials) {
        const config = getSmtpConfig();
        console.log(`📧 Initializing SMTP transport: ${config.host}:${config.port} (auth: yes)`);
        this.transporter = nodemailer.createTransport(config);
      } else {
        console.warn('⚠️  SMTP credentials not configured and not using localhost. Using mock email service.');
        this.transporter = null;
      }

      // Verify connection only when we have a real transporter
      if (this.transporter) {
        this.transporter.verify((error, success) => {
          if (error) {
            console.error('❌ SMTP verification failed:', error.message);
            // Don't null out transporter - let sendEmail try anyway
          } else {
            console.log('✅ SMTP connection verified successfully');
          }
        });
      }

      this.initialized = true;
    } catch (error: any) {
      const errorMessage = error?.message || error?.toString() || 'Unknown error';
      console.error("Email service initialization failed:", errorMessage);
      this.transporter = null;
      this.initialized = true;
    }
  }

  static async sendEmail(emailData: EmailData): Promise<boolean> {
    if (!this.initialized) {
      this.initialize();
    }

    try {
      const { html, text } = this.renderTemplate(emailData.template, emailData.data);

      // Mock mode - log email instead of sending
      if (!this.transporter) {
        console.log('📧 [MOCK EMAIL] Would send email:', {
          to: emailData.to,
          subject: emailData.subject,
          template: emailData.template,
          html: html.substring(0, 200) + '...',
        });

        // Simulate successful sending for development
        return true;
      }

      const mailOptions = {
        from: `"${process.env.FROM_NAME || 'EventheOdds.ai'}" <${process.env.EMAIL_FROM || process.env.SMTP_USER || 'noreply@eventheodds.ai'}>`,
        to: emailData.to,
        subject: emailData.subject,
        html,
        text,
      };

      const result = await this.transporter.sendMail(mailOptions);

      logger.info('Email sent successfully', {
        to: emailData.to,
        template: emailData.template,
        messageId: result.messageId,
      });

      // Record email in database for analytics
      await DatabaseService.recordSystemEvent('EMAIL_SENT', {
        to: emailData.to,
        template: emailData.template,
        subject: emailData.subject,
        messageId: result.messageId,
      });

      return true;
    } catch (error) {
      logger.error('Failed to send email', {
        error: error.message,
        to: emailData.to,
        template: emailData.template,
      });

      // Log as mock email for debugging
      console.log('📧 [EMAIL FAILED - Mock fallback]:', {
        to: emailData.to,
        subject: emailData.subject,
        template: emailData.template,
        error: error.message,
      });

      // Record failed email but don't block the calling function
      try {
        await DatabaseService.recordSystemEvent('EMAIL_FAILED', {
          to: emailData.to,
          template: emailData.template,
          error: error.message,
        });
      } catch (dbError) {
        console.error('Failed to record email failure:', dbError);
      }

      // Return true to not block signup/other flows when email fails
      // Users can request a resend later
      return true;
    }
  }

  static renderTemplate(template: EmailTemplate, data: Record<string, any>): { html: string; text: string } {
    switch (template) {
      case EMAIL_TEMPLATES.WELCOME:
        return this.renderWelcomeEmail(data);
      case EMAIL_TEMPLATES.EMAIL_VERIFICATION:
        return this.renderEmailVerification(data);
      case EMAIL_TEMPLATES.PASSWORD_RESET:
        return this.renderPasswordReset(data);
      case EMAIL_TEMPLATES.SUBSCRIPTION_CONFIRMED:
        return this.renderSubscriptionConfirmed(data);
      case EMAIL_TEMPLATES.PAYMENT_FAILED:
        return this.renderPaymentFailed(data);
      case EMAIL_TEMPLATES.TRIAL_ENDING:
        return this.renderTrialEnding(data);
      case EMAIL_TEMPLATES.ACCOUNT_DEACTIVATED:
        return this.renderAccountDeactivated(data);
      default:
        throw new Error(`Unknown email template: ${template}`);
    }
  }

  private static renderWelcomeEmail(data: Record<string, any>): { html: string; text: string } {
    const { firstName, email, subscriptionTier } = data;

    const html = `
      <!DOCTYPE html>
      <html>
        <head>
          <meta charset="utf-8">
          <title>Welcome to EventheOdds.ai</title>
          <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 30px; text-align: center; border-radius: 8px 8px 0 0; }
            .content { background: white; padding: 30px; border: 1px solid #ddd; border-top: none; border-radius: 0 0 8px 8px; }
            .button { display: inline-block; padding: 12px 24px; background: #667eea; color: white; text-decoration: none; border-radius: 6px; margin: 20px 0; }
            .footer { text-align: center; margin-top: 30px; color: #666; font-size: 12px; }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="header">
              <h1>Welcome to EventheOdds.ai! 🎉</h1>
            </div>
            <div class="content">
              <h2>Hi ${firstName}!</h2>
              <p>Thank you for joining EventheOdds.ai. You're now ready to transform your trading ideas into profitable strategies.</p>

              <p>Your <strong>${subscriptionTier}</strong> account gives you access to:</p>
              <ul>
                <li>AI-powered strategy generation</li>
                <li>Advanced backtesting across multiple markets</li>
                <li>Real-time strategy optimization</li>
                <li>Comprehensive performance analytics</li>
              </ul>

              <p>Get started by describing your first trading strategy in plain English!</p>

              <a href="${process.env.NEXT_PUBLIC_BASE_URL || 'http://localhost:3000'}/dashboard" class="button">Start Trading</a>

              <p>Need help? Reply to this email or contact our support team.</p>
            </div>
            <div class="footer">
              <p>© 2024 EventheOdds.ai. All rights reserved.</p>
              <p>You received this email because you signed up for EventheOdds.ai.</p>
            </div>
          </div>
        </body>
      </html>
    `;

    const text = `
      Welcome to EventheOdds.ai!

      Hi ${firstName}!

      Thank you for joining EventheOdds.ai. Your ${subscriptionTier} account is now active.

      Get started: ${process.env.NEXT_PUBLIC_BASE_URL || 'http://localhost:3000'}/dashboard

      Need help? Contact our support team.
    `;

    return { html, text };
  }

  private static renderEmailVerification(data: Record<string, any>): { html: string; text: string } {
    const { firstName, verificationToken, email } = data;
    const verificationUrl = `${process.env.NEXT_PUBLIC_BASE_URL || 'http://localhost:3000'}/verify-email?token=${verificationToken}`;

    const html = `
      <!DOCTYPE html>
      <html>
        <head>
          <meta charset="utf-8">
          <title>Verify Your Email - EventheOdds.ai</title>
          <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 30px; text-align: center; border-radius: 8px 8px 0 0; }
            .content { background: white; padding: 30px; border: 1px solid #ddd; border-top: none; border-radius: 0 0 8px 8px; }
            .button { display: inline-block; padding: 12px 24px; background: #667eea; color: white; text-decoration: none; border-radius: 6px; margin: 20px 0; }
            .code { background: #f8f9fa; padding: 15px; border-radius: 6px; font-family: monospace; font-size: 18px; text-align: center; margin: 20px 0; }
            .footer { text-align: center; margin-top: 30px; color: #666; font-size: 12px; }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="header">
              <h1>Verify Your Email Address</h1>
            </div>
            <div class="content">
              <h2>Hi ${firstName}!</h2>
              <p>Thanks for signing up for EventheOdds.ai. To complete your registration, please verify your email address.</p>

              <div style="text-align: center; margin: 30px 0;">
                <a href="${verificationUrl}" class="button">Verify Email Address</a>
              </div>

              <p>If the button doesn't work, copy and paste this link into your browser:</p>
              <p style="word-break: break-all; color: #667eea;">${verificationUrl}</p>

              <p><strong>This link will expire in 24 hours.</strong></p>

              <p>If you didn't create an account, you can safely ignore this email.</p>
            </div>
            <div class="footer">
              <p>© 2024 EventheOdds.ai. All rights reserved.</p>
            </div>
          </div>
        </body>
      </html>
    `;

    const text = `
      Verify Your Email - EventheOdds.ai

      Hi ${firstName}!

      Thanks for signing up. To complete your registration, verify your email:

      ${verificationUrl}

      This link expires in 24 hours.

      If you didn't create an account, ignore this email.
    `;

    return { html, text };
  }

  private static renderPasswordReset(data: Record<string, any>): { html: string; text: string } {
    const { firstName, resetToken } = data;
    const resetUrl = `${process.env.NEXT_PUBLIC_BASE_URL || 'http://localhost:3000'}/reset-password?token=${resetToken}`;

    const html = `
      <!DOCTYPE html>
      <html>
        <head>
          <meta charset="utf-8">
          <title>Reset Your Password - EventheOdds.ai</title>
          <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 30px; text-align: center; border-radius: 8px 8px 0 0; }
            .content { background: white; padding: 30px; border: 1px solid #ddd; border-top: none; border-radius: 0 0 8px 8px; }
            .button { display: inline-block; padding: 12px 24px; background: #667eea; color: white; text-decoration: none; border-radius: 6px; margin: 20px 0; }
            .warning { background: #fff3cd; border: 1px solid #ffeaa7; padding: 15px; border-radius: 6px; margin: 20px 0; }
            .footer { text-align: center; margin-top: 30px; color: #666; font-size: 12px; }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="header">
              <h1>Reset Your Password</h1>
            </div>
            <div class="content">
              <h2>Hi ${firstName}!</h2>
              <p>You requested to reset your password for your EventheOdds.ai account.</p>

              <div style="text-align: center; margin: 30px 0;">
                <a href="${resetUrl}" class="button">Reset Password</a>
              </div>

              <p>If the button doesn't work, copy and paste this link:</p>
              <p style="word-break: break-all; color: #667eea;">${resetUrl}</p>

              <div class="warning">
                <strong>Security Notice:</strong> This link will expire in 1 hour and can only be used once. If you didn't request this reset, please ignore this email.
              </div>

              <p>For your security, we recommend choosing a strong password with at least 8 characters including uppercase, lowercase, and numbers.</p>
            </div>
            <div class="footer">
              <p>© 2024 EventheOdds.ai. All rights reserved.</p>
            </div>
          </div>
        </body>
      </html>
    `;

    const text = `
      Reset Your Password - EventheOdds.ai

      Hi ${firstName}!

      You requested to reset your password.

      Reset link: ${resetUrl}

      This link expires in 1 hour. If you didn't request this, ignore this email.
    `;

    return { html, text };
  }

  private static renderSubscriptionConfirmed(data: Record<string, any>): { html: string; text: string } {
    const { firstName, tier, price, trialEndsAt } = data;

    const html = `
      <!DOCTYPE html>
      <html>
        <head>
          <meta charset="utf-8">
          <title>Subscription Confirmed - EventheOdds.ai</title>
          <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: linear-gradient(135deg, #28a745 0%, #20c997 100%); color: white; padding: 30px; text-align: center; border-radius: 8px 8px 0 0; }
            .content { background: white; padding: 30px; border: 1px solid #ddd; border-top: none; border-radius: 0 0 8px 8px; }
            .success { background: #d4edda; border: 1px solid #c3e6cb; padding: 15px; border-radius: 6px; margin: 20px 0; }
            .button { display: inline-block; padding: 12px 24px; background: #667eea; color: white; text-decoration: none; border-radius: 6px; margin: 20px 0; }
            .footer { text-align: center; margin-top: 30px; color: #666; font-size: 12px; }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="header">
              <h1>🎉 Subscription Confirmed!</h1>
            </div>
            <div class="content">
              <h2>Welcome to ${tier}, ${firstName}!</h2>

              <div class="success">
                <strong>✓ Your subscription has been activated</strong>
                <p>You now have access to all ${tier} features.</p>
              </div>

              <p><strong>Plan Details:</strong></p>
              <ul>
                <li>Plan: ${tier}</li>
                <li>Billing: $${price}/month</li>
                ${trialEndsAt ? `<li>Free Trial: Ends ${new Date(trialEndsAt).toLocaleDateString()}</li>` : ''}
              </ul>

              <a href="${process.env.NEXT_PUBLIC_BASE_URL || 'http://localhost:3000'}/dashboard" class="button">Start Trading</a>

              <p>You can manage your subscription and billing information anytime from your account settings.</p>
            </div>
            <div class="footer">
              <p>© 2024 EventheOdds.ai. All rights reserved.</p>
            </div>
          </div>
        </body>
      </html>
    `;

    const text = `
      Subscription Confirmed - EventheOdds.ai

      Welcome to ${tier}, ${firstName}!

      Your subscription has been activated.
      Plan: ${tier}
      Billing: $${price}/month
      ${trialEndsAt ? `Free Trial Ends: ${new Date(trialEndsAt).toLocaleDateString()}` : ''}

      Start trading: ${process.env.NEXT_PUBLIC_BASE_URL || 'http://localhost:3000'}/dashboard
    `;

    return { html, text };
  }

  private static renderPaymentFailed(data: Record<string, any>): { html: string; text: string } {
    const { firstName, amount, last4 } = data;

    const html = `
      <!DOCTYPE html>
      <html>
        <head>
          <meta charset="utf-8">
          <title>Payment Failed - EventheOdds.ai</title>
          <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: linear-gradient(135deg, #dc3545 0%, #c82333 100%); color: white; padding: 30px; text-align: center; border-radius: 8px 8px 0 0; }
            .content { background: white; padding: 30px; border: 1px solid #ddd; border-top: none; border-radius: 0 0 8px 8px; }
            .warning { background: #f8d7da; border: 1px solid #f5c6cb; padding: 15px; border-radius: 6px; margin: 20px 0; }
            .button { display: inline-block; padding: 12px 24px; background: #667eea; color: white; text-decoration: none; border-radius: 6px; margin: 20px 0; }
            .footer { text-align: center; margin-top: 30px; color: #666; font-size: 12px; }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="header">
              <h1>Payment Failed</h1>
            </div>
            <div class="content">
              <h2>Hi ${firstName},</h2>

              <div class="warning">
                <strong>⚠️ We couldn't process your payment</strong>
                <p>There was an issue charging $${amount} to your card ending in ${last4}.</p>
              </div>

              <p><strong>Possible reasons:</strong></p>
              <ul>
                <li>Insufficient funds</li>
                <li>Expired card</li>
                <li>Incorrect card details</li>
                <li>Bank declined the transaction</li>
              </ul>

              <p>Please update your payment method to continue using EventheOdds.ai.</p>

              <a href="${process.env.NEXT_PUBLIC_BASE_URL || 'http://localhost:3000'}/billing" class="button">Update Payment Method</a>

              <p>If this issue persists, contact our support team for assistance.</p>
            </div>
            <div class="footer">
              <p>© 2024 EventheOdds.ai. All rights reserved.</p>
            </div>
          </div>
        </body>
      </html>
    `;

    const text = `
      Payment Failed - EventheOdds.ai

      Hi ${firstName},

      We couldn't process your payment of $${amount} to card ending in ${last4}.

      Please update your payment method: ${process.env.NEXT_PUBLIC_BASE_URL || 'http://localhost:3000'}/billing

      Contact support if issues persist.
    `;

    return { html, text };
  }

  private static renderTrialEnding(data: Record<string, any>): { html: string; text: string } {
    const { firstName, daysLeft } = data;

    const html = `
      <!DOCTYPE html>
      <html>
        <head>
          <meta charset="utf-8">
          <title>Trial Ending Soon - EventheOdds.ai</title>
          <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: linear-gradient(135deg, #ffc107 0%, #fd7e14 100%); color: white; padding: 30px; text-align: center; border-radius: 8px 8px 0 0; }
            .content { background: white; padding: 30px; border: 1px solid #ddd; border-top: none; border-radius: 0 0 8px 8px; }
            .urgent { background: #fff3cd; border: 1px solid #ffeaa7; padding: 15px; border-radius: 6px; margin: 20px 0; }
            .button { display: inline-block; padding: 12px 24px; background: #667eea; color: white; text-decoration: none; border-radius: 6px; margin: 20px 0; }
            .cancel { color: #6c757d; text-decoration: underline; }
            .footer { text-align: center; margin-top: 30px; color: #666; font-size: 12px; }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="header">
              <h1>⏰ Trial Ending Soon</h1>
            </div>
            <div class="content">
              <h2>Hi ${firstName},</h2>

              <div class="urgent">
                <strong>Your free trial ends in ${daysLeft} day${daysLeft > 1 ? 's' : ''}!</strong>
                <p>Starting tomorrow, you'll be billed $20/month for continued access.</p>
              </div>

              <p>Love EventheOdds.ai? Keep your subscription active. Not ready yet? Cancel before your trial ends.</p>

              <div style="text-align: center; margin: 30px 0;">
                <a href="${process.env.NEXT_PUBLIC_BASE_URL || 'http://localhost:3000'}/billing" class="button">Manage Subscription</a>
              </div>

              <p>
                <a href="#" class="cancel">Cancel subscription</a> if you no longer want to continue.
              </p>

              <p>Questions? Reply to this email or contact support.</p>
            </div>
            <div class="footer">
              <p>© 2024 EventheOdds.ai. All rights reserved.</p>
            </div>
          </div>
        </body>
      </html>
    `;

    const text = `
      Trial Ending Soon - EventheOdds.ai

      Hi ${firstName},

      Your free trial ends in ${daysLeft} day${daysLeft > 1 ? 's' : ''}.

      Starting tomorrow, you'll be billed $20/month.

      Manage subscription: ${process.env.NEXT_PUBLIC_BASE_URL || 'http://localhost:3000'}/billing

      Questions? Contact support.
    `;

    return { html, text };
  }

  private static renderAccountDeactivated(data: Record<string, any>): { html: string; text: string } {
    const { firstName, reason } = data;

    const html = `
      <!DOCTYPE html>
      <html>
        <head>
          <meta charset="utf-8">
          <title>Account Deactivated - EventheOdds.ai</title>
          <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: linear-gradient(135deg, #6c757d 0%, #495057 100%); color: white; padding: 30px; text-align: center; border-radius: 8px 8px 0 0; }
            .content { background: white; padding: 30px; border: 1px solid #ddd; border-top: none; border-radius: 0 0 8px 8px; }
            .info { background: #f8f9fa; border: 1px solid #dee2e6; padding: 15px; border-radius: 6px; margin: 20px 0; }
            .footer { text-align: center; margin-top: 30px; color: #666; font-size: 12px; }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="header">
              <h1>Account Deactivated</h1>
            </div>
            <div class="content">
              <h2>Hi ${firstName},</h2>

              <div class="info">
                <p>Your EventheOdds.ai account has been deactivated.</p>
                ${reason ? `<p><strong>Reason:</strong> ${reason}</p>` : ''}
              </div>

              <p>If this was a mistake or you'd like to reactivate your account, please contact our support team.</p>

              <p>We hope to see you back soon!</p>
            </div>
            <div class="footer">
              <p>© 2024 EventheOdds.ai. All rights reserved.</p>
            </div>
          </div>
        </body>
      </html>
    `;

    const text = `
      Account Deactivated - EventheOdds.ai

      Hi ${firstName},

      Your account has been deactivated.
      ${reason ? `Reason: ${reason}` : ''}

      Contact support to reactivate.
    `;

    return { html, text };
  }
}

// Utility functions
export class EmailUtils {
  static generateVerificationToken(): string {
    return require('crypto').randomBytes(32).toString('hex');
  }

  static generatePasswordResetToken(): string {
    return require('crypto').randomBytes(32).toString('hex');
  }

  static async sendWelcomeEmail(user: { email: string; firstName: string; subscriptionTier: string }): Promise<boolean> {
    return EmailService.sendEmail({
      to: user.email,
      subject: 'Welcome to EventheOdds.ai! 🎉',
      template: EMAIL_TEMPLATES.WELCOME,
      data: {
        firstName: user.firstName,
        email: user.email,
        subscriptionTier: user.subscriptionTier,
      },
    });
  }

  static async sendEmailVerification(user: { email: string; firstName: string }, token: string): Promise<boolean> {
    return EmailService.sendEmail({
      to: user.email,
      subject: 'Verify Your Email - EventheOdds.ai',
      template: EMAIL_TEMPLATES.EMAIL_VERIFICATION,
      data: {
        firstName: user.firstName,
        verificationToken: token,
        email: user.email,
      },
    });
  }

  static async sendPasswordReset(user: { email: string; firstName: string }, token: string): Promise<boolean> {
    return EmailService.sendEmail({
      to: user.email,
      subject: 'Reset Your Password - EventheOdds.ai',
      template: EMAIL_TEMPLATES.PASSWORD_RESET,
      data: {
        firstName: user.firstName,
        resetToken: token,
      },
    });
  }

  static async sendSubscriptionConfirmed(user: { email: string; firstName: string }, subscription: any): Promise<boolean> {
    return EmailService.sendEmail({
      to: user.email,
      subject: 'Subscription Confirmed - EventheOdds.ai',
      template: EMAIL_TEMPLATES.SUBSCRIPTION_CONFIRMED,
      data: {
        firstName: user.firstName,
        tier: subscription.tier || 'Pro',
        price: subscription.price || 20,
        trialEndsAt: subscription.trialEndsAt,
      },
    });
  }

  static async sendPaymentFailed(user: { email: string; firstName: string }, payment: any): Promise<boolean> {
    return EmailService.sendEmail({
      to: user.email,
      subject: 'Payment Failed - Action Required',
      template: EMAIL_TEMPLATES.PAYMENT_FAILED,
      data: {
        firstName: user.firstName,
        amount: payment.amount || 20,
        last4: payment.last4 || '****',
      },
    });
  }

  static async sendTrialEnding(user: { email: string; firstName: string }, daysLeft: number): Promise<boolean> {
    return EmailService.sendEmail({
      to: user.email,
      subject: `Trial Ending in ${daysLeft} Day${daysLeft > 1 ? 's' : ''} - EventheOdds.ai`,
      template: EMAIL_TEMPLATES.TRIAL_ENDING,
      data: {
        firstName: user.firstName,
        daysLeft,
      },
    });
  }

  static async sendAccountDeactivated(user: { email: string; firstName: string }, reason?: string): Promise<boolean> {
    return EmailService.sendEmail({
      to: user.email,
      subject: 'Account Deactivated - EventheOdds.ai',
      template: EMAIL_TEMPLATES.ACCOUNT_DEACTIVATED,
      data: {
        firstName: user.firstName,
        reason,
      },
    });
  }
}
