/**
 * Database Service Layer
 * Provides a clean interface for database operations using Prisma
 */

import { PrismaClient, User, Strategy, Conversation, Message, SubscriptionTier, Domain, BacktestResult, Feedback } from '@prisma/client';
import bcrypt from 'bcryptjs';
import jwt from 'jsonwebtoken';
import crypto from 'crypto';

// Prisma Client Singleton Pattern for Next.js
const globalForPrisma = globalThis as unknown as {
  prisma: PrismaClient | undefined;
};

const prisma =
  globalForPrisma.prisma ??
  new PrismaClient({
    log: process.env.NODE_ENV === 'development' ? ['query', 'error', 'warn'] : ['error'],
  });

if (process.env.NODE_ENV !== 'production') globalForPrisma.prisma = prisma;

// JWT Secret
const JWT_SECRET = process.env.JWT_SECRET || 'your-jwt-secret-key-here';

// Types for our service
export interface CreateUserData {
  email: string;
  password: string;
  firstName?: string;
  lastName?: string;
  subscriptionTier?: SubscriptionTier;
  emailVerificationToken?: string;
  emailVerificationExpires?: Date;
  emailVerified?: boolean;
}

export interface UpdateUserData {
  firstName?: string;
  lastName?: string;
  riskTolerance?: string;
  experienceLevel?: string;
  preferredMarkets?: string[];
  favoriteStrategies?: string[];
  passwordResetToken?: string | null;
  passwordResetExpires?: Date | null;
  passwordHash?: string;
  emailVerified?: boolean;
  emailVerifiedAt?: Date | null;
  emailVerificationToken?: string | null;
  emailVerificationExpires?: Date | null;
  stripeCustomerId?: string | null;
  subscriptionId?: string | null;
  subscriptionTier?: string;
  trialEndsAt?: Date | null;
  subscriptionEndsAt?: Date | null;
  canceledAt?: Date | null;
}

export interface CreateStrategyData {
  name: string;
  description?: string;
  domain: Domain;
  rules: any;
  isPublic?: boolean;
  isTemplate?: boolean;
  tags?: string[];
  userId: string;
}

export interface CreateConversationData {
  title?: string;
  domain?: Domain;
  userId: string;
}

export interface CreateBacktestResultData {
  jobId?: string;
  userId?: string | null;
  market: string;
  strategyName: string;
  strategyCode?: string | null;
  parameters: any;
  timeframe?: string | null;
  timePeriod?: any;
  results: any;
  trades: any[];
  performance?: any;
  riskMetrics?: any;
  chartData?: any;
  status?: 'COMPLETED' | 'FAILED' | 'RUNNING' | string;
  executionTime?: number | null;
  errorMessage?: string | null;
  startedAt?: Date | null;
  completedAt?: Date | null;
}

export interface CreateFeedbackData {
  userId: string;
  feedbackType: string;
  subject: string;
  message: string;
  priority?: string;
  status?: string;
  metadata?: any;
}

export class DatabaseService {
  // User Management
  static async createUser(data: CreateUserData): Promise<User> {
    const hashedPassword = await bcrypt.hash(data.password, 12);

    return prisma.user.create({
      data: {
        email: data.email,
        passwordHash: hashedPassword,
        firstName: data.firstName,
        lastName: data.lastName,
        subscriptionTier: data.subscriptionTier || 'FREE',
        emailVerificationToken: data.emailVerificationToken,
        emailVerificationExpires: data.emailVerificationExpires,
        emailVerified: data.emailVerified || false,
      },
    });
  }

  static async findUserByEmail(email: string): Promise<User | null> {
    return prisma.user.findUnique({
      where: { email },
    });
  }

  static async findUserByVerificationToken(token: string): Promise<User | null> {
    return prisma.user.findUnique({
      where: { emailVerificationToken: token },
    });
  }

  static async findUserByPasswordResetToken(token: string): Promise<User | null> {
    return prisma.user.findUnique({
      where: { passwordResetToken: token },
    });
  }

  static async findUserById(id: string): Promise<User | null> {
    return prisma.user.findUnique({
      where: { id },
      include: {
        strategies: true,
        conversations: {
          include: {
            messages: {
              orderBy: { createdAt: 'desc' },
              take: 1,
            },
          },
          orderBy: { lastActivityAt: 'desc' },
          take: 5,
        },
      },
    });
  }

  static async updateUser(id: string, data: UpdateUserData): Promise<User> {
    // Build update object only with defined fields
    const updateData: Record<string, any> = {
      updatedAt: new Date(),
    };

    // Profile fields
    if (data.firstName !== undefined) updateData.firstName = data.firstName;
    if (data.lastName !== undefined) updateData.lastName = data.lastName;
    if (data.riskTolerance !== undefined) updateData.riskTolerance = data.riskTolerance;
    if (data.experienceLevel !== undefined) updateData.experienceLevel = data.experienceLevel;
    if (data.preferredMarkets !== undefined) updateData.preferredMarkets = JSON.stringify(data.preferredMarkets);
    if (data.favoriteStrategies !== undefined) updateData.favoriteStrategies = JSON.stringify(data.favoriteStrategies);

    // Password reset fields
    if (data.passwordResetToken !== undefined) updateData.passwordResetToken = data.passwordResetToken;
    if (data.passwordResetExpires !== undefined) updateData.passwordResetExpires = data.passwordResetExpires;
    if (data.passwordHash !== undefined) updateData.passwordHash = data.passwordHash;

    // Email verification fields
    if (data.emailVerified !== undefined) updateData.emailVerified = data.emailVerified;
    if (data.emailVerifiedAt !== undefined) updateData.emailVerifiedAt = data.emailVerifiedAt;
    if (data.emailVerificationToken !== undefined) updateData.emailVerificationToken = data.emailVerificationToken;
    if (data.emailVerificationExpires !== undefined) updateData.emailVerificationExpires = data.emailVerificationExpires;

    // Subscription fields
    if (data.stripeCustomerId !== undefined) updateData.stripeCustomerId = data.stripeCustomerId;
    if (data.subscriptionId !== undefined) updateData.subscriptionId = data.subscriptionId;
    if (data.subscriptionTier !== undefined) updateData.subscriptionTier = data.subscriptionTier;
    if (data.trialEndsAt !== undefined) updateData.trialEndsAt = data.trialEndsAt;
    if (data.subscriptionEndsAt !== undefined) updateData.subscriptionEndsAt = data.subscriptionEndsAt;
    if (data.canceledAt !== undefined) updateData.canceledAt = data.canceledAt;

    return prisma.user.update({
      where: { id },
      data: updateData,
    });
  }

  static async updateUserLastLogin(id: string): Promise<void> {
    await prisma.user.update({
      where: { id },
      data: { lastLoginAt: new Date() },
    });
  }

  static async verifyPassword(password: string, hashedPassword: string): Promise<boolean> {
    return bcrypt.compare(password, hashedPassword);
  }

  static async hashPassword(password: string): Promise<string> {
    return bcrypt.hash(password, 12);
  }

  static generateVerificationToken(): string {
    return crypto.randomBytes(32).toString('hex');
  }

  static generatePasswordResetToken(): string {
    return crypto.randomBytes(32).toString('hex');
  }

  static async generateAuthToken(user: User): Promise<string> {
    return jwt.sign(
      {
        userId: user.id,
        email: user.email,
        subscriptionTier: user.subscriptionTier,
      },
      JWT_SECRET,
      { expiresIn: '7d' }
    );
  }

  static async verifyAuthToken(token: string): Promise<any> {
    return jwt.verify(token, JWT_SECRET);
  }

  // Strategy Management
  static async createStrategy(data: CreateStrategyData): Promise<Strategy> {
    return prisma.strategy.create({
      data: {
        name: data.name,
        description: data.description,
        domain: data.domain,
        rules: typeof data.rules === 'string' ? data.rules : JSON.stringify(data.rules ?? {}),
        isPublic: data.isPublic || false,
        isTemplate: data.isTemplate || false,
        tags: data.tags ? JSON.stringify(data.tags) : null,
        userId: data.userId,
      },
    });
  }

  static async findStrategyById(id: string): Promise<Strategy | null> {
    return prisma.strategy.findUnique({
      where: { id },
      include: {
        user: {
          select: { id: true, firstName: true, lastName: true, email: true },
        },
        comments: {
          include: {
            user: { select: { id: true, firstName: true, lastName: true } },
          },
          orderBy: { createdAt: 'desc' },
        },
        ratings: true,
      },
    });
  }

  static async findStrategiesByUser(userId: string, limit: number = 50): Promise<Strategy[]> {
    return prisma.strategy.findMany({
      where: { userId },
      orderBy: { updatedAt: 'desc' },
      take: limit,
    });
  }

  static async findPublicStrategies(limit: number = 50, domain?: Domain): Promise<Strategy[]> {
    return prisma.strategy.findMany({
      where: {
        isPublic: true,
        ...(domain && { domain }),
      },
      include: {
        user: {
          select: { id: true, firstName: true, lastName: true },
        },
        ratings: true,
      },
      orderBy: { likes: 'desc' },
      take: limit,
    });
  }

  static async findStrategyTemplates(domain?: Domain): Promise<Strategy[]> {
    return prisma.strategy.findMany({
      where: {
        isTemplate: true,
        ...(domain && { domain }),
      },
      orderBy: { likes: 'desc' },
    });
  }

  static async updateStrategy(id: string, data: Partial<Strategy>): Promise<Strategy> {
    return prisma.strategy.update({
      where: { id },
      data: {
        ...data,
        updatedAt: new Date(),
      },
    });
  }

  static async incrementStrategyStats(id: string, field: 'likes' | 'forks' | 'views'): Promise<void> {
    const updateData = { [field]: { increment: 1 } };
    await prisma.strategy.update({
      where: { id },
      data: updateData,
    });
  }

  static async deleteStrategy(id: string): Promise<void> {
    await prisma.strategy.delete({
      where: { id },
    });
  }

  // Conversation Management
  static async createConversation(data: CreateConversationData): Promise<Conversation> {
    return prisma.conversation.create({
      data: {
        title: data.title,
        domain: data.domain,
        userId: data.userId,
      },
    });
  }

  static async findConversationById(id: string): Promise<Conversation | null> {
    return prisma.conversation.findUnique({
      where: { id },
      include: {
        messages: {
          orderBy: { createdAt: 'asc' },
        },
      },
    });
  }

  static async findConversationsByUser(userId: string, limit: number = 20): Promise<Conversation[]> {
    return prisma.conversation.findMany({
      where: { userId },
      include: {
        messages: {
          orderBy: { createdAt: 'desc' },
          take: 1,
        },
      },
      orderBy: { lastActivityAt: 'desc' },
      take: limit,
    });
  }

  static async updateConversationActivity(id: string): Promise<void> {
    await prisma.conversation.update({
      where: { id },
      data: { lastActivityAt: new Date() },
    });
  }

  static async addMessageToConversation(
    conversationId: string,
    role: 'USER' | 'ASSISTANT' | 'SYSTEM',
    content: string,
    tokenCount?: number
  ): Promise<Message> {
    const message = await prisma.message.create({
      data: {
        conversationId,
        role,
        content,
        tokenCount,
      },
    });

    // Update conversation activity and message count
    await prisma.conversation.update({
      where: { id: conversationId },
      data: {
        lastActivityAt: new Date(),
        messageCount: { increment: 1 },
      },
    });

    return message;
  }

  // Backtest Results
  static async createBacktestResult(data: CreateBacktestResultData): Promise<BacktestResult> {
    return prisma.backtestResult.create({
      data: {
        jobId: data.jobId,
        userId: data.userId ?? null,
        market: data.market,
        strategyName: data.strategyName,
        strategyCode: data.strategyCode ?? null,
        parameters: JSON.stringify(data.parameters ?? {}),
        timeframe: data.timeframe ?? null,
        timePeriod: data.timePeriod ? JSON.stringify(data.timePeriod) : null,
        results: JSON.stringify(data.results ?? {}),
        trades: JSON.stringify(data.trades ?? []),
        performance: JSON.stringify(data.performance ?? {}),
        riskMetrics: JSON.stringify(data.riskMetrics ?? {}),
        chartData: data.chartData ? JSON.stringify(data.chartData) : null,
        status: data.status ?? 'COMPLETED',
        executionTime: data.executionTime ?? null,
        errorMessage: data.errorMessage ?? null,
        startedAt: data.startedAt ?? null,
        completedAt: data.completedAt ?? null,
      },
    });
  }

  static async findBacktestsByUser(userId: string, limit: number = 50): Promise<BacktestResult[]> {
    return prisma.backtestResult.findMany({
      where: { userId },
      orderBy: { createdAt: 'desc' },
      take: limit,
    });
  }

  static async findBacktestById(id: string): Promise<BacktestResult | null> {
    return prisma.backtestResult.findUnique({
      where: { id },
    });
  }

  // Analytics
  static async recordUserEvent(
    userId: string,
    eventType: string,
    eventData?: any,
    metadata?: any
  ): Promise<void> {
    await prisma.userAnalytics.create({
      data: {
        userId,
        eventType,
        eventData: JSON.stringify(eventData || {}),
        metadata: JSON.stringify(metadata || {}),
      },
    });
  }

  // Feedback
  static async createFeedback(data: CreateFeedbackData): Promise<Feedback> {
    return prisma.feedback.create({
      data: {
        userId: data.userId,
        feedbackType: data.feedbackType,
        subject: data.subject,
        message: data.message,
        priority: data.priority ?? 'MEDIUM',
        status: data.status ?? 'OPEN',
      },
    });
  }

  static async recordStrategyEvent(
    strategyId: string,
    eventType: string,
    eventData?: any
  ): Promise<void> {
    await prisma.strategyAnalytics.create({
      data: {
        strategyId,
        eventType,
        eventData,
      },
    });
  }

  static async recordSystemEvent(eventType: string, eventData?: any, metadata?: any): Promise<void> {
    await prisma.systemAnalytics.create({
      data: {
        eventType,
        eventData: JSON.stringify(eventData || {}),
        metadata: JSON.stringify(metadata || {}),
      },
    });
  }

  // Utility Methods
  static async getUserStats(userId: string) {
    const monthStart = new Date();
    monthStart.setDate(1);
    monthStart.setHours(0, 0, 0, 0);

    const [
      totalStrategies,
      totalConversations,
      totalLikesAgg,
      strategiesCreatedThisMonth,
      aiGenerationsThisMonth,
      chatMessagesThisMonth,
      backtestsRunThisMonth,
    ] = await Promise.all([
      prisma.strategy.count({ where: { userId } }),
      prisma.conversation.count({ where: { userId } }),
      prisma.strategy.aggregate({
        where: { userId },
        _sum: { likes: true },
      }),
      prisma.strategy.count({
        where: {
          userId,
          createdAt: { gte: monthStart },
        },
      }),
      prisma.userAnalytics.count({
        where: {
          userId,
          eventType: 'AI_GENERATION',
          createdAt: { gte: monthStart },
        },
      }),
      prisma.userAnalytics.count({
        where: {
          userId,
          eventType: 'CHAT_MESSAGE_SENT',
          createdAt: { gte: monthStart },
        },
      }),
      prisma.userAnalytics.count({
        where: {
          userId,
          eventType: 'BACKTEST_RUN',
          createdAt: { gte: monthStart },
        },
      }),
    ]);

    return {
      // Legacy stats (still used by subscription gating)
      totalStrategies,
      totalConversations,
      totalLikes: totalLikesAgg._sum.likes || 0,

      // Usage this month (UI)
      aiGenerations: aiGenerationsThisMonth,
      strategiesCreated: strategiesCreatedThisMonth,
      backtestsRun: backtestsRunThisMonth,
      chatMessages: chatMessagesThisMonth,
    };
  }

  // Connection management
  static async connect(): Promise<void> {
    await prisma.$connect();
  }

  static async disconnect(): Promise<void> {
    await prisma.$disconnect();
  }

  static async getPlatformStats(): Promise<{
    totalUsers: number;
    activeUsers24h: number;
    totalStrategies: number;
    totalConversations: number;
    systemHealth: 'healthy' | 'warning' | 'critical';
  }> {
    try {
      const [
        totalUsers,
        activeUsers24h,
        totalStrategies,
        totalConversations,
      ] = await Promise.all([
        prisma.user.count(),
        prisma.user.count({
          where: {
            lastLoginAt: {
              gte: new Date(Date.now() - 24 * 60 * 60 * 1000), // Last 24 hours
            },
          },
        }),
        prisma.strategy.count(),
        prisma.conversation.count(),
      ]);

      return {
        totalUsers,
        activeUsers24h,
        totalStrategies,
        totalConversations,
        systemHealth: 'healthy', // This would be calculated based on actual health checks
      };
    } catch (error) {
      console.error('DatabaseService.getPlatformStats failed', { error });
      throw error;
    }
  }

  static async healthCheck(): Promise<boolean> {
    try {
      await prisma.$queryRaw`SELECT 1`;
      return true;
    } catch {
      return false;
    }
  }

  // GDPR Compliance Methods
  static async getUserStrategies(userId: string): Promise<Strategy[]> {
    return prisma.strategy.findMany({
      where: { userId },
      orderBy: { createdAt: 'desc' },
    });
  }

  static async getUserConversations(userId: string): Promise<Conversation[]> {
    return prisma.conversation.findMany({
      where: { userId },
      include: {
        messages: true,
      },
      orderBy: { createdAt: 'desc' },
    });
  }

  static async getUserEvents(userId: string): Promise<any[]> {
    // If UserEvent model exists
    try {
      return await prisma.userEvent.findMany({
        where: { userId },
        orderBy: { createdAt: 'desc' },
      });
    } catch {
      // UserEvent table might not exist
      return [];
    }
  }

  static async deleteUserConversations(userId: string): Promise<void> {
    // First delete all messages in user's conversations
    const conversations = await prisma.conversation.findMany({
      where: { userId },
      select: { id: true },
    });

    const conversationIds = conversations.map(c => c.id);

    if (conversationIds.length > 0) {
      await prisma.message.deleteMany({
        where: { conversationId: { in: conversationIds } },
      });
    }

    await prisma.conversation.deleteMany({
      where: { userId },
    });
  }

  static async deleteUserStrategies(userId: string): Promise<void> {
    // First delete related data (comments, ratings)
    const strategies = await prisma.strategy.findMany({
      where: { userId },
      select: { id: true },
    });

    const strategyIds = strategies.map(s => s.id);

    if (strategyIds.length > 0) {
      // Delete comments
      await prisma.comment.deleteMany({
        where: { strategyId: { in: strategyIds } },
      }).catch(() => { }); // Ignore if table doesn't exist

      // Delete ratings
      await prisma.rating.deleteMany({
        where: { strategyId: { in: strategyIds } },
      }).catch(() => { }); // Ignore if table doesn't exist
    }

    await prisma.strategy.deleteMany({
      where: { userId },
    });
  }

  static async deleteUserEvents(userId: string): Promise<void> {
    try {
      await prisma.userEvent.deleteMany({
        where: { userId },
      });
    } catch {
      // UserEvent table might not exist
    }
  }

  static async deleteUser(userId: string): Promise<void> {
    await prisma.user.delete({
      where: { id: userId },
    });
  }
}

// Export the Prisma client for advanced queries
export { prisma };
export default DatabaseService;
