import { loadCryptoPrices, loadSportsGames, loadStockPrices, loadForexPairs } from './csvLoader';
import { Domain, DomainRecord } from './types';

export interface DatasetSummary {
  domain: Domain;
  rows: number;
  lastUpdated: string | null;
  sampleFields: string[];
  storage: 'csv' | 'api' | 'rag';
}

export async function loadDomainDataset(domain: Domain, sportFilter?: string): Promise<DomainRecord[]> {
  switch (domain) {
    case 'sports':
      return loadSportsGames(sportFilter);
    case 'crypto':
      return loadCryptoPrices();
    case 'stocks':
      return loadStockPrices();
    case 'forex':
      return loadForexPairs();
    default:
      return [];
  }
}

export async function getDatasetCatalog(): Promise<DatasetSummary[]> {
  const [sports, crypto, stocks, forex] = await Promise.all([
    loadSportsGames(),
    loadCryptoPrices(),
    loadStockPrices(),
    loadForexPairs(),
  ]);

  return [
    {
      domain: 'sports',
      rows: sports.length,
      lastUpdated: sports.at(-1)?.date ?? null,
      sampleFields: ['team', 'opponent', 'losingStreakBefore', 'odds'],
      storage: 'csv',
    },
    {
      domain: 'crypto',
      rows: crypto.length,
      lastUpdated: crypto.at(-1)?.timestamp ?? null,
      sampleFields: ['asset', 'changePct', 'volume', 'newsSentiment'],
      storage: 'csv',
    },
    {
      domain: 'stocks',
      rows: stocks.length,
      lastUpdated: stocks.at(-1)?.date ?? null,
      sampleFields: ['ticker', 'changePct', 'volume', 'macroSignal'],
      storage: 'csv',
    },
    {
      domain: 'forex',
      rows: forex.length,
      lastUpdated: forex.at(-1)?.timestamp ?? null,
      sampleFields: ['pair', 'changePct', 'spread', 'newsImpact'],
      storage: 'csv',
    },
  ];
}

